﻿// 下水データ管理システム
// Copyright (C) 2018 公益財団法人とちぎ建設技術センター(https://www.tochigictc.or.jp/)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.If not, see<http://www.gnu.org/licenses/>.

using System;
using System.Collections.Generic;
using System.Data.SQLite;
using System.Windows.Forms;

namespace WindowsFormsApplication1 {
    /// <summary>
    /// 管理者と処理場の組合わせのクラス
    /// </summary>
    public class KanrisyaSyorijou : IEquatable<KanrisyaSyorijou> {
        public string kanrisya { get; set; }
        public string syorijou { get; set; }
        public string selectionName { get { return kanrisya + "：" + syorijou; } }

        public KanrisyaSyorijou(string k, string s) {
            kanrisya = k;
            syorijou = s;
        }

        public string primaryKey(int c) {
            switch(c) {
                case 0:
                    return kanrisya;
                case 1:
                    return syorijou;
                default:
                    return "";
            }
        }

		public bool Equals(KanrisyaSyorijou other) {
			if (this.kanrisya == other.kanrisya && this.syorijou == other.syorijou) {
				return true;
			}
			else {
				return false;
			}
		}
	}




    /// <summary>
    /// 選択可能月のクラス
    /// </summary>
    public class SelectableMonth {
        public int year { get; set; }
        public int month { get; set; }
        public string selectionName {
            get {
                if(year == -1 && month == -1) {
                    return "";
                }
                else if(year == 0 && month == 0) {
                    return "全期間";
                }
                else {
                    return ("0000" + year.ToString()).Substring(year.ToString().Length, 4) + "年" + ("00" + month.ToString()).Substring(month.ToString().Length, 2) + "月";
                }
            }
        }

        public SelectableMonth(int y, int m) {
            year = y;
            month = m;
        }

        public int primaryKey(int c) {
            switch(c) {
                case 0:
                    return year;
                case 1:
                    return month;
                default:
                    return 0;
            }
        }
    }




    /// <summary>
    /// 選択可能年度のクラス
    /// </summary>
    public class SelectableBYear {
        public int bYear { get; set; }
        public string selectionName {
            get {
                if(bYear == -1) {
                    return "";
                }
                else {
                    return ("0000" + bYear.ToString()).Substring(bYear.ToString().Length, 4) + "年度";
                }
            }
        }

        public SelectableBYear(int y) {
            bYear = y;
        }

        public int primaryKey(int c) {
            switch(c) {
                case 0:
                    return bYear;
                default:
                    return 0;
            }
        }
    }




    /// <summary>
    /// 選択可能四半期のクラス
    /// </summary>
    public class SelectableQuarter {
        public int bYear { get; set; }
        public int quarter { get; set; }
        public string selectionName {
            get {
                if(bYear == -1 && quarter == -1) {
                    return "";
                }
                else {
                    return ("0000" + bYear.ToString()).Substring(bYear.ToString().Length, 4) + "年度　第" + quarter.ToString() + "四半期";
                }
            }
        }

        public SelectableQuarter(int y, int q) {
            bYear = y;
            quarter = q;
        }

        public int primaryKey(int c) {
            switch(c) {
                case 0:
                    return bYear;
                case 1:
                    return quarter;
                default:
                    return 0;
            }
        }
    }




    /// <summary>
    /// グラフ化する期間のクラス
    /// </summary>
    public class SelectableSpan {
        public int months { get; set; }
        public string selectionName {
            get {
                if(months < 0) {
                    return "";
                }
                else if(months == 0) {
                    return "　～　";
                }
                else {
                    int year = (int)(months / 12);
                    int month = months % 12;
                    string sYear = "";
                    string sMonth = "";

                    if(year > 0) {
                        sYear = string.Format("{0}年", year);
                    }

                    if(month > 0) {
                        sMonth = string.Format("{0}ヶ月", month);
                    }

                    return (sYear + sMonth);
                }
            }
        }

        public SelectableSpan(int ms) {
            months = ms;
        }
    }




    /// <summary>
    /// グラフの系列のクラス
    /// </summary>
    public class graphSeries {
        public string kanrisya { get; set; }
        public string syorijou { get; set; }
        public int columnsIndex { get; set; } = 0;
        public int scaleNo { get; set; } = 0;

        public void delete() {
            kanrisya = null;
            syorijou = null;
            columnsIndex = 0;
            scaleNo = 0;
        }
    }




    /// <summary>
    /// プリンターの用紙と印刷範囲のクラス
    /// </summary>
    public static class PrintArea {
        // 1/100インチ単位
        public static bool landscape { get; set; }
        public static int paperSizeWSet { get; set; }
        public static int paperSizeHSet { get; set; }
        public static int marginL { get; set; } = 100;
        public static int marginR { get; set; } = 100;
        public static int marginT { get; set; } = 100;
        public static int marginB { get; set; } = 100;
        public static float hardMarginX { get; set; }
        public static float hardMarginY { get; set; }

        public static int paperSizeW {
            get {
                if(landscape) {
                    return paperSizeWSet;
                }
                else {
                    return paperSizeHSet;
                }
            }
        }
        public static int paperSizeH {
            get {
                if(landscape) {
                    return paperSizeHSet;
                }
                else {
                    return paperSizeWSet;
                }
            }
        }

        public static int printAreaW { get { return paperSizeW - marginL - marginR; } }
        public static int printAreaH { get { return paperSizeH - marginT - marginB; } }
        public static float printerMarginL { get { return marginL - (int)hardMarginX; } }
        public static float printerMarginT { get { return marginT - (int)hardMarginY; } }

        //ミリメートル単位
        public static float printAreaWmm { get { return printAreaW * 0.254F; } }
        public static float printAreaHmm { get { return printAreaH * 0.254F; } }
        public static float printerMarginLmm { get { return printerMarginL * 0.254F; } }
        public static float printerMarginTmm { get { return printerMarginT * 0.254F; } }
    }




    /// <summary>
    /// 印刷に使用するデータのコンテナのクラス
    /// </summary>
    public static class PrintContainer {
        public static string printType { get; set; }

        public static DataGridView dataGridView1 { get; set; }

        //帳票印刷
        public static ComboBox comboBox1 { get; set; }
        public static ComboBox comboBox2 { get; set; }
        public static List<int> printColumnsNo { get; set; }
        public static int printWidth { get; set; }

        public static int dataGridView1Width {
            get {
                int allWidth = 0;

                for(int i = 0; i < dataGridView1.ColumnCount; i++) {
                    if(dataGridView1.Columns[i].Visible) {
                        allWidth += dataGridView1.Columns[i].Width;
                    }
                }

                return allWidth;
            }
        }

        //グラフ印刷
        public static DateTime timeSpanSt;
        public static DateTime timeSpanEd;
        public static bool monthAverage = false;
        public static bool colorPrint = true;

        //グラフの系統数(+1)
        public static int graphSeriesNo = 7;
        public static graphSeries[] graphSerieses = new graphSeries[graphSeriesNo];
    }




    /// <summary>
    /// エクセル関係の処理のクラス
    /// </summary>
    public static class FunctionForExcel {
        //列番号からエクセル形式の列表示文字列を返す
        public static string columnNoToLetter(int columnNo) {
            string alphabet = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

            if(columnNo < 1) {
                return "";
            }

            return columnNoToLetter((columnNo - 1) / 26) + alphabet[(columnNo - 1) % 26];
        }
    }




    /// <summary>
    /// データベース関係の処理のクラス
    /// </summary>
    public static class FunctionForDatabase {
        //テーブルsuishitsu_kanri_settingにsuishitsu_kanriの列を行として用意し初期値を設定
        public static void setSuishitsuKanriRow() {
            List<ColumnConfig> ccSet = ColumnsConfigSet.getSet;

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    cmd.CommandText = "";

                    foreach(ColumnConfig cc in ccSet) {
                        if(cc.Name != "id" && cc.Name != "del_flg" && cc.Name != "kanrisya" && cc.Name != "syorijou" && cc.Name != "sokutei_dt") {
                            cmd.CommandText += "INSERT OR IGNORE INTO suishitsu_kanri_setting (column_name, visible, printform1, printform2, printform3, printform4, printform5, printform6, printform7, printform8, printform9, printform10) VALUES('" + cc.Name + "',1 ,0 ,0 ,0 ,0 ,0 ,0 ,0 ,0 ,0 ,0 ); ";
                        }
                    }

                    cmd.ExecuteNonQuery();
                }

                con.Close();
            }
        }




        //テーブルsuishitsu_kanri_settingにデフォルト値を設定
        public static void setDefaultSuishitsuKanri() {
            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var transaction = con.BeginTransaction()) {
                    using(var cmd = con.CreateCommand()) {
                        for(int i = 0; i < SettingPrintSet.getSet.Count; i++) {
                            string[] printColumn = SettingPrintSet.getSet[i];

                            if(printColumn.Length > 1) {
                                cmd.CommandText = "UPDATE suishitsu_kanri_setting SET printform" + (i + 1).ToString() + " = 1 WHERE column_name IN (";

                                int j = 0;
                                foreach(string settingValue in printColumn) {
                                    j++;

                                    if(j == 1) {
                                        continue;
                                    }
                                    else if(j > 2) {
                                        cmd.CommandText += ", ";
                                    }

                                    cmd.CommandText += "'" + settingValue + "'";
                                }

                                cmd.CommandText += ");";
                                cmd.ExecuteNonQuery();
                            }
                        }
                    }

                    transaction.Commit();
                }

                con.Close();
            }
        }




        //テーブルprint_form_settingに初期設定として帳票分の行を用意
        public static void setPrintFormRow() {
            List<ColumnConfig> ccSet = ColumnsConfigSet.getSet;

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    cmd.CommandText = "";

                    for(int i = 0; i < MaxSizeConfig.maxPrintForm; i++) {
                        cmd.CommandText += "INSERT OR IGNORE INTO print_form_setting (form_no, name) VALUES(" + (i + 1) + ", '帳票" + (i + 1).ToString() + "');";
                    }

                    cmd.ExecuteNonQuery();
                }

                con.Close();
            }
        }




        //テーブルprint_form_settingにデフォルト値を設定
        public static void setDefaultPrintForm() {
            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var transaction = con.BeginTransaction()) {
                    using(var cmd = con.CreateCommand()) {
                        for(int i = 0; i < SettingPrintSet.getSet.Count; i++) {
                            string[] printColumn = SettingPrintSet.getSet[i];

                            if(printColumn.Length > 0) {
                                cmd.CommandText = "UPDATE print_form_setting SET name = '" + printColumn[0] + "' WHERE form_no = " + (i + 1).ToString() + ";";
                                cmd.ExecuteNonQuery();
                            }
                        }
                    }

                    transaction.Commit();
                }

                con.Close();
            }
        }



        //テーブル格納時に小数は10のn乗倍し整数で格納する(doubleをDecimalとして利用する)
        public static decimal? decConvForDB(decimal? dec, string format) {
            if(dec == null) {
                return null;
            }
            else {
                if(format == "N1") {
                    return dec * 10;
                }
                else if(format == "N2") {
                    return dec * 100;
                }
                else if(format == "N3") {
                    return dec * 1000;
                }
                else if(format == "N4") {
                    return dec * 10000;
                }
                else if(format == "N5") {
                    return dec * 100000;
                }
                else {
                    return dec;
                }
            }
        }




        //テーブル格納時に小数は10のn乗倍し整数で格納したものを元に戻す(doubleをDecimalとして利用する)
        public static decimal? decConvFromDB(decimal? dec, string format) {
            if(dec == null) {
                return null;
            }
            else {
                if(format == "N1") {
                    return dec / 10;
                }
                else if(format == "N2") {
                    return dec / 100;
                }
                else if(format == "N3") {
                    return dec / 1000;
                }
                else if(format == "N4") {
                    return dec / 10000;
                }
                else if(format == "N5") {
                    return dec / 100000;
                }
                else {
                    return dec;
                }
            }
        }




        //晴天日を判定してデータベースをアップデートする(sokutei_dt指定時は影響のある翌日翌々日のみ、指定なしは全データ)
        public static void judgeSeitenUpdateDB(string kanrisya, string syorijou, DateTime? sokutei_dt = null) {
            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    string sql1 = "UPDATE suishitsu_kanri SET seiten='○' WHERE del_flg IS NULL AND kanrisya = @kanrisya AND syorijou = @syorijou";

                    if(sokutei_dt != null) {
                        sql1 += " AND sokutei_dt IN (@sokutei_dt0, @sokutei_dt1, @sokutei_dt2);";
                    }
                    else {
                        sql1 += ";";
                    }

                    string sql2 = "UPDATE suishitsu_kanri AS s0 SET seiten=null WHERE del_flg IS NULL AND kanrisya = @kanrisya AND syorijou = @syorijou";

                    if(sokutei_dt != null) {
                        sql2 += " AND sokutei_dt IN (@sokutei_dt0, @sokutei_dt1, @sokutei_dt2)";
                    }

                    string sql3 = "";

                    if(SyorijouConfigSet.getUtenLLUryouDay0(kanrisya, syorijou) != null) {
                        sql3 += "uryou >= @uten_ll_uryou_day0";
                    }

                    if(SyorijouConfigSet.getUtenLLUryouDay1(kanrisya, syorijou) != null) {
                        if(sql3 != "") {
                            sql3 += " OR ";
                        }

                        sql3 += "(SELECT uryou FROM suishitsu_kanri AS s1 WHERE s1.del_flg IS NULL AND s1.kanrisya = @kanrisya AND s1.syorijou = @syorijou AND s1.sokutei_dt = strftime('%Y-%m-%d %H:%M:%S',s0.sokutei_dt, '-1 day')) >= @uten_ll_uryou_day1";
                    }

                    if(SyorijouConfigSet.getUtenLLUryouDay2(kanrisya, syorijou) != null) {
                        if(sql3 != "") {
                            sql3 += " OR ";
                        }

                        sql3 += "(SELECT uryou FROM suishitsu_kanri AS s2 WHERE s2.del_flg IS NULL AND s2.kanrisya = @kanrisya AND s2.syorijou = @syorijou AND s2.sokutei_dt = strftime('%Y-%m-%d %H:%M:%S',s0.sokutei_dt, '-2 day')) >= @uten_ll_uryou_day2";
                    }

                    if(sql3 != "") {
                        sql3 = " AND (" + sql3 + ")";
                    }

                    string sql4 = ";";

                    cmd.CommandText = sql1 + sql2 + sql3 + sql4;

                    cmd.Parameters.Add(new SQLiteParameter("@kanrisya", kanrisya));
                    cmd.Parameters.Add(new SQLiteParameter("@syorijou", syorijou));

                    if(sokutei_dt != null) {
                        DateTime cDate = (DateTime)sokutei_dt;
                        cmd.Parameters.Add(new SQLiteParameter("@sokutei_dt0", cDate.ToString("yyyy-MM-dd HH:mm:ss")));
                        cmd.Parameters.Add(new SQLiteParameter("@sokutei_dt1", cDate.AddDays(1).ToString("yyyy-MM-dd HH:mm:ss")));
                        cmd.Parameters.Add(new SQLiteParameter("@sokutei_dt2", cDate.AddDays(2).ToString("yyyy-MM-dd HH:mm:ss")));
                    }

                    string ccFormat = ColumnsConfigSet.nameToFormat("uryou");

                    if(SyorijouConfigSet.getUtenLLUryouDay0(kanrisya, syorijou) != null) {
                        cmd.Parameters.Add(new SQLiteParameter("@uten_ll_uryou_day0", FunctionForDatabase.decConvForDB(SyorijouConfigSet.getUtenLLUryouDay0(kanrisya, syorijou), ccFormat)));
                    }

                    if(SyorijouConfigSet.getUtenLLUryouDay1(kanrisya, syorijou) != null) {
                        cmd.Parameters.Add(new SQLiteParameter("@uten_ll_uryou_day1", FunctionForDatabase.decConvForDB(SyorijouConfigSet.getUtenLLUryouDay1(kanrisya, syorijou), ccFormat)));
                    }

                    if(SyorijouConfigSet.getUtenLLUryouDay2(kanrisya, syorijou) != null) {
                        cmd.Parameters.Add(new SQLiteParameter("@uten_ll_uryou_day2", FunctionForDatabase.decConvForDB(SyorijouConfigSet.getUtenLLUryouDay2(kanrisya, syorijou), ccFormat)));
                    }

                    cmd.ExecuteNonQuery();
                }
            }
        }




        //二次データを全て再計算しデータベースをアップデートする(処理場ごとの計算条件設定変更時)
        public static void reCalculate2ndValueUpdateDB(string kanrisya, string syorijou, List<ColumnConfig> ccSet, FormA2ProgressSplash progressSplash1) {
            //処理場ごとの計算条件設定が可能な列
            List<ColumnConfig> settingForSyorijouTargetC = new List<ColumnConfig>();

            foreach(ColumnConfig cc in ccSet) {
                if(cc.settingForSyorijouTarget) {
                    settingForSyorijouTargetC.Add(cc);
                }
            }

            //処理場ごとの計算条件設定が可能な列の計算の元になる列
            List<ColumnConfig> settingForSyorijouSourceC = new List<ColumnConfig>();

            foreach(ColumnConfig cc in ccSet) {
                if(cc.forCalc != null && Array.Exists(cc.forCalc, x => settingForSyorijouTargetC.Exists(y => y.Name == x))) {
                    settingForSyorijouSourceC.Add(cc);
                }
            }

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                //再計算対象の全レコード数を取得
                int targetRowCount = 0;

                using(var cmd = con.CreateCommand()) {
                    cmd.CommandText = "SELECT COUNT(*) FROM suishitsu_kanri WHERE kanrisya = @kanrisya AND syorijou = @syorijou;";

                    cmd.Parameters.Add(new SQLiteParameter("@kanrisya", kanrisya));
                    cmd.Parameters.Add(new SQLiteParameter("@syorijou", syorijou));

                    using(var reader = cmd.ExecuteReader()) {
                        if(reader.Read()) {
                            if(!reader.IsDBNull(0)) {
                                targetRowCount = reader.GetInt32(0);
                            }
                        }
                    }
                }

                //DBから対象処理場の全件をSELECT→再計算→UPDATE
                using(var transaction = con.BeginTransaction()) {
                    try {
                        using(var cmd = con.CreateCommand()) {
                            //再計算結果を書込むSQLを予め生成
                            string updateSQL = "UPDATE suishitsu_kanri SET ";

                            int i = 1;
                            foreach(ColumnConfig cc in settingForSyorijouTargetC) {
                                if(i > 1) {
                                    updateSQL += ", ";
                                }

                                updateSQL += cc.Name + " = @p" + i.ToString();
                                i++;
                            }

                            updateSQL += " WHERE id = @id;";

                            //再計算の元になる値を取得
                            cmd.CommandText = "SELECT id";

                            foreach(ColumnConfig cc in settingForSyorijouSourceC) {
                                cmd.CommandText += ", " + cc.Name;
                            }

                            cmd.CommandText += " FROM suishitsu_kanri WHERE kanrisya = @kanrisya AND syorijou = @syorijou;";
                            cmd.Parameters.Add(new SQLiteParameter("@kanrisya", kanrisya));
                            cmd.Parameters.Add(new SQLiteParameter("@syorijou", syorijou));

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;
                                while(reader.Read()) {
                                    d++;
                                    progressSplash1.ProgressMessage = "二次データ再計算中（" + d.ToString() + "/" + targetRowCount.ToString() + "）";
                                    Application.DoEvents();

                                    int id = reader.GetInt32(0);

                                    int j = 1;
                                    foreach(ColumnConfig cc in settingForSyorijouSourceC) {
                                        if(!reader.IsDBNull(j)) {
                                            cc.tempValueDec = FunctionForDatabase.decConvFromDB(reader.GetDecimal(j), cc.Format);
                                        }
                                        else {
                                            cc.tempValueDec = null;
                                        }

                                        j++;
                                    }

                                    //再計算・UPDATE
                                    using(var cmd2 = con.CreateCommand()) {
                                        cmd2.CommandText = updateSQL;
                                        cmd2.Parameters.Add(new SQLiteParameter("@id", id));

                                        int k = 1;
                                        foreach(ColumnConfig cc in settingForSyorijouTargetC) {
                                            cmd2.Parameters.Add(FunctionForDatabase.makeSQLiteParameter(kanrisya, syorijou, cc, ccSet, k));
                                            k++;
                                        }

                                        cmd2.ExecuteNonQuery();
                                    }
                                }
                            }
                        }

                        transaction.Commit();
                    }
                    catch(Exception ex) {
                        transaction.Rollback();
                        MessageBox.Show(ex.Message);
                    }
                }
            }

        }




        //SQLiteDataReaderからdecimal?を取得する
        public static decimal? getDecimalOrNull(SQLiteDataReader reader, int columnNo) {
            return reader.IsDBNull(columnNo) ? null : (decimal?)reader.GetDecimal(columnNo);
        }




        //SQLiteDataReaderからstring?を取得する
        public static string getStringOrNull(SQLiteDataReader reader, int columnNo) {
            return reader.IsDBNull(columnNo) ? null : reader.GetString(columnNo);
        }




        //UPDATE用のSQLiteパラメータを行ごとに生成(データ修正、処理場ごとの計算条件設定変更時)
        public static SQLiteParameter makeSQLiteParameter(string kanrisya, string syorijou, ColumnConfig cc, List<ColumnConfig> ccSet, int i, string input_dtSTR = null) {
            SQLiteParameter newSQLiteParameter;

            if(cc.Name == "seiten") {
                newSQLiteParameter = new SQLiteParameter("@p" + i.ToString(), null);
            }
            else if(cc.calculatedDecimal) {
                //二次データ(数値)
                decimal? newValue = Calculate2ndValue.calc2ndDecimal(kanrisya, syorijou, cc.Name, dgv1Row: null, ccSet: ccSet);

                newValue = FunctionForDatabase.decConvForDB(newValue, cc.Format);

                newSQLiteParameter = new SQLiteParameter("@p" + i.ToString(), newValue);
            }
            else if(cc.calculatedString) {
                //二次データ(文字)
                newSQLiteParameter = new SQLiteParameter("@p" + i.ToString(), Calculate2ndValue.calc2ndString(cc.Name, dgv1Row: null, ccSet: ccSet));
            }
            else if(cc.Name == "input_dt") {
                //入力日
                newSQLiteParameter = new SQLiteParameter("@p" + i.ToString(), input_dtSTR);
            }
            else if(cc.inputType == "TextBox" || cc.inputType == "DateTimePicker" || cc.inputType == "ComboBox") {
                //一次データ(文字)
                string tString = null;

                if(!string.IsNullOrEmpty(cc.tempValueStr)) {
                    tString = cc.tempValueStr;
                }

                newSQLiteParameter = new SQLiteParameter("@p" + i.ToString(), tString);
            }
            else if(cc.inputType == "NumericUpDown") {
                //一次データ(数値)
                decimal? newValue = FunctionForDatabase.decConvForDB(cc.tempValueDec, cc.Format);

                newSQLiteParameter = new SQLiteParameter("@p" + i.ToString(), newValue);
            }
            else {
                newSQLiteParameter = new SQLiteParameter("@p" + i.ToString(), null);
            }

            return newSQLiteParameter;
        }

    }




    /// <summary>
    /// 計算によって求める二次データを求めるクラス
    /// </summary>
    public static class Calculate2ndValue {
        //その日が晴天日扱いならtrueを返す（その処理場の前日・前々日の雨量が影響する）
        public static bool judgeSeiten(string kanrisya, string syorijou, DateTime cDate) {
            decimal? utenLLUryouDay0 = SyorijouConfigSet.getUtenLLUryouDay0(kanrisya, syorijou);
            decimal? utenLLUryouDay1 = SyorijouConfigSet.getUtenLLUryouDay1(kanrisya, syorijou);
            decimal? utenLLUryouDay2 = SyorijouConfigSet.getUtenLLUryouDay2(kanrisya, syorijou);

            string D0 = cDate.ToString("yyyy-MM-dd") + " 00:00:00";
            string D1 = cDate.AddDays(-1).ToString("yyyy-MM-dd") + " 00:00:00";
            string D2 = cDate.AddDays(-2).ToString("yyyy-MM-dd") + " 00:00:00";

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    string sql = "";
                    string ccFormat = ColumnsConfigSet.nameToFormat("uryou");

                    if(utenLLUryouDay0 != null) {
                        sql += "sokutei_dt='" + D0 + "' AND del_flg IS NULL AND uryou>=" + FunctionForDatabase.decConvForDB(utenLLUryouDay0, ccFormat).ToString();
                    }

                    if(utenLLUryouDay1 != null) {
                        if(sql != "") {
                            sql += " OR ";
                        }

                        sql += "sokutei_dt='" + D1 + "' AND del_flg IS NULL AND uryou>=" + FunctionForDatabase.decConvForDB(utenLLUryouDay1, ccFormat).ToString();
                    }

                    if(utenLLUryouDay2 != null) {
                        if(sql != "") {
                            sql += " OR ";
                        }

                        sql += "sokutei_dt='" + D2 + "' AND del_flg IS NULL AND uryou>=" + FunctionForDatabase.decConvForDB(utenLLUryouDay2, ccFormat).ToString();
                    }

                    if(sql != "") {
                        sql = " AND (" + sql + ")";
                    }
                    else {
                        sql = " AND 1=0";
                    }

                    //雨量が閾値以上の日があれば雨天
                    cmd.CommandText = "SELECT COUNT(*) FROM suishitsu_kanri WHERE kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "'" + sql + ";";

                    using(var reader = cmd.ExecuteReader()) {
                        reader.Read();

                        if(reader.HasRows) {
                            if(!reader.IsDBNull(0)) {
                                if(reader.GetInt32(0) >= 1) {
                                    return false;
                                }
                            }
                        }
                    }
                }
            }

            //それ以外なら晴天扱いする
            return true;
        }




        //二次データを求める計算（数値戻り）
        public static decimal? calc2ndDecimal(string kanrisya, string syorijou, string ccName, DataGridViewRow dgv1Row = null, List<ColumnConfig> ccSet = null) {
            decimal? retValue = null;

            string calcSource1 = SyorijouConfigSet.getConfigString(kanrisya, syorijou, 1);

            #region P場・幹線計算定義

            if(ccName == "wv_pumpjou_all") {
                //P場排水量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue = getCcValue("wv_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ee_pumpjou_all") {
                //P場電力量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue = getCcValue("ee_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "v_hao_pumpjou_all") {
                //P場A重油計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue = getCcValue("v_hao_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "v_lo_pumpjou_all") {
                //P場軽油計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue = getCcValue("v_lo_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "v_chinsa_pumpjou_all") {
                //P場沈砂発生量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue = getCcValue("v_chinsa_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "v_shisa_pumpjou_all") {
                //P場しさ発生量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue = getCcValue("v_shisa_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }

            #endregion


            #region 処理場計算定義

            else if(ccName == "ee_extra") {
                //その他電力量
                decimal? ee_juden = getCcValue("ee_juden", dgv1Row, ccSet);
                decimal? ee_jikahatsu = getCcValue("ee_jikahatsu", dgv1Row, ccSet);
                decimal? ee_jounaipump = getCcValue("ee_jounaipump", dgv1Row, ccSet);
                decimal? ee_mizusyori = getCcValue("ee_mizusyori", dgv1Row, ccSet);
                decimal? ee_odeisyori = getCcValue("ee_odeisyori", dgv1Row, ccSet);

                if(ee_juden != null || ee_jikahatsu != null || ee_jounaipump != null || ee_mizusyori != null || ee_odeisyori != null) {
                    decimal sum = 0;

                    sum += ee_juden ?? 0;
                    sum += ee_jikahatsu ?? 0;
                    sum -= ee_jounaipump ?? 0;
                    sum -= ee_mizusyori ?? 0;
                    sum -= ee_odeisyori ?? 0;

                    retValue = sum;
                }
            }

            #endregion


            #region 初沈計算定義

            else if(ccName == "h_syochin") {
                //初沈沈殿時間
                decimal? pv_syochin = getCcValue("pv_syochin", dgv1Row, ccSet);
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 1)) {
                    case "初沈流入水量":
                        wv_kijun = getCcValue("wv_syochin", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                }

                if(pv_syochin != null && wv_kijun != null && wv_kijun > 0) {
                    decimal retValue2 = (decimal)pv_syochin / (decimal)wv_kijun * 24;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "ov_nama") {
                //生汚泥引抜量計
                decimal? ov_nama_nousyuku = getCcValue("ov_nama_nousyuku", dgv1Row, ccSet);
                decimal? ov_nama_syouka = getCcValue("ov_nama_syouka", dgv1Row, ccSet);
                decimal? ov_nama_dassui = getCcValue("ov_nama_dassui", dgv1Row, ccSet);
                decimal? ov_nama_jougai = getCcValue("ov_nama_jougai", dgv1Row, ccSet);

                if(ov_nama_nousyuku != null || ov_nama_syouka != null || ov_nama_dassui != null || ov_nama_jougai != null) {
                    decimal sum = 0;

                    sum += ov_nama_nousyuku ?? 0;
                    sum += ov_nama_syouka ?? 0;
                    sum += ov_nama_dassui ?? 0;
                    sum += ov_nama_jougai ?? 0;

                    retValue = sum;
                }
            }
            else if(ccName == "sg_nama") {
                //生汚泥固形物量
                decimal? ov_nama_nousyuku = getCcValue("ov_nama_nousyuku", dgv1Row, ccSet);
                decimal? ov_nama_syouka = getCcValue("ov_nama_syouka", dgv1Row, ccSet);
                decimal? ov_nama_dassui = getCcValue("ov_nama_dassui", dgv1Row, ccSet);
                decimal? ov_nama_jougai = getCcValue("ov_nama_jougai", dgv1Row, ccSet);
                decimal? oc_nama = getCcValue("oc_nama", dgv1Row, ccSet);

                if(oc_nama != null && (ov_nama_nousyuku != null || ov_nama_syouka != null || ov_nama_dassui != null || ov_nama_jougai != null)) {
                    decimal sum = 0;

                    sum += ov_nama_nousyuku ?? 0;
                    sum += ov_nama_syouka ?? 0;
                    sum += ov_nama_dassui ?? 0;
                    sum += ov_nama_jougai ?? 0;

                    decimal retValue2 = sum * (decimal)oc_nama / 100 * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "ss_jokyo_syo_etsuryuu") {
                //初沈越除去率SS
                decimal? ss_syo_ryuunyuu = getCcValue("ss_syo_ryuunyuu", dgv1Row, ccSet);
                decimal? ss_syo_etsuryuu = getCcValue("ss_syo_etsuryuu", dgv1Row, ccSet);

                if(ss_syo_ryuunyuu != null && ss_syo_etsuryuu != null && ss_syo_ryuunyuu > 0) {
                    decimal retValue2 = ((decimal)ss_syo_ryuunyuu - (decimal)ss_syo_etsuryuu) / (decimal)ss_syo_ryuunyuu * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }

            #endregion


            #region 反応T共通計算定義

            else if(ccName == "wt_hannoutank_all") {
                //反応T平均水温
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("wt_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "ph_hannoutank_all") {
                //反応T平均pH
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("ph_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sv_hannoutank_all") {
                //反応T平均SV
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("sv_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "mldo_p_hannoutank_all") {
                //反応T平均MLDO前
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("mldo_p_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "mldo_a_hannoutank_all") {
                //反応T平均MLDO後
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("mldo_a_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "mlss_hannoutank_all") {
                //反応T平均MLSS
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("mlss_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "mlvss_hannoutank_all") {
                //反応T平均MLVSS
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("mlvss_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "vsspss_hannoutank_all") {
                //反応T平均VSS/SS
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? mlss_hannoutank = getCcValue("mlss_hannoutank_" + i.ToString(), dgv1Row, ccSet);
                    decimal? mlvss_hannoutank = getCcValue("mlvss_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(mlss_hannoutank != null && mlvss_hannoutank != null && mlss_hannoutank > 0) {
                        sum += (decimal)mlvss_hannoutank / (decimal)mlss_hannoutank * 100;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "svi_hannoutank_all") {
                //反応T平均SVI
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? mlss_hannoutank = getCcValue("mlss_hannoutank_" + i.ToString(), dgv1Row, ccSet);
                    decimal? sv_hannoutank = getCcValue("sv_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(mlss_hannoutank != null && sv_hannoutank != null && mlss_hannoutank > 0) {
                        sum += (decimal)sv_hannoutank * 10000 / (decimal)mlss_hannoutank;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "bodpmlss_hannoutank_all") {
                //反応T平均BOD/MLSS負荷
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 2)) {
                    case "初沈流入水量":
                        wv_kijun = getCcValue("wv_syochin", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                }

                decimal? wv_kani = getCcValue("wv_kani", dgv1Row, ccSet);
                decimal? bod_kijun = getCcValue("bod_syo_etsuryuu", dgv1Row, ccSet);

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 6)) {
                    case "初沈越流水質":
                        bod_kijun = getCcValue("bod_syo_etsuryuu", dgv1Row, ccSet);
                        break;
                    case "流入水質":
                        bod_kijun = getCcValue("bod_ryuunyuu", dgv1Row, ccSet);
                        break;
                }

                decimal? pv_hannoutank = getCcValue("pv_hannoutank", dgv1Row, ccSet);

                if(wv_kijun != null && bod_kijun != null && pv_hannoutank != null && pv_hannoutank > 0) {
                    wv_kani = wv_kani ?? 0;

                    decimal sum = 0;
                    int count = 0;

                    for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                        decimal? mlss_hannoutank = getCcValue("mlss_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                        if(mlss_hannoutank != null && mlss_hannoutank > 0) {
                            sum += ((decimal)wv_kijun - (decimal)wv_kani) * (decimal)bod_kijun / ((decimal)pv_hannoutank * (decimal)mlss_hannoutank);
                            count++;
                        }
                    }

                    if(count > 0) {
                        decimal retValue2 = sum / count;
                        retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                    }

                }
            }
            else if(ccName == "ar_hannoutank_all") {
                //反応T空気倍率
                decimal? av_hannoutank_all = getCcValue("av_hannoutank_all", dgv1Row, ccSet);
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 2)) {
                    case "初沈流入水量":
                        wv_kijun = getCcValue("wv_syochin", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                }

                decimal? wv_kani = getCcValue("wv_kani", dgv1Row, ccSet);

                wv_kani = wv_kani ?? 0;

                if(av_hannoutank_all != null && wv_kijun != null) {
                    if((decimal)wv_kijun - (decimal)wv_kani > 0) {
                        decimal retValue2 = (decimal)av_hannoutank_all / ((decimal)wv_kijun - (decimal)wv_kani);
                        retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                    }
                }
            }
            else if(ccName == "st_hannoutank_all") {
                //反応T滞留時間
                decimal? pv_hannoutank = getCcValue("pv_hannoutank", dgv1Row, ccSet);
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 2)) {
                    case "初沈流入水量":
                        wv_kijun = getCcValue("wv_syochin", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                }

                decimal? wv_kani = getCcValue("wv_kani", dgv1Row, ccSet);

                wv_kani = wv_kani ?? 0;

                if(pv_hannoutank != null && wv_kijun != null) {
                    if((decimal)wv_kijun - (decimal)wv_kani > 0) {
                        decimal retValue2 = (decimal)pv_hannoutank / ((decimal)wv_kijun - (decimal)wv_kani) * 24;
                        retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                    }
                }
            }
            else if(ccName == "srt_hannoutank_all") {
                //反応T_SRT
                decimal? pv_hannoutank = getCcValue("pv_hannoutank", dgv1Row, ccSet);
                decimal? ov_yj_syochin = getCcValue("ov_yj_syochin", dgv1Row, ccSet);
                decimal? ov_yj_nousyuku = getCcValue("ov_yj_nousyuku", dgv1Row, ccSet);
                decimal? ov_yj_syouka = getCcValue("ov_yj_syouka", dgv1Row, ccSet);
                decimal? ov_yj_dassui = getCcValue("ov_yj_dassui", dgv1Row, ccSet);
                decimal? ov_yj_jougai = getCcValue("ov_yj_jougai", dgv1Row, ccSet);
                decimal? rsss_hensou = getCcValue("rsss_hensou", dgv1Row, ccSet);
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 2)) {
                    case "初沈流入水量":
                        wv_kijun = getCcValue("wv_syochin", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                }

                decimal? ss_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 5)) {
                    case "塩混出口水質":
                        ss_kijun = getCcValue("ss_enkondeguchi", dgv1Row, ccSet);
                        break;
                    case "放流口水質":
                        ss_kijun = getCcValue("ss_houryuukou", dgv1Row, ccSet);
                        break;
                }

                decimal ov_yojou = 0;

                ov_yojou += ov_yj_syochin ?? 0;
                ov_yojou += ov_yj_nousyuku ?? 0;
                ov_yojou += ov_yj_syouka ?? 0;
                ov_yojou += ov_yj_dassui ?? 0;
                ov_yojou += ov_yj_jougai ?? 0;

                if(pv_hannoutank != null && rsss_hensou != null && wv_kijun != null && ss_kijun != null && (ov_yojou * (decimal)rsss_hensou + ((decimal)wv_kijun - ov_yojou) * (decimal)ss_kijun) > 0) {
                    decimal sum = 0;
                    int count = 0;

                    for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                        decimal? ccValue = getCcValue("mlss_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }

                    if(count > 0) {
                        decimal mlss_hannoutank_all = sum / count;
                        decimal retValue2 = (decimal)pv_hannoutank * mlss_hannoutank_all / (ov_yojou * (decimal)rsss_hensou + ((decimal)wv_kijun - ov_yojou) * (decimal)ss_kijun);
                        retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                    }
                }
            }
            else if(ccName == "orp_ken_hannoutank_all") {
                //反応T平均ORP嫌気
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("orp_ken_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "orp_kou_hannoutank_all") {
                //反応T平均ORP好気
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("orp_kou_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "rr_zen_hannoutank_all") {
                //反応T平均Rr前段
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("rr_zen_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "rr_kou_hannoutank_all") {
                //反応T平均Rr後段
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    decimal? ccValue = getCcValue("rr_kou_hannoutank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    decimal retValue2 = sum / count;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "vsspss_hensou") {
                //返送汚泥VSS/SS
                decimal? rsvss_hensou = getCcValue("rsvss_hensou", dgv1Row, ccSet);
                decimal? rsss_hensou = getCcValue("rsss_hensou", dgv1Row, ccSet);

                if(rsvss_hensou != null && rsss_hensou != null && rsss_hensou > 0) {
                    decimal retValue2 = (decimal)rsvss_hensou / (decimal)rsss_hensou * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "svi_hensou") {
                //返送汚泥SVI
                decimal? sv_hensou = getCcValue("sv_hensou", dgv1Row, ccSet);
                decimal? rsss_hensou = getCcValue("rsss_hensou", dgv1Row, ccSet);

                if(sv_hensou != null && rsss_hensou != null && rsss_hensou > 0) {
                    decimal retValue2 = (decimal)sv_hensou * 10000 / (decimal)rsss_hensou;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }

            #endregion


            #region 終沈計算定義

            else if(ccName == "h_syuuchin") {
                //終沈沈殿時間
                decimal? pv_syuuchin = getCcValue("pv_syuuchin", dgv1Row, ccSet);
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 3)) {
                    case "初沈流入水量":
                        wv_kijun = getCcValue("wv_syochin", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                }

                decimal? wv_kani = getCcValue("wv_kani", dgv1Row, ccSet);

                wv_kani = wv_kani ?? 0;

                if(pv_syuuchin != null && wv_kijun != null && (wv_kijun - wv_kani) > 0) {
                    decimal retValue2 = (decimal)pv_syuuchin / ((decimal)wv_kijun - (decimal)wv_kani) * 24;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "or_hensou") {
                //返送汚泥率
                decimal? ov_hensou = getCcValue("ov_hensou", dgv1Row, ccSet);
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 3)) {
                    case "初沈流入水量":
                        wv_kijun = getCcValue("wv_syochin", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                }

                decimal? wv_kani = getCcValue("wv_kani", dgv1Row, ccSet);

                wv_kani = wv_kani ?? 0;

                if(ov_hensou != null && wv_kijun != null && (wv_kijun - wv_kani) > 0) {
                    decimal retValue2 = (decimal)ov_hensou / ((decimal)wv_kijun - (decimal)wv_kani) * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "ov_yojou") {
                //余剰汚泥計
                decimal? ov_yj_syochin = getCcValue("ov_yj_syochin", dgv1Row, ccSet);
                decimal? ov_yj_nousyuku = getCcValue("ov_yj_nousyuku", dgv1Row, ccSet);
                decimal? ov_yj_syouka = getCcValue("ov_yj_syouka", dgv1Row, ccSet);
                decimal? ov_yj_dassui = getCcValue("ov_yj_dassui", dgv1Row, ccSet);
                decimal? ov_yj_jougai = getCcValue("ov_yj_jougai", dgv1Row, ccSet);

                if(ov_yj_syochin != null || ov_yj_nousyuku != null || ov_yj_syouka != null || ov_yj_dassui != null || ov_yj_jougai != null) {
                    decimal sum = 0;

                    sum += ov_yj_syochin ?? 0;
                    sum += ov_yj_nousyuku ?? 0;
                    sum += ov_yj_syouka ?? 0;
                    sum += ov_yj_dassui ?? 0;
                    sum += ov_yj_jougai ?? 0;

                    retValue = sum;
                }
            }
            else if(ccName == "sg_yojou") {
                //余剰汚泥固形物量
                decimal? ov_yj_syochin = getCcValue("ov_yj_syochin", dgv1Row, ccSet);
                decimal? ov_yj_nousyuku = getCcValue("ov_yj_nousyuku", dgv1Row, ccSet);
                decimal? ov_yj_syouka = getCcValue("ov_yj_syouka", dgv1Row, ccSet);
                decimal? ov_yj_dassui = getCcValue("ov_yj_dassui", dgv1Row, ccSet);
                decimal? ov_yj_jougai = getCcValue("ov_yj_jougai", dgv1Row, ccSet);
                decimal? oc_yojou = getCcValue("oc_yojou", dgv1Row, ccSet);

                if((ov_yj_syochin != null || ov_yj_nousyuku != null || ov_yj_syouka != null || ov_yj_dassui != null || ov_yj_jougai != null) && oc_yojou != null) {
                    decimal sum = 0;

                    sum += ov_yj_syochin ?? 0;
                    sum += ov_yj_nousyuku ?? 0;
                    sum += ov_yj_syouka ?? 0;
                    sum += ov_yj_dassui ?? 0;
                    sum += ov_yj_jougai ?? 0;

                    decimal retValue2 = sum * (decimal)oc_yojou / 100 * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }

            #endregion


            #region 塩混・放流計算定義

            else if(ccName == "mr_jia") {
                //次亜注入率
                decimal? wv_jia = getCcValue("wv_jia", dgv1Row, ccSet);
                decimal? gs_jia = getCcValue("gs_jia", dgv1Row, ccSet);
                decimal? er_jia = getCcValue("er_jia", dgv1Row, ccSet);
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 4)) {
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                }

                if(wv_jia != null && gs_jia != null && er_jia != null && wv_kijun != null && wv_kijun > 0) {
                    decimal retValue2 = (decimal)wv_jia * (decimal)gs_jia * (decimal)er_jia * 10 / (decimal)wv_kijun;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "min_enkon") {
                //塩混接触時間
                decimal? pv_enkon = getCcValue("pv_enkon", dgv1Row, ccSet);
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 4)) {
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                }

                decimal? wv_kani = getCcValue("wv_kani", dgv1Row, ccSet);

                wv_kani = wv_kani ?? 0;

                if(pv_enkon != null && wv_kijun != null && (wv_kijun - wv_kani) > 0) {
                    decimal retValue2 = (decimal)pv_enkon / ((decimal)wv_kijun - (decimal)wv_kani) * 24 * 60;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "cod_jokyo_sougou") {
                //総合除去率COD
                decimal? cod_ryuunyuu = getCcValue("cod_ryuunyuu", dgv1Row, ccSet);
                decimal? cod_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 5)) {
                    case "塩混出口水質":
                        cod_kijun = getCcValue("cod_enkondeguchi", dgv1Row, ccSet);
                        break;
                    case "放流口水質":
                        cod_kijun = getCcValue("cod_houryuukou", dgv1Row, ccSet);
                        break;
                }

                if(cod_ryuunyuu != null && cod_kijun != null && cod_ryuunyuu > 0) {
                    decimal retValue2 = ((decimal)cod_ryuunyuu - (decimal)cod_kijun) / (decimal)cod_ryuunyuu * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "bod_jokyo_sougou") {
                //総合除去率BOD
                decimal? bod_ryuunyuu = getCcValue("bod_ryuunyuu", dgv1Row, ccSet);
                decimal? bod_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 5)) {
                    case "塩混出口水質":
                        bod_kijun = getCcValue("bod_enkondeguchi", dgv1Row, ccSet);
                        break;
                    case "放流口水質":
                        bod_kijun = getCcValue("bod_houryuukou", dgv1Row, ccSet);
                        break;
                }

                if(bod_ryuunyuu != null && bod_kijun != null && bod_ryuunyuu > 0) {
                    decimal retValue2 = ((decimal)bod_ryuunyuu - (decimal)bod_kijun) / (decimal)bod_ryuunyuu * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "ss_jokyo_sougou") {
                //総合除去率SS
                decimal? ss_ryuunyuu = getCcValue("ss_ryuunyuu", dgv1Row, ccSet);
                decimal? ss_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 5)) {
                    case "塩混出口水質":
                        ss_kijun = getCcValue("ss_enkondeguchi", dgv1Row, ccSet);
                        break;
                    case "放流口水質":
                        ss_kijun = getCcValue("ss_houryuukou", dgv1Row, ccSet);
                        break;
                }

                if(ss_ryuunyuu != null && ss_kijun != null && ss_ryuunyuu > 0) {
                    decimal retValue2 = ((decimal)ss_ryuunyuu - (decimal)ss_kijun) / (decimal)ss_ryuunyuu * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }

            #endregion


            #region 汚泥濃縮計算定義

            else if(ccName == "ov_nousyukuki_input_juuryoku") {
                //投入汚泥量重力濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) == "重力式") {
                        decimal? ccValue = getCcValue("ov_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_nousyukuki_input_ex") {
                //投入汚泥量他濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) != "重力式") {
                        decimal? ccValue = getCcValue("ov_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_nousyukuki_input_all") {
                //投入汚泥濃縮設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    decimal? ccValue = getCcValue("ov_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "sg_nousyukuki_input_juuryoku") {
                //投汚固形物量重力濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) == "重力式") {
                        decimal? oci_nousyukuki = getCcValue("oci_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? ov_nousyukuki = getCcValue("ov_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(oci_nousyukuki != null && ov_nousyukuki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)ov_nousyukuki * (decimal)oci_nousyukuki / 100 * 1000, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "sg_nousyukuki_input_ex") {
                //投汚固形物量他濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) != "重力式") {
                        decimal? oci_nousyukuki = getCcValue("oci_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? ov_nousyukuki = getCcValue("ov_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(oci_nousyukuki != null && ov_nousyukuki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)ov_nousyukuki * (decimal)oci_nousyukuki / 100 * 1000, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "sg_nousyukuki_input_all") {
                //投汚固形物濃縮設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    decimal? oci_nousyukuki = getCcValue("oci_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                    decimal? ov_nousyukuki = getCcValue("ov_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                    if(oci_nousyukuki != null && ov_nousyukuki != null) {
                        sum += ColumnsConfigSet.roundingByName((decimal)ov_nousyukuki * (decimal)oci_nousyukuki / 100 * 1000, ccName);
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_gyousyuuzai_nousyukuki_juuryoku") {
                //高分子使用量重力濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) == "重力式") {
                        decimal? v_gyousyuuzai_nousyukuki = getCcValue("v_gyousyuuzai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? oc_gyousyuuzai_nousyukuki = getCcValue("oc_gyousyuuzai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(v_gyousyuuzai_nousyukuki != null && oc_gyousyuuzai_nousyukuki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)v_gyousyuuzai_nousyukuki * (decimal)oc_gyousyuuzai_nousyukuki / 100, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_gyousyuuzai_nousyukuki_ex") {
                //高分子使用量他濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) != "重力式") {
                        decimal? v_gyousyuuzai_nousyukuki = getCcValue("v_gyousyuuzai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? oc_gyousyuuzai_nousyukuki = getCcValue("oc_gyousyuuzai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(v_gyousyuuzai_nousyukuki != null && oc_gyousyuuzai_nousyukuki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)v_gyousyuuzai_nousyukuki * (decimal)oc_gyousyuuzai_nousyukuki / 100, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_gyousyuuzai_nousyukuki_all") {
                //高分子使用濃縮設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    decimal? v_gyousyuuzai_nousyukuki = getCcValue("v_gyousyuuzai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                    decimal? oc_gyousyuuzai_nousyukuki = getCcValue("oc_gyousyuuzai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                    if(v_gyousyuuzai_nousyukuki != null && oc_gyousyuuzai_nousyukuki != null) {
                        sum += ColumnsConfigSet.roundingByName((decimal)v_gyousyuuzai_nousyukuki * (decimal)oc_gyousyuuzai_nousyukuki / 100, ccName);
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_kihoujozai_nousyukuki_juuryoku") {
                //起泡剤使用量重力濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) == "重力式") {
                        decimal? v_kihoujozai_nousyukuki = getCcValue("v_kihoujozai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? oc_kihoujozai_nousyukuki = getCcValue("oc_kihoujozai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(v_kihoujozai_nousyukuki != null && oc_kihoujozai_nousyukuki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)v_kihoujozai_nousyukuki * (decimal)oc_kihoujozai_nousyukuki / 100, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_kihoujozai_nousyukuki_ex") {
                //起泡剤使用量他濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) != "重力式") {
                        decimal? v_kihoujozai_nousyukuki = getCcValue("v_kihoujozai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? oc_kihoujozai_nousyukuki = getCcValue("oc_kihoujozai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(v_kihoujozai_nousyukuki != null && oc_kihoujozai_nousyukuki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)v_kihoujozai_nousyukuki * (decimal)oc_kihoujozai_nousyukuki / 100, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_kihoujozai_nousyukuki_all") {
                //起泡剤使用濃縮設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    decimal? v_kihoujozai_nousyukuki = getCcValue("v_kihoujozai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);
                    decimal? oc_kihoujozai_nousyukuki = getCcValue("oc_kihoujozai_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                    if(v_kihoujozai_nousyukuki != null && oc_kihoujozai_nousyukuki != null) {
                        sum += ColumnsConfigSet.roundingByName((decimal)v_kihoujozai_nousyukuki * (decimal)oc_kihoujozai_nousyukuki / 100, ccName);
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_nousyuku_output_all") {
                //濃縮汚泥発生量計
                decimal? ov_nousyuku_output_juuryoku = getCcValue("ov_nousyuku_output_juuryoku", dgv1Row, ccSet);
                decimal? ov_nousyuku_output_ex = getCcValue("ov_nousyuku_output_ex", dgv1Row, ccSet);

                if(ov_nousyuku_output_juuryoku != null || ov_nousyuku_output_ex != null) {
                    decimal sum = 0;

                    sum += ov_nousyuku_output_juuryoku ?? 0;
                    sum += ov_nousyuku_output_ex ?? 0;

                    retValue = sum;
                }
            }
            else if(ccName == "sg_nousyuku_output_juuryoku") {
                //濃縮汚泥固形物量重力
                decimal? oc_nousyuku_output_juuryoku = getCcValue("oc_nousyuku_output_juuryoku", dgv1Row, ccSet);
                decimal? ov_nousyuku_output_juuryoku = getCcValue("ov_nousyuku_output_juuryoku", dgv1Row, ccSet);

                if(oc_nousyuku_output_juuryoku != null && ov_nousyuku_output_juuryoku != null) {
                    decimal retValue2 = (decimal)ov_nousyuku_output_juuryoku * (decimal)oc_nousyuku_output_juuryoku / 100 * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sg_nousyuku_output_ex") {
                //濃縮汚泥固形物量他
                decimal? oc_nousyuku_output_ex = getCcValue("oc_nousyuku_output_ex", dgv1Row, ccSet);
                decimal? ov_nousyuku_output_ex = getCcValue("ov_nousyuku_output_ex", dgv1Row, ccSet);

                if(oc_nousyuku_output_ex != null && ov_nousyuku_output_ex != null) {
                    decimal retValue2 = (decimal)ov_nousyuku_output_ex * (decimal)oc_nousyuku_output_ex / 100 * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sg_nousyuku_output_all") {
                //濃縮汚泥固形物量計
                decimal? oc_nousyuku_output_juuryoku = getCcValue("oc_nousyuku_output_juuryoku", dgv1Row, ccSet);
                decimal? ov_nousyuku_output_juuryoku = getCcValue("ov_nousyuku_output_juuryoku", dgv1Row, ccSet);

                decimal? retValue2 = null;

                if(oc_nousyuku_output_juuryoku != null && ov_nousyuku_output_juuryoku != null) {
                    retValue2 = ColumnsConfigSet.roundingByName((decimal)ov_nousyuku_output_juuryoku * (decimal)oc_nousyuku_output_juuryoku / 100 * 1000, ccName);
                }

                decimal? oc_nousyuku_output_ex = getCcValue("oc_nousyuku_output_ex", dgv1Row, ccSet);
                decimal? ov_nousyuku_output_ex = getCcValue("ov_nousyuku_output_ex", dgv1Row, ccSet);

                decimal? retValue3 = null;

                if(oc_nousyuku_output_ex != null && ov_nousyuku_output_ex != null) {
                    retValue3 = ColumnsConfigSet.roundingByName((decimal)ov_nousyuku_output_ex * (decimal)oc_nousyuku_output_ex / 100 * 1000, ccName);
                }

                if(retValue2 != null || retValue3 != null) {
                    retValue2 = retValue2 ?? 0;
                    retValue3 = retValue3 ?? 0;
                    retValue = retValue2 + retValue3;
                }
            }
            else if(ccName == "dt_nousyukuki_juuryoku") {
                //運転時間重力濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) == "重力式") {
                        decimal? ccValue = getCcValue("dt_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "dt_nousyukuki_ex") {
                //運転時間他濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(getCcValueStr("type_nousyukuki_" + i.ToString(), dgv1Row, ccSet) != "重力式") {
                        decimal? ccValue = getCcValue("dt_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "dt_nousyukuki_all") {
                //運転時間濃縮設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    decimal? ccValue = getCcValue("dt_nousyukuki_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }

            #endregion


            #region 消化計算定義

            else if(ccName == "sg_syouka_input") {
                //消化槽投汚固形物量
                decimal? oc_syouka_input = getCcValue("oc_syouka_input", dgv1Row, ccSet);
                decimal? ov_syouka_input = getCcValue("ov_syouka_input", dgv1Row, ccSet);

                if(oc_syouka_input != null && ov_syouka_input != null) {
                    decimal retValue2 = (decimal)ov_syouka_input * (decimal)oc_syouka_input / 100 * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sg_syouka_output") {
                //消化汚泥固形物量
                decimal? oc_syouka_output = getCcValue("oc_syouka_output", dgv1Row, ccSet);
                decimal? ov_syouka_output = getCcValue("ov_syouka_output", dgv1Row, ccSet);

                if(oc_syouka_output != null && ov_syouka_output != null) {
                    decimal retValue2 = (decimal)ov_syouka_output * (decimal)oc_syouka_output / 100 * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "r_syouka") {
                //消化率
                decimal? oy_syouka_input = getCcValue("oy_syouka_input", dgv1Row, ccSet);
                decimal? oy_syouka_output = getCcValue("oy_syouka_output", dgv1Row, ccSet);

                if(oy_syouka_input != null && oy_syouka_output != null && (oy_syouka_input * (100 - oy_syouka_output)) > 0) {
                    decimal retValue2 = (1 - (100 - (decimal)oy_syouka_input) * (decimal)oy_syouka_output / ((decimal)oy_syouka_input * (100 - (decimal)oy_syouka_output))) * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "d_syouka") {
                //消化日数
                decimal? pv_syouka = getCcValue("pv_syouka", dgv1Row, ccSet);
                decimal? ov_syouka_input = getCcValue("ov_syouka_input", dgv1Row, ccSet);

                if(pv_syouka != null && ov_syouka_input != null && ov_syouka_input > 0) {
                    decimal retValue2 = (decimal)pv_syouka / (decimal)ov_syouka_input;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "gr_syoukagas") {
                //ガス発生倍率
                decimal? v_syoukag_hassei = getCcValue("v_syoukag_hassei", dgv1Row, ccSet);
                decimal? ov_syouka_input = getCcValue("ov_syouka_input", dgv1Row, ccSet);

                if(v_syoukag_hassei != null && ov_syouka_input != null && ov_syouka_input > 0) {
                    decimal retValue2 = (decimal)v_syoukag_hassei / (decimal)ov_syouka_input;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "dt_syoukatank_all") {
                //運転時間消化槽計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxSyoukatank; i++) {
                    decimal? ccValue = getCcValue("dt_syoukatank_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }

            #endregion


            #region 汚泥脱水計算定義

            else if(ccName == "ov_dassuiki_input_enshin") {
                //供給汚泥量遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "遠心分離") {
                        decimal? ccValue = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_dassuiki_input_bp") {
                //供給汚泥量BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "ベルトプレス") {
                        decimal? ccValue = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_dassuiki_input_ex") {
                //供給汚泥量他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "遠心分離" && getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "ベルトプレス") {
                        decimal? ccValue = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_dassuiki_input_all") {
                //供給汚泥量脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal? ccValue = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "sg_dassuiki_input_enshin") {
                //脱投汚固形物量遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "遠心分離") {
                        decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ov_dassuiki != null) {
                            oci_dassuiki = oci_dassuiki ?? 0;
                            sum += ColumnsConfigSet.roundingByName((decimal)ov_dassuiki * (decimal)oci_dassuiki / 100 * 1000, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "sg_dassuiki_input_bp") {
                //脱投汚固形物量BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "ベルトプレス") {
                        decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ov_dassuiki != null) {
                            oci_dassuiki = oci_dassuiki ?? 0;
                            sum += ColumnsConfigSet.roundingByName((decimal)ov_dassuiki * (decimal)oci_dassuiki / 100 * 1000, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "sg_dassuiki_input_ex") {
                //脱投汚固形物量他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "遠心分離" && getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "ベルトプレス") {
                        decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ov_dassuiki != null) {
                            oci_dassuiki = oci_dassuiki ?? 0;
                            sum += ColumnsConfigSet.roundingByName((decimal)ov_dassuiki * (decimal)oci_dassuiki / 100 * 1000, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "sg_dassuiki_input_all") {
                //脱投汚固形物量脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                    decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                    if(ov_dassuiki != null) {
                        oci_dassuiki = oci_dassuiki ?? 0;
                        sum += ColumnsConfigSet.roundingByName((decimal)ov_dassuiki * (decimal)oci_dassuiki / 100 * 1000, ccName);
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_gyousyuuzai_dassuiki_enshin") {
                //高分子使用量遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "遠心分離") {
                        decimal? v_gyousyuuzai_dassuiki = getCcValue("v_gyousyuuzai_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? oc_gyousyuuzai_dassuiki = getCcValue("oc_gyousyuuzai_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(v_gyousyuuzai_dassuiki != null && oc_gyousyuuzai_dassuiki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)v_gyousyuuzai_dassuiki * (decimal)oc_gyousyuuzai_dassuiki / 100, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_gyousyuuzai_dassuiki_bp") {
                //高分子使用量BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "ベルトプレス") {
                        decimal? v_gyousyuuzai_dassuiki = getCcValue("v_gyousyuuzai_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? oc_gyousyuuzai_dassuiki = getCcValue("oc_gyousyuuzai_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(v_gyousyuuzai_dassuiki != null && oc_gyousyuuzai_dassuiki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)v_gyousyuuzai_dassuiki * (decimal)oc_gyousyuuzai_dassuiki / 100, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_gyousyuuzai_dassuiki_ex") {
                //高分子使用量他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "遠心分離" && getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "ベルトプレス") {
                        decimal? v_gyousyuuzai_dassuiki = getCcValue("v_gyousyuuzai_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? oc_gyousyuuzai_dassuiki = getCcValue("oc_gyousyuuzai_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(v_gyousyuuzai_dassuiki != null && oc_gyousyuuzai_dassuiki != null) {
                            sum += ColumnsConfigSet.roundingByName((decimal)v_gyousyuuzai_dassuiki * (decimal)oc_gyousyuuzai_dassuiki / 100, ccName);
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_gyousyuuzai_dassuiki_all") {
                //高分子使用量脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal? v_gyousyuuzai_dassuiki = getCcValue("v_gyousyuuzai_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                    decimal? oc_gyousyuuzai_dassuiki = getCcValue("oc_gyousyuuzai_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                    if(v_gyousyuuzai_dassuiki != null && oc_gyousyuuzai_dassuiki != null) {
                        sum += ColumnsConfigSet.roundingByName((decimal)v_gyousyuuzai_dassuiki * (decimal)oc_gyousyuuzai_dassuiki / 100, ccName);
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_exc1_dassuiki_enshin") {
                //他薬品1使用量遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "遠心分離") {
                        decimal? ug_exc1_dassuiki = getCcValue("ug_exc1_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ug_exc1_dassuiki != null) {
                            sum += (decimal)ug_exc1_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_exc1_dassuiki_bp") {
                //他薬品1使用量BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "ベルトプレス") {
                        decimal? ug_exc1_dassuiki = getCcValue("ug_exc1_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ug_exc1_dassuiki != null) {
                            sum += (decimal)ug_exc1_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_exc1_dassuiki_ex") {
                //他薬品1使用量他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "遠心分離" && getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "ベルトプレス") {
                        decimal? ug_exc1_dassuiki = getCcValue("ug_exc1_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ug_exc1_dassuiki != null) {
                            sum += (decimal)ug_exc1_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_exc1_dassuiki_all") {
                //他薬品1使用量脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal? ug_exc1_dassuiki = getCcValue("ug_exc1_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                    if(ug_exc1_dassuiki != null) {
                        sum += (decimal)ug_exc1_dassuiki;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_exc2_dassuiki_enshin") {
                //他薬品2使用量遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "遠心分離") {
                        decimal? ug_exc2_dassuiki = getCcValue("ug_exc2_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ug_exc2_dassuiki != null) {
                            sum += (decimal)ug_exc2_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_exc2_dassuiki_bp") {
                //他薬品2使用量BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "ベルトプレス") {
                        decimal? ug_exc2_dassuiki = getCcValue("ug_exc2_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ug_exc2_dassuiki != null) {
                            sum += (decimal)ug_exc2_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_exc2_dassuiki_ex") {
                //他薬品2使用量他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "遠心分離" && getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "ベルトプレス") {
                        decimal? ug_exc2_dassuiki = getCcValue("ug_exc2_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ug_exc2_dassuiki != null) {
                            sum += (decimal)ug_exc2_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ug_exc2_dassuiki_all") {
                //他薬品2使用量脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal? ug_exc2_dassuiki = getCcValue("ug_exc2_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                    if(ug_exc2_dassuiki != null) {
                        sum += (decimal)ug_exc2_dassuiki;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "dt_dassuiki_enshin") {
                //運転時間遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "遠心分離") {
                        decimal? ccValue = getCcValue("dt_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "dt_dassuiki_bp") {
                //運転時間BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "ベルトプレス") {
                        decimal? ccValue = getCcValue("dt_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "dt_dassuiki_ex") {
                //運転時間他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "遠心分離" && getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "ベルトプレス") {
                        decimal? ccValue = getCcValue("dt_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(ccValue != null) {
                            sum += (decimal)ccValue;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "dt_dassuiki_all") {
                //運転時間脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal? ccValue = getCcValue("dt_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_dassui_output_all") {
                //脱水汚泥発生量計
                decimal? ov_dassui_output_enshin = getCcValue("ov_dassui_output_enshin", dgv1Row, ccSet);
                decimal? ov_dassui_output_bp = getCcValue("ov_dassui_output_bp", dgv1Row, ccSet);
                decimal? ov_dassui_output_ex = getCcValue("ov_dassui_output_ex", dgv1Row, ccSet);

                if(ov_dassui_output_enshin != null || ov_dassui_output_bp != null || ov_dassui_output_ex != null) {
                    ov_dassui_output_enshin = ov_dassui_output_enshin ?? 0;
                    ov_dassui_output_bp = ov_dassui_output_bp ?? 0;
                    ov_dassui_output_ex = ov_dassui_output_ex ?? 0;

                    retValue = ov_dassui_output_enshin + ov_dassui_output_bp + ov_dassui_output_ex;
                }
            }
            else if(ccName == "sg_dassui_output_enshin") {
                //脱水汚泥固形物量遠心
                decimal? owr_dassui_output_enshin = getCcValue("owr_dassui_output_enshin", dgv1Row, ccSet);
                decimal? ov_dassui_output_enshin = getCcValue("ov_dassui_output_enshin", dgv1Row, ccSet);

                if(owr_dassui_output_enshin != null && ov_dassui_output_enshin != null) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output_enshin) / 100 * (decimal)ov_dassui_output_enshin * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sg_dassui_output_bp") {
                //脱水汚泥固形物量BP
                decimal? owr_dassui_output_bp = getCcValue("owr_dassui_output_bp", dgv1Row, ccSet);
                decimal? ov_dassui_output_bp = getCcValue("ov_dassui_output_bp", dgv1Row, ccSet);

                if(owr_dassui_output_bp != null && ov_dassui_output_bp != null) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output_bp) / 100 * (decimal)ov_dassui_output_bp * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sg_dassui_output_ex") {
                //脱水汚泥固形物量他
                decimal? owr_dassui_output_ex = getCcValue("owr_dassui_output_ex", dgv1Row, ccSet);
                decimal? ov_dassui_output_ex = getCcValue("ov_dassui_output_ex", dgv1Row, ccSet);

                if(owr_dassui_output_ex != null && ov_dassui_output_ex != null) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output_ex) / 100 * (decimal)ov_dassui_output_ex * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sg_dassui_output_all") {
                //脱水汚泥固形物量計
                decimal? owr_dassui_output_enshin = getCcValue("owr_dassui_output_enshin", dgv1Row, ccSet);
                decimal? ov_dassui_output_enshin = getCcValue("ov_dassui_output_enshin", dgv1Row, ccSet);
                decimal? sgo_enshin = null;

                if(owr_dassui_output_enshin != null && ov_dassui_output_enshin != null) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output_enshin) / 100 * (decimal)ov_dassui_output_enshin * 1000;
                    sgo_enshin = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }

                decimal? owr_dassui_output_bp = getCcValue("owr_dassui_output_bp", dgv1Row, ccSet);
                decimal? ov_dassui_output_bp = getCcValue("ov_dassui_output_bp", dgv1Row, ccSet);
                decimal? sgo_bp = null;

                if(owr_dassui_output_bp != null && ov_dassui_output_bp != null) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output_bp) / 100 * (decimal)ov_dassui_output_bp * 1000;
                    sgo_bp = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }

                decimal? owr_dassui_output_ex = getCcValue("owr_dassui_output_ex", dgv1Row, ccSet);
                decimal? ov_dassui_output_ex = getCcValue("ov_dassui_output_ex", dgv1Row, ccSet);
                decimal? sgo_ex = null;

                if(owr_dassui_output_ex != null && ov_dassui_output_ex != null) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output_ex) / 100 * (decimal)ov_dassui_output_ex * 1000;
                    sgo_ex = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }

                if(sgo_enshin != null || sgo_bp != null || sgo_ex != null) {
                    sgo_enshin = sgo_enshin ?? 0;
                    sgo_bp = sgo_bp ?? 0;
                    sgo_ex = sgo_ex ?? 0;

                    retValue = sgo_enshin + sgo_bp + sgo_ex;
                }
            }
            else if(ccName == "sgcr_dassui_output_enshin") {
                //固形物回収率遠心
                decimal? owr_dassui_output = getCcValue("owr_dassui_output_enshin", dgv1Row, ccSet);
                decimal? ov_dassui_output = getCcValue("ov_dassui_output_enshin", dgv1Row, ccSet);
                decimal? sg_dassuiki_input = null;

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "遠心分離") {
                        decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(oci_dassuiki != null && ov_dassuiki != null) {
                            sum += (decimal)ov_dassuiki * (decimal)oci_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    sg_dassuiki_input = sum / 100 * 1000;
                }

                if(owr_dassui_output != null && ov_dassui_output != null && sg_dassuiki_input != null && sg_dassuiki_input > 0) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output) / 100 * (decimal)ov_dassui_output * 1000 / (decimal)sg_dassuiki_input * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sgcr_dassui_output_bp") {
                //固形物回収率BP
                decimal? owr_dassui_output = getCcValue("owr_dassui_output_bp", dgv1Row, ccSet);
                decimal? ov_dassui_output = getCcValue("ov_dassui_output_bp", dgv1Row, ccSet);
                decimal? sg_dassuiki_input = null;

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) == "ベルトプレス") {
                        decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(oci_dassuiki != null && ov_dassuiki != null) {
                            sum += (decimal)ov_dassuiki * (decimal)oci_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    sg_dassuiki_input = sum / 100 * 1000;
                }

                if(owr_dassui_output != null && ov_dassui_output != null && sg_dassuiki_input != null && sg_dassuiki_input > 0) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output) / 100 * (decimal)ov_dassui_output * 1000 / (decimal)sg_dassuiki_input * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sgcr_dassui_output_ex") {
                //固形物回収率他
                decimal? owr_dassui_output = getCcValue("owr_dassui_output_ex", dgv1Row, ccSet);
                decimal? ov_dassui_output = getCcValue("ov_dassui_output_ex", dgv1Row, ccSet);
                decimal? sg_dassuiki_input = null;

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "遠心分離" && getCcValueStr("type_dassuiki_" + i.ToString(), dgv1Row, ccSet) != "ベルトプレス") {
                        decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                        decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                        if(oci_dassuiki != null && ov_dassuiki != null) {
                            sum += (decimal)ov_dassuiki * (decimal)oci_dassuiki;
                            count++;
                        }
                    }
                }

                if(count > 0) {
                    sg_dassuiki_input = sum / 100 * 1000;
                }

                if(owr_dassui_output != null && ov_dassui_output != null && sg_dassuiki_input != null && sg_dassuiki_input > 0) {
                    decimal retValue2 = (100 - (decimal)owr_dassui_output) / 100 * (decimal)ov_dassui_output * 1000 / (decimal)sg_dassuiki_input * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName == "sgcr_dassui_output_all") {
                //固形物回収率全体
                decimal? owr_dassui_output_enshin = getCcValue("owr_dassui_output_enshin", dgv1Row, ccSet);
                decimal? ov_dassui_output_enshin = getCcValue("ov_dassui_output_enshin", dgv1Row, ccSet);
                decimal? owr_dassui_output_bp = getCcValue("owr_dassui_output_bp", dgv1Row, ccSet);
                decimal? ov_dassui_output_bp = getCcValue("ov_dassui_output_bp", dgv1Row, ccSet);
                decimal? owr_dassui_output_ex = getCcValue("owr_dassui_output_ex", dgv1Row, ccSet);
                decimal? ov_dassui_output_ex = getCcValue("ov_dassui_output_ex", dgv1Row, ccSet);

                decimal? sg_dassui_output = null;

                if((owr_dassui_output_enshin != null && ov_dassui_output_enshin != null) || (owr_dassui_output_bp != null && ov_dassui_output_bp != null) || (owr_dassui_output_ex != null && ov_dassui_output_ex != null)) {
                    sg_dassui_output = 0;
                    sg_dassui_output += (100 - owr_dassui_output_enshin) / 100 * ov_dassui_output_enshin * 1000 ?? 0;
                    sg_dassui_output += (100 - owr_dassui_output_bp) / 100 * ov_dassui_output_bp * 1000 ?? 0;
                    sg_dassui_output += (100 - owr_dassui_output_ex) / 100 * ov_dassui_output_ex * 1000 ?? 0;
                }

                decimal? sg_dassuiki_input = null;

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + i.ToString(), dgv1Row, ccSet);
                    decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + i.ToString(), dgv1Row, ccSet);

                    if(oci_dassuiki != null && ov_dassuiki != null) {
                        sum += (decimal)ov_dassuiki * (decimal)oci_dassuiki;
                        count++;
                    }
                }

                if(count > 0) {
                    sg_dassuiki_input = sum / 100 * 1000;
                }

                if(sg_dassui_output != null && sg_dassuiki_input != null && sg_dassuiki_input > 0) {
                    decimal retValue2 = (decimal)sg_dassui_output / (decimal)sg_dassuiki_input * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }

            #endregion


            #region 汚泥等搬出

            else if(ccName == "ov_hansyutsu_chinsa_pumpjou_all") {
                //P場沈砂搬出量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue = getCcValue("ov_hansyutsu_chinsa_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_hansyutsu_shisa_pumpjou_all") {
                //P場しさ搬出量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue = getCcValue("ov_hansyutsu_shisa_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue != null) {
                        sum += (decimal)ccValue;
                        count++;
                    }
                }

                if(count > 0) {
                    retValue = sum;
                }
            }
            else if(ccName == "ov_hansyutsu_all") {
                //汚泥等搬出量計
                decimal? ov_hansyutsu_chinsa_pumpjou = null;
                decimal? ov_hansyutsu_shisa_pumpjou = null;

                decimal sum_pc = 0;
                int count_pc = 0;

                decimal sum_ps = 0;
                int count_ps = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    decimal? ccValue_pc = getCcValue("ov_hansyutsu_chinsa_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue_pc != null) {
                        sum_pc += (decimal)ccValue_pc;
                        count_pc++;
                    }

                    decimal? ccValue_ps = getCcValue("ov_hansyutsu_shisa_pumpjou_" + i.ToString(), dgv1Row, ccSet);

                    if(ccValue_ps != null) {
                        sum_ps += (decimal)ccValue_ps;
                        count_ps++;
                    }
                }

                if(count_pc > 0) {
                    ov_hansyutsu_chinsa_pumpjou = sum_pc;
                }

                if(count_ps > 0) {
                    ov_hansyutsu_shisa_pumpjou = sum_ps;
                }

                decimal? ov_hansyutsu_chinsa = getCcValue("ov_hansyutsu_chinsa", dgv1Row, ccSet);
                decimal? ov_hansyutsu_shisa = getCcValue("ov_hansyutsu_shisa", dgv1Row, ccSet);
                decimal? ov_hansyutsu_namaodei = getCcValue("ov_hansyutsu_namaodei", dgv1Row, ccSet);
                decimal? ov_hansyutsu_yojouodei = getCcValue("ov_hansyutsu_yojouodei", dgv1Row, ccSet);
                decimal? ov_hansyutsu_nousyukuodei = getCcValue("ov_hansyutsu_nousyukuodei", dgv1Row, ccSet);
                decimal? ov_hansyutsu_syoukaodei = getCcValue("ov_hansyutsu_syoukaodei", dgv1Row, ccSet);
                decimal? ov_hansyutsu_dassuiodei = getCcValue("ov_hansyutsu_dassuiodei", dgv1Row, ccSet);

                if(ov_hansyutsu_chinsa_pumpjou != null || ov_hansyutsu_shisa_pumpjou != null || ov_hansyutsu_chinsa != null || ov_hansyutsu_shisa != null || ov_hansyutsu_namaodei != null || ov_hansyutsu_yojouodei != null || ov_hansyutsu_nousyukuodei != null || ov_hansyutsu_syoukaodei != null || ov_hansyutsu_dassuiodei != null) {
                    decimal sum = 0;

                    sum += ov_hansyutsu_chinsa_pumpjou / 1000 ?? 0;
                    sum += ov_hansyutsu_shisa_pumpjou / 1000 ?? 0;
                    sum += ov_hansyutsu_chinsa ?? 0;
                    sum += ov_hansyutsu_shisa ?? 0;
                    sum += ov_hansyutsu_namaodei ?? 0;
                    sum += ov_hansyutsu_yojouodei ?? 0;
                    sum += ov_hansyutsu_nousyukuodei ?? 0;
                    sum += ov_hansyutsu_syoukaodei ?? 0;
                    sum += ov_hansyutsu_dassuiodei ?? 0;

                    retValue = ColumnsConfigSet.roundingByName(sum, ccName);
                }
            }

            #endregion


            #region 反応T個別計算定義

            else if(ccName.Length > 18 && ccName.Substring(0, 18) == "vsspss_hannoutank_") {
                //反応TnVSS/SS
                string tankNo = ccName.Replace("vsspss_hannoutank_", "");

                decimal? mlvss_hannoutunk = getCcValue("mlvss_hannoutank_" + tankNo, dgv1Row, ccSet);
                decimal? mlss_hannoutunk = getCcValue("mlss_hannoutank_" + tankNo, dgv1Row, ccSet);

                if(mlvss_hannoutunk != null && mlss_hannoutunk != null) {
                    decimal retValue2 = (decimal)mlvss_hannoutunk / (decimal)mlss_hannoutunk * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName.Length > 15 && ccName.Substring(0, 15) == "svi_hannoutank_") {
                //反応TnSVI
                string tankNo = ccName.Replace("svi_hannoutank_", "");

                decimal? sv_hannoutank = getCcValue("sv_hannoutank_" + tankNo, dgv1Row, ccSet);
                decimal? mlss_hannoutunk = getCcValue("mlss_hannoutank_" + tankNo, dgv1Row, ccSet);

                if(sv_hannoutank != null && mlss_hannoutunk != null && mlss_hannoutunk > 0) {
                    decimal retValue2 = (decimal)sv_hannoutank * 10000 / (decimal)mlss_hannoutunk;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName.Length > 20 && ccName.Substring(0, 20) == "bodpmlss_hannoutank_") {
                //反応TnBOD/MLSS負荷
                decimal? wv_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 2)) {
                    case "初沈流入水量":
                        wv_kijun = getCcValue("wv_syochin", dgv1Row, ccSet);
                        break;
                    case "流入水量":
                        wv_kijun = getCcValue("wv_ryuunyuu", dgv1Row, ccSet);
                        break;
                    case "放流水量":
                        wv_kijun = getCcValue("wv_houryuu", dgv1Row, ccSet);
                        break;
                }

                decimal? wv_kani = getCcValue("wv_kani", dgv1Row, ccSet);
                decimal? bod_kijun = null;

                switch(SyorijouConfigSet.getConfigString(kanrisya, syorijou, 6)) {
                    case "初沈越流水質":
                        bod_kijun = getCcValue("bod_syo_etsuryuu", dgv1Row, ccSet);
                        break;
                    case "流入水質":
                        bod_kijun = getCcValue("bod_ryuunyuu", dgv1Row, ccSet);
                        break;
                }

                decimal? pv_hannoutank = getCcValue("pv_hannoutank", dgv1Row, ccSet);

                string tankNo = ccName.Replace("bodpmlss_hannoutank_", "");

                decimal? mlss_hannoutank = getCcValue("mlss_hannoutank_" + tankNo, dgv1Row, ccSet);

                if(wv_kijun != null && bod_kijun != null && pv_hannoutank != null && mlss_hannoutank != null && pv_hannoutank > 0 && mlss_hannoutank > 0) {
                    wv_kani = wv_kani ?? 0;

                    decimal retValue2 = ((decimal)wv_kijun - (decimal)wv_kani) * (decimal)bod_kijun / ((decimal)pv_hannoutank * (decimal)mlss_hannoutank);
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }

            #endregion


            #region 濃縮設備個別計算定義

            else if(ccName.Length > 14 && ccName.Substring(0, 14) == "sg_nousyukuki_") {
                //濃縮設備n投入汚泥固形物量
                string nousyukukiNo = ccName.Replace("sg_nousyukuki_", "");

                decimal? oci_nousyukuki = getCcValue("oci_nousyukuki_" + nousyukukiNo, dgv1Row, ccSet);
                decimal? ov_nousyukuki = getCcValue("ov_nousyukuki_" + nousyukukiNo, dgv1Row, ccSet);

                if(oci_nousyukuki != null && ov_nousyukuki != null) {
                    decimal retValue2 = (decimal)oci_nousyukuki * (decimal)ov_nousyukuki / 100 * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName.Length > 26 && ccName.Substring(0, 26) == "ug_gyousyuuzai_nousyukuki_") {
                //濃縮設備n高分子凝集剤使用量
                string nousyukukiNo = ccName.Replace("ug_gyousyuuzai_nousyukuki_", "");

                decimal? v_gyousyuuzai_nousyukuki = getCcValue("v_gyousyuuzai_nousyukuki_" + nousyukukiNo, dgv1Row, ccSet);
                decimal? oc_gyousyuuzai_nousyukuki = getCcValue("oc_gyousyuuzai_nousyukuki_" + nousyukukiNo, dgv1Row, ccSet);

                if(v_gyousyuuzai_nousyukuki != null && oc_gyousyuuzai_nousyukuki != null) {
                    decimal retValue2 = (decimal)v_gyousyuuzai_nousyukuki * (decimal)oc_gyousyuuzai_nousyukuki / 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName.Length > 25 && ccName.Substring(0, 25) == "ug_kihoujozai_nousyukuki_") {
                //濃縮設備n起泡助剤使用量
                string nousyukukiNo = ccName.Replace("ug_kihoujozai_nousyukuki_", "");

                decimal? v_kihoujozai_nousyukuki = getCcValue("v_kihoujozai_nousyukuki_" + nousyukukiNo, dgv1Row, ccSet);
                decimal? oc_kihoujozai_nousyukuki = getCcValue("oc_kihoujozai_nousyukuki_" + nousyukukiNo, dgv1Row, ccSet);

                if(v_kihoujozai_nousyukuki != null && oc_kihoujozai_nousyukuki != null) {
                    decimal retValue2 = (decimal)v_kihoujozai_nousyukuki * (decimal)oc_kihoujozai_nousyukuki / 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }

            #endregion


            #region 脱水設備個別計算定義

            else if(ccName.Length > 12 && ccName.Substring(0, 12) == "sg_dassuiki_") {
                //脱水設備n投入汚泥固形物量
                string dassuikiNo = ccName.Replace("sg_dassuiki_", "");

                decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + dassuikiNo, dgv1Row, ccSet);

                if(oci_dassuiki != null && ov_dassuiki != null) {
                    decimal retValue2 = (decimal)oci_dassuiki * (decimal)ov_dassuiki / 100 * 1000;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName.Length > 24 && ccName.Substring(0, 24) == "ug_gyousyuuzai_dassuiki_") {
                //脱水設備n高分子凝集剤使用量
                string dassuikiNo = ccName.Replace("ug_gyousyuuzai_dassuiki_", "");

                decimal? v_gyousyuuzai_dassuiki = getCcValue("v_gyousyuuzai_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? oc_gyousyuuzai_dassuiki = getCcValue("oc_gyousyuuzai_dassuiki_" + dassuikiNo, dgv1Row, ccSet);

                if(v_gyousyuuzai_dassuiki != null && oc_gyousyuuzai_dassuiki != null) {
                    decimal retValue2 = (decimal)v_gyousyuuzai_dassuiki * (decimal)oc_gyousyuuzai_dassuiki / 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName.Length > 23 && ccName.Substring(0, 23) == "r_gyousyuuzai_dassuiki_") {
                //脱水設備n高分子凝集剤注入率
                string dassuikiNo = ccName.Replace("r_gyousyuuzai_dassuiki_", "");

                decimal? v_gyousyuuzai_dassuiki = getCcValue("v_gyousyuuzai_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? oc_gyousyuuzai_dassuiki = getCcValue("oc_gyousyuuzai_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + dassuikiNo, dgv1Row, ccSet);

                if(v_gyousyuuzai_dassuiki != null && oc_gyousyuuzai_dassuiki != null && oci_dassuiki != null && ov_dassuiki != null && (oci_dassuiki / 100 * ov_dassuiki * 1000) > 0) {
                    decimal retValue2 = ((decimal)v_gyousyuuzai_dassuiki * (decimal)oc_gyousyuuzai_dassuiki / 100) / ((decimal)oci_dassuiki / 100 * (decimal)ov_dassuiki * 1000) * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName.Length > 16 && ccName.Substring(0, 16) == "r_exc1_dassuiki_") {
                //脱水設備n他薬品1注入率
                string dassuikiNo = ccName.Replace("r_exc1_dassuiki_", "");

                decimal? ug_exc1_dassuiki = getCcValue("ug_exc1_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + dassuikiNo, dgv1Row, ccSet);

                if(ug_exc1_dassuiki != null && oci_dassuiki != null && ov_dassuiki != null && (oci_dassuiki / 100 * ov_dassuiki * 1000) > 0) {
                    decimal retValue2 = (decimal)ug_exc1_dassuiki / ((decimal)oci_dassuiki / 100 * (decimal)ov_dassuiki * 1000) * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }
            else if(ccName.Length > 16 && ccName.Substring(0, 16) == "r_exc2_dassuiki_") {
                //脱水設備n他薬品2注入率
                string dassuikiNo = ccName.Replace("r_exc2_dassuiki_", "");

                decimal? ug_exc2_dassuiki = getCcValue("ug_exc2_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? oci_dassuiki = getCcValue("oci_dassuiki_" + dassuikiNo, dgv1Row, ccSet);
                decimal? ov_dassuiki = getCcValue("ov_dassuiki_" + dassuikiNo, dgv1Row, ccSet);

                if(ug_exc2_dassuiki != null && oci_dassuiki != null && ov_dassuiki != null && (oci_dassuiki / 100 * ov_dassuiki * 1000) > 0) {
                    decimal retValue2 = (decimal)ug_exc2_dassuiki / ((decimal)oci_dassuiki / 100 * (decimal)ov_dassuiki * 1000) * 100;
                    retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
                }
            }

            #endregion


            //【ローカル関数】計算元データ取得
            decimal? getCcValue(string ccName2, DataGridViewRow dgv1Row2 = null, List<ColumnConfig> ccSet2 = null) {
                decimal? retValue2 = null;

                if(dgv1Row2 != null) {
                    if(dgv1Row.Cells[ccName2].Value != null) {
                        decimal retValue2t = 0;

                        if(decimal.TryParse(dgv1Row.Cells[ccName2].Value.ToString(), out retValue2t)) {
                            retValue2 = retValue2t;
                        }
                    }
                }
                else if(ccSet2 != null) {
                    retValue2 = ccSet2[ColumnsConfigSet.nameToIndex(ccName2)].tempValueDec;
                }

                return retValue2;
            }


            //【ローカル関数】計算元データ取得(文字列)
            string getCcValueStr(string ccName2, DataGridViewRow dgv1Row2 = null, List<ColumnConfig> ccSet2 = null) {
                string retValue2 = null;

                if(dgv1Row2 != null) {
                    if(dgv1Row.Cells[ccName2].Value != null) {
                        retValue2 = dgv1Row.Cells[ccName2].Value.ToString();
                    }
                }
                else if(ccSet2 != null) {
                    retValue2 = ccSet2[ColumnsConfigSet.nameToIndex(ccName2)].tempValueStr;
                }

                return retValue2;
            }


            return retValue;
        }




        //二次データを求める計算（文字列戻り）
        public static string calc2ndString(string ccName, DataGridViewRow dgv1Row = null, List<ColumnConfig> ccSet = null) {
            string retValue = null;

            switch(ccName) {
                case "rei":

                    break;
            }

            return retValue;
        }

    }
}
