﻿// 下水データ管理システム
// Copyright (C) 2018 公益財団法人とちぎ建設技術センター(https://www.tochigictc.or.jp/)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.If not, see<http://www.gnu.org/licenses/>.

using ClosedXML.Excel;
using System;
using System.ComponentModel;
using System.Data.SQLite;
using System.IO;
using System.Windows.Forms;

namespace WindowsFormsApplication1 {
    public partial class FormD1MonthlyReportSelect : Form {
        public FormD1MonthlyReportSelect() {
            InitializeComponent();
        }

        int cYear = 0;
        int cMonth = 0;

        BindingList<KanrisyaSyorijou> KSCBItem = new BindingList<KanrisyaSyorijou>();
        BindingList<SelectableMonth> SMCBItem = new BindingList<SelectableMonth>();




        //ロード時イベント
        private void FormD1MonthlyReportSelect_Load(object sender, EventArgs e) {
            //選択リスト準備
            dataSelectInitialize();
            monthSelectInitialize();

            dataSelectConstruction();
            monthSelectConstruction();
        }




        //管理者・処理場選択ボックス事前準備
        private void dataSelectInitialize() {
            comboBox1.DisplayMember = "selectionName";
            comboBox1.ValueMember = "primaryKey";
            comboBox1.DataSource = KSCBItem;
            comboBox1.SelectedIndexChanged += new EventHandler(dataSelect_SelectedIndexChanged);
        }




        //月選択ボックス事前準備
        private void monthSelectInitialize() {
            comboBox2.DisplayMember = "selectionName";
            comboBox2.ValueMember = "primaryKey";
            comboBox2.DataSource = SMCBItem;
            comboBox2.SelectedIndexChanged += new EventHandler(monthSelect_SelectedIndexChanged);
        }




        //管理者・処理場選択ボックス生成
        private void dataSelectConstruction(string cKanrisya = null, string cSyorijou = null) {
            comboBox1.SelectedIndexChanged -= dataSelect_SelectedIndexChanged;
            comboBox1.BeginUpdate();
            KSCBItem.Clear();

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    cmd.CommandText = "SELECT t1.kanrisya, t1.syorijou, t2.ck FROM suishitsu_kanri AS t1 " +
                        "INNER JOIN (SELECT kanrisya, COUNT(kanrisya) AS ck FROM suishitsu_kanri WHERE del_flg IS NULL GROUP BY kanrisya) AS t2 ON t1.kanrisya = t2.kanrisya " +
                        "GROUP BY t1.kanrisya, t1.syorijou HAVING t2.ck > 0 ORDER BY t2.ck DESC, COUNT(t1.syorijou) DESC";

                    using(var reader = cmd.ExecuteReader()) {
                        int i = 0;
                        int c = 0;
                        while(reader.Read()) {
                            string kanrisya = reader.GetString(0);
                            string syorijou = reader.GetString(1);
                            KSCBItem.Add(new KanrisyaSyorijou(kanrisya, syorijou));

                            if(kanrisya == cKanrisya && syorijou == cSyorijou) {
                                c = i;
                            }

                            i++;
                        }

                        if(i > 0) {
                            comboBox1.Enabled = true;
                            comboBox1.SelectedIndex = c;
                        }
                        else {
                            comboBox1.Enabled = false;
                        }
                    }
                }
            }

            comboBox1.EndUpdate();
            comboBox1.SelectedIndexChanged += new EventHandler(dataSelect_SelectedIndexChanged);
        }




        //月選択ボックス生成
        private void monthSelectConstruction(int cYear = -1, int cMonth = -1) {
            comboBox2.SelectedIndexChanged -= monthSelect_SelectedIndexChanged;
            comboBox2.BeginUpdate();
            SMCBItem.Clear();

            if(comboBox1.SelectedItem != null) {
                comboBox2.Enabled = true;

                KanrisyaSyorijou item1 = (KanrisyaSyorijou)comboBox1.SelectedItem;
                string kanrisya = item1.primaryKey(0);
                string syorijou = item1.primaryKey(1);

                using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                    con.Open();

                    using(var cmd = con.CreateCommand()) {
                        cmd.CommandText = "SELECT MIN(sokutei_dt) AS min_dt, MAX(sokutei_dt) AS max_dt FROM suishitsu_kanri WHERE del_flg IS NULL;";

                        using(var reader = cmd.ExecuteReader()) {
                            if(reader.Read()) {
                                if(!reader.IsDBNull(0) && !reader.IsDBNull(1)) {
                                    //存在する年月の最小と最大
                                    DateTime minDt = DateTime.Parse(reader.GetString(0));
                                    DateTime maxDt = DateTime.Parse(reader.GetString(1));

                                    DateTime stDt = DateTime.Parse(minDt.Year.ToString() + "/" + minDt.Month.ToString() + "/1 00:00:00");
                                    DateTime edDt = DateTime.Parse(maxDt.Year.ToString() + "/" + maxDt.Month.ToString() + "/1 00:00:00");

                                    int i = 0;
                                    int c = 0;
                                    for(DateTime cDt = stDt; cDt <= edDt; cDt = cDt.AddMonths(1)) {
                                        DateTime cnDt = cDt.AddMonths(1);

                                        using(var cmd2 = con.CreateCommand()) {
                                            cmd2.CommandText = "SELECT COUNT(*) FROM suishitsu_kanri WHERE kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "'  AND sokutei_dt>='" + cDt.ToString("yyyy-MM-dd HH:mm:ss") + "' AND sokutei_dt<'" + cnDt.ToString("yyyy-MM-dd HH:mm:ss") + "' AND del_flg IS NULL";

                                            using(var reader2 = cmd2.ExecuteReader()) {
                                                if(reader2.Read()) {
                                                    if(reader2.GetInt32(0) > 0) {
                                                        //選択中の管理者処理場のデータがある月
                                                        SMCBItem.Add(new SelectableMonth(cDt.Year, cDt.Month));
                                                    }
                                                    else {
                                                        //選択中の管理者処理場のデータがない場合は空欄
                                                        SMCBItem.Add(new SelectableMonth(-1, -1));
                                                    }
                                                }
                                            }
                                        }

                                        if(cYear == -1 || cMonth == -1) {
                                            //指定がない場合は先月を選択中に
                                            DateTime lastMonth = DateTime.Now.AddMonths(-1);

                                            if(cDt.Year == lastMonth.Year && cDt.Month == lastMonth.Month) {
                                                c = i;
                                            }
                                        }
                                        else {
                                            if(cDt.Year == cYear && cDt.Month == cMonth) {
                                                c = i;
                                            }
                                        }

                                        i++;
                                    }

                                    comboBox2.SelectedIndex = c;
                                }
                                else {
                                    comboBox2.Enabled = false;
                                }
                            }
                        }
                    }
                }
            }
            else {
                comboBox2.Enabled = false;
            }

            comboBox2.EndUpdate();
            comboBox2.SelectedIndexChanged += new EventHandler(monthSelect_SelectedIndexChanged);
        }




        //管理者・処理場選択ボックス選択変更イベント
        private void dataSelect_SelectedIndexChanged(object sender, EventArgs e) {
            monthSelectConstruction(cYear, cMonth);
            okButtonSwitch();
        }




        //月選択ボックス選択変更イベント
        private void monthSelect_SelectedIndexChanged(object sender, EventArgs e) {
            SelectableMonth item2 = (SelectableMonth)comboBox2.SelectedItem;

            if(item2 != null) {
                if(item2.primaryKey(0) != -1 && item2.primaryKey(1) != -1) {
                    cYear = item2.primaryKey(0);
                    cMonth = item2.primaryKey(1);
                }
            }

            okButtonSwitch();
        }




        //管理者・処理場、月が無効な場合、OKボタンを無効に
        private void okButtonSwitch() {
            if(comboBox1.Enabled && comboBox2.Enabled) {
                SelectableMonth item2 = (SelectableMonth)comboBox2.SelectedItem;

                if(item2 != null) {
                    if(item2.primaryKey(0) != -1 && item2.primaryKey(1) != -1) {
                        button1.Enabled = true;
                        return;
                    }
                }
            }

            button1.Enabled = false;
        }




        //OKボタンクリックイベント
        private void button1_Click(object sender, EventArgs e) {
            monthlyReportExportExcel();
        }




        //月報出力
        private void monthlyReportExportExcel() {
            SaveFileDialog saveFileDialog1 = new SaveFileDialog();

            KanrisyaSyorijou item1 = (KanrisyaSyorijou)comboBox1.SelectedItem;
            string kanrisya = item1.primaryKey(0);
            string syorijou = item1.primaryKey(1);

            SelectableMonth item2 = (SelectableMonth)comboBox2.SelectedItem;
            int cDtY = item2.primaryKey(0);
            int cDtM = item2.primaryKey(1);

            //年度範囲
            int mP = 0;

            if(cDtM < 4) {
                mP = 1;
            }

            string stBYSTR = (cDtY - mP).ToString() + "-04-01 00:00:00";
            string edBYSTR = (cDtY - mP + 1).ToString() + "-04-01 00:00:00";

            //月範囲
            DateTime stDt = new DateTime(cDtY, cDtM, 1, 0, 0, 0);
            DateTime edDt = stDt.AddMonths(1);
            int monthLastDay = edDt.AddDays(-1).Day;

            string stDtSTR = stDt.ToString("yyyy-MM-dd HH:mm:ss");
            string edDtSTR = edDt.ToString("yyyy-MM-dd HH:mm:ss");

            //年度集計の範囲は年度開始日から月報を出力する月の末までとする
            edBYSTR = edDtSTR;

            //ファイル名
            string kikan;

            if(cDtY < 1 || cDtM < 1) {
                return;
            }
            else {
                kikan = ("0000" + cDtY.ToString()).Substring(cDtY.ToString().Length, 4) + ("00" + cDtM.ToString()).Substring(cDtM.ToString().Length, 2);
            }

            string fileName = "月報_" + kanrisya + "_" + syorijou + "_" + kikan;
            string fileNameExtension = "xlsx";

            saveFileDialog1.FileName = fileName + "." + fileNameExtension;
            saveFileDialog1.InitialDirectory = @"%USERPROFILE%\Desktop";
            saveFileDialog1.Filter = "Excel ブック(*.xlsx)|*.xlsx";
            saveFileDialog1.FilterIndex = 1;
            saveFileDialog1.Title = "月報出力";
            saveFileDialog1.RestoreDirectory = true;
            saveFileDialog1.OverwritePrompt = true;
            saveFileDialog1.CheckPathExists = true;

            //ダイアログの表示
            if(saveFileDialog1.ShowDialog() == DialogResult.OK) {
                FormA2ProgressSplash progressSplash1 = new FormA2ProgressSplash();
                progressSplash1.Owner = this;
                progressSplash1.Show();
                progressSplash1.Refresh();

                progressSplash1.ProgressMessage = "ファイル出力準備中";
                Application.DoEvents();

                fileName = saveFileDialog1.FileName;
                fileNameExtension = fileName.Substring(fileName.Length - 5);

                if(fileNameExtension != ".xlsx") {
                    fileName += ".xlsx";
                }

                try {
                    using(XLWorkbook workbook1 = new XLWorkbook(@"Template\monthly_report.xlsx")) {
                        progressSplash1.ProgressMessage = "水処理概況シート作成中";
                        Application.DoEvents();
                        makeSheet1(workbook1);

                        progressSplash1.ProgressMessage = "水処理運転月報シート作成中";
                        Application.DoEvents();
                        makeSheet2(workbook1);

                        progressSplash1.ProgressMessage = "水質試験月報1シート作成中";
                        Application.DoEvents();
                        makeSheet3(workbook1);

                        progressSplash1.ProgressMessage = "水質試験月報2シート作成中";
                        Application.DoEvents();
                        makeSheet4(workbook1);

                        progressSplash1.ProgressMessage = "水質試験月報3シート作成中";
                        Application.DoEvents();
                        makeSheet5(workbook1);

                        progressSplash1.ProgressMessage = "水質試験月報4シート作成中";
                        Application.DoEvents();
                        makeSheet6(workbook1);

                        progressSplash1.ProgressMessage = "反応タンク試験月報1シート作成中";
                        Application.DoEvents();
                        makeSheet7(workbook1);

                        progressSplash1.ProgressMessage = "反応タンク試験月報2シート作成中";
                        Application.DoEvents();
                        makeSheet8(workbook1);

                        progressSplash1.ProgressMessage = "消化槽月報シート作成中";
                        Application.DoEvents();
                        makeSheet9(workbook1);

                        progressSplash1.ProgressMessage = "汚泥処理月報1シート作成中";
                        Application.DoEvents();
                        makeSheet10(workbook1);

                        progressSplash1.ProgressMessage = "汚泥処理月報2シート作成中";
                        Application.DoEvents();
                        makeSheet11(workbook1);

                        progressSplash1.ProgressMessage = "汚泥処理月報3シート作成中";
                        Application.DoEvents();
                        makeSheet12(workbook1);

                        progressSplash1.ProgressMessage = "汚泥処理月報4シート作成中";
                        Application.DoEvents();
                        makeSheet13(workbook1);

                        progressSplash1.ProgressMessage = "施設管理月報1シート作成中";
                        Application.DoEvents();
                        makeSheet14(workbook1);

                        progressSplash1.ProgressMessage = "施設管理月報2シート作成中";
                        Application.DoEvents();
                        makeSheet15(workbook1);

                        progressSplash1.ProgressMessage = "運転時間月報1シート作成中";
                        Application.DoEvents();
                        makeSheet16(workbook1);

                        progressSplash1.ProgressMessage = "運転時間月報2シート作成中";
                        Application.DoEvents();
                        makeSheet17(workbook1);

                        progressSplash1.ProgressMessage = "運転時間月報3シート作成中";
                        Application.DoEvents();
                        makeSheet18(workbook1);

                        //ファイルの保存
                        progressSplash1.ProgressMessage = "ファイル保存中";
                        Application.DoEvents();

                        workbook1.SaveAs(fileName);
                        MessageBox.Show("月報の出力が完了しました");
                    }
                }
                catch(DirectoryNotFoundException) {
                    MessageBox.Show("テンプレートファイルのディレクトリがありません");
                }
                catch(FileNotFoundException) {
                    MessageBox.Show("テンプレートファイルがありません");
                }
                catch(IOException) {
                    MessageBox.Show("ファイルが使用中等の理由により、保存できませんでした");
                }
                catch(Exception ex) {
                    MessageBox.Show(ex.Message);
                }
                finally {
                    progressSplash1.Close();
                    progressSplash1.Dispose();
                    this.Activate();
                }
            }




            //水処理概況シートの記入
            void makeSheet1(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("水処理概況", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 11).SetValue(cDtY);
                    worksheet1.Cell(1, 13).SetValue(cDtM);
                    worksheet1.Cell(3, 8).SetValue(kanrisya + "　" + syorijou);

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(wv_ryuunyuu), " +
                                "SUM(CASE WHEN seiten='○' THEN wv_ryuunyuu ELSE NULL END), " +
                                "SUM(wv_houryuu), " +
                                "SUM(IFNULL(ov_hansyutsu_namaodei, 0) + IFNULL(ov_hansyutsu_yojouodei, 0) + IFNULL(ov_hansyutsu_nousyukuodei, 0) + IFNULL(ov_hansyutsu_syoukaodei, 0) + IFNULL(ov_hansyutsu_dassuiodei, 0)), " +
                                "SUM(IFNULL(ov_hansyutsu_shisa_pumpjou_all, 0)), " +
                                "SUM(IFNULL(ov_hansyutsu_shisa, 0)), " +
                                "SUM(IFNULL(ov_hansyutsu_chinsa_pumpjou_all, 0)), " +
                                "SUM(IFNULL(ov_hansyutsu_chinsa, 0)) " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(6, 19).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(7, 19).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(8, 19).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("wv_houryuu")));

                                    worksheet1.Cell(26, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ov_hansyutsu_all")));

                                    decimal pShisa = (decimal)FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ov_hansyutsu_shisa_pumpjou_all"));
                                    decimal sShisa = (decimal)FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("ov_hansyutsu_shisa"));

                                    worksheet1.Cell(27, 16).SetValue(pShisa / 1000 + sShisa);

                                    decimal pChinsa = (decimal)FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ov_hansyutsu_chinsa_pumpjou_all"));
                                    decimal sChinsa = (decimal)FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("ov_hansyutsu_chinsa"));

                                    worksheet1.Cell(28, 16).SetValue(pChinsa / 1000 + sChinsa);
                                }
                            }
                        }
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(wv_ryuunyuu), AVG(wv_ryuunyuu), MAX(wv_ryuunyuu), MIN(wv_ryuunyuu), " +
                                "AVG(CASE WHEN seiten='○' THEN wv_ryuunyuu ELSE NULL END), MAX(CASE WHEN seiten='○' THEN wv_ryuunyuu ELSE NULL END), MIN(CASE WHEN seiten='○' THEN wv_ryuunyuu ELSE NULL END), SUM(CASE WHEN seiten='○' THEN 1 ELSE 0 END), " +
                                "SUM(wv_houryuu), AVG(wv_houryuu), MAX(wv_houryuu), MIN(wv_houryuu), " +
                                "SUM(IFNULL(ov_hansyutsu_namaodei, 0) + IFNULL(ov_hansyutsu_yojouodei, 0) + IFNULL(ov_hansyutsu_nousyukuodei, 0) + IFNULL(ov_hansyutsu_syoukaodei, 0) + IFNULL(ov_hansyutsu_dassuiodei, 0)), " +
                                "SUM(IFNULL(ov_hansyutsu_shisa_pumpjou_all, 0)), " +
                                "SUM(IFNULL(ov_hansyutsu_shisa, 0)), " +
                                "SUM(IFNULL(ov_hansyutsu_chinsa_pumpjou_all, 0)), " +
                                "SUM(IFNULL(ov_hansyutsu_chinsa, 0)), " +
                                "SUM(IFNULL(ov_dassuiki_input_all, 0)), " +
                                "AVG(pv_syochin), AVG(pv_hannoutank), AVG(pv_syuuchin), AVG(pv_enkon), AVG(pv_syouka), " +
                                "AVG(ph_ryuunyuu), MAX(ph_ryuunyuu), MIN(ph_ryuunyuu), " +
                                "AVG(bod_ryuunyuu), MAX(bod_ryuunyuu), MIN(bod_ryuunyuu), " +
                                "AVG(ss_ryuunyuu), MAX(ss_ryuunyuu), MIN(ss_ryuunyuu), " +
                                "AVG(ph_enkondeguchi), MAX(ph_enkondeguchi), MIN(ph_enkondeguchi), " +
                                "AVG(bod_enkondeguchi), MAX(bod_enkondeguchi), MIN(bod_enkondeguchi), " +
                                "AVG(ss_enkondeguchi), MAX(ss_enkondeguchi), MIN(ss_enkondeguchi), " +
                                "AVG(nd_enkondeguchi), MAX(nd_enkondeguchi), MIN(nd_enkondeguchi), " +
                                "AVG(pr_enkondeguchi), MAX(pr_enkondeguchi), MIN(pr_enkondeguchi), " +
                                "AVG(ph_houryuukou), MAX(ph_houryuukou), MIN(ph_houryuukou), " +
                                "AVG(bod_houryuukou), MAX(bod_houryuukou), MIN(bod_houryuukou), " +
                                "AVG(ss_houryuukou), MAX(ss_houryuukou), MIN(ss_houryuukou), " +
                                "AVG(nd_houryuukou), MAX(nd_houryuukou), MIN(nd_houryuukou), " +
                                "AVG(pr_houryuukou), MAX(pr_houryuukou), MIN(pr_houryuukou), ";

                            for(int i = 1; i <= MaxSizeConfig.maxSyochin; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN COUNT(name_syochin_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN COUNT(name_hannoutank_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            for(int i = 1; i <= MaxSizeConfig.maxSyuuchin; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN COUNT(name_syuuchin_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            for(int i = 1; i <= MaxSizeConfig.maxSyoukatank; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN COUNT(name_syoukatank_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN COUNT(name_dassuiki_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            for(int i = 1; i <= MaxSizeConfig.maxSyochin; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN SUM(dt_syochin_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN SUM(dt_hannoutank_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            for(int i = 1; i <= MaxSizeConfig.maxSyuuchin; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN SUM(dt_syuuchin_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            for(int i = 1; i <= MaxSizeConfig.maxSyoukatank; i++) {
                                if(i > 1) {
                                    cmd.CommandText += " + ";
                                }

                                cmd.CommandText += "(CASE WHEN SUM(dt_syoukatank_" + i.ToString() + ")>0 THEN 1 ELSE 0 END)";
                            }

                            cmd.CommandText += ", ";

                            cmd.CommandText += "SUM(dt_dassuiki_all) " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(6, 18).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(6, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(6, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(6, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));

                                    worksheet1.Cell(7, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(7, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(7, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(4, 19).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 7));

                                    worksheet1.Cell(8, 18).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("wv_houryuu")));
                                    worksheet1.Cell(8, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("wv_houryuu")));
                                    worksheet1.Cell(8, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("wv_houryuu")));
                                    worksheet1.Cell(8, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("wv_houryuu")));

                                    worksheet1.Cell(26, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("ov_hansyutsu_all")));
                                    worksheet1.Cell(26, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("ov_hansyutsu_all")) / monthLastDay);

                                    decimal pShisa = (decimal)FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("ov_hansyutsu_shisa_pumpjou_all"));
                                    decimal sShisa = (decimal)FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("ov_hansyutsu_shisa"));

                                    worksheet1.Cell(27, 15).SetValue(pShisa / 1000 + sShisa);

                                    decimal pChinsa = (decimal)FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("ov_hansyutsu_chinsa_pumpjou_all"));
                                    decimal sChinsa = (decimal)FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("ov_hansyutsu_chinsa"));

                                    worksheet1.Cell(28, 15).SetValue(pChinsa / 1000 + sChinsa);

                                    worksheet1.Cell(41, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_all")));

                                    worksheet1.Cell(32, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("pv_syochin")));
                                    worksheet1.Cell(33, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("pv_hannoutank")));
                                    worksheet1.Cell(34, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("pv_syuuchin")));
                                    worksheet1.Cell(35, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("pv_enkon")));
                                    worksheet1.Cell(36, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 22), ColumnsConfigSet.nameToFormat("pv_syouka")));

                                    worksheet1.Cell(11, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 23), ColumnsConfigSet.nameToFormat("ph_ryuunyuu")));
                                    worksheet1.Cell(11, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 24), ColumnsConfigSet.nameToFormat("ph_ryuunyuu")));
                                    worksheet1.Cell(11, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 25), ColumnsConfigSet.nameToFormat("ph_ryuunyuu")));

                                    worksheet1.Cell(12, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 26), ColumnsConfigSet.nameToFormat("bod_ryuunyuu")));
                                    worksheet1.Cell(12, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 27), ColumnsConfigSet.nameToFormat("bod_ryuunyuu")));
                                    worksheet1.Cell(12, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 28), ColumnsConfigSet.nameToFormat("bod_ryuunyuu")));

                                    worksheet1.Cell(13, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 29), ColumnsConfigSet.nameToFormat("ss_ryuunyuu")));
                                    worksheet1.Cell(13, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 30), ColumnsConfigSet.nameToFormat("ss_ryuunyuu")));
                                    worksheet1.Cell(13, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 31), ColumnsConfigSet.nameToFormat("ss_ryuunyuu")));

                                    worksheet1.Cell(14, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 32), ColumnsConfigSet.nameToFormat("ph_enkondeguchi")));
                                    worksheet1.Cell(14, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 33), ColumnsConfigSet.nameToFormat("ph_enkondeguchi")));
                                    worksheet1.Cell(14, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 34), ColumnsConfigSet.nameToFormat("ph_enkondeguchi")));

                                    worksheet1.Cell(15, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 35), ColumnsConfigSet.nameToFormat("bod_enkondeguchi")));
                                    worksheet1.Cell(15, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 36), ColumnsConfigSet.nameToFormat("bod_enkondeguchi")));
                                    worksheet1.Cell(15, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 37), ColumnsConfigSet.nameToFormat("bod_enkondeguchi")));

                                    worksheet1.Cell(16, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 38), ColumnsConfigSet.nameToFormat("ss_enkondeguchi")));
                                    worksheet1.Cell(16, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 39), ColumnsConfigSet.nameToFormat("ss_enkondeguchi")));
                                    worksheet1.Cell(16, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 40), ColumnsConfigSet.nameToFormat("ss_enkondeguchi")));

                                    worksheet1.Cell(17, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 41), ColumnsConfigSet.nameToFormat("nd_enkondeguchi")));
                                    worksheet1.Cell(17, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 42), ColumnsConfigSet.nameToFormat("nd_enkondeguchi")));
                                    worksheet1.Cell(17, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 43), ColumnsConfigSet.nameToFormat("nd_enkondeguchi")));

                                    worksheet1.Cell(18, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 44), ColumnsConfigSet.nameToFormat("pr_enkondeguchi")));
                                    worksheet1.Cell(18, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 45), ColumnsConfigSet.nameToFormat("pr_enkondeguchi")));
                                    worksheet1.Cell(18, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 46), ColumnsConfigSet.nameToFormat("pr_enkondeguchi")));

                                    worksheet1.Cell(19, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 47), ColumnsConfigSet.nameToFormat("ph_houryuukou")));
                                    worksheet1.Cell(19, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 48), ColumnsConfigSet.nameToFormat("ph_houryuukou")));
                                    worksheet1.Cell(19, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 49), ColumnsConfigSet.nameToFormat("ph_houryuukou")));

                                    worksheet1.Cell(20, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 50), ColumnsConfigSet.nameToFormat("bod_houryuukou")));
                                    worksheet1.Cell(20, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 51), ColumnsConfigSet.nameToFormat("bod_houryuukou")));
                                    worksheet1.Cell(20, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 52), ColumnsConfigSet.nameToFormat("bod_houryuukou")));

                                    worksheet1.Cell(21, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 53), ColumnsConfigSet.nameToFormat("ss_houryuukou")));
                                    worksheet1.Cell(21, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 54), ColumnsConfigSet.nameToFormat("ss_houryuukou")));
                                    worksheet1.Cell(21, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 55), ColumnsConfigSet.nameToFormat("ss_houryuukou")));

                                    worksheet1.Cell(22, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 56), ColumnsConfigSet.nameToFormat("nd_houryuukou")));
                                    worksheet1.Cell(22, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 57), ColumnsConfigSet.nameToFormat("nd_houryuukou")));
                                    worksheet1.Cell(22, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 58), ColumnsConfigSet.nameToFormat("nd_houryuukou")));

                                    worksheet1.Cell(23, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 59), ColumnsConfigSet.nameToFormat("pr_houryuukou")));
                                    worksheet1.Cell(23, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 60), ColumnsConfigSet.nameToFormat("pr_houryuukou")));
                                    worksheet1.Cell(23, 17).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 61), ColumnsConfigSet.nameToFormat("pr_houryuukou")));

                                    worksheet1.Cell(32, 17).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 62));
                                    worksheet1.Cell(33, 17).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 63));
                                    worksheet1.Cell(34, 17).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 64));
                                    worksheet1.Cell(36, 17).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 65));
                                    worksheet1.Cell(43, 19).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 66));

                                    worksheet1.Cell(32, 16).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 67));
                                    worksheet1.Cell(33, 16).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 68));
                                    worksheet1.Cell(34, 16).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 69));
                                    worksheet1.Cell(36, 16).SetValue(FunctionForDatabase.getDecimalOrNull(reader, 70));

                                    worksheet1.Cell(43, 16).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 71), ColumnsConfigSet.nameToFormat("dt_dassuiki_all")));
                                }
                            }
                        }
                    }

                    //最新値
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "wv_genyuumizusyori, wv_genyuuodeidassui " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt DESC LIMIT 1";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(40, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("wv_genyuumizusyori")));
                                    worksheet1.Cell(42, 15).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wv_genyuuodeidassui")));
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //水処理運転月報シートの記入
            void makeSheet2(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("水処理運転月報", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 30).SetValue(cDtY);
                    worksheet1.Cell(1, 32).SetValue(cDtM);
                    worksheet1.Cell(1, 27).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(uryou), SUM(wv_ryuunyuu), " +
                                "SUM(CASE WHEN seiten='○' THEN wv_ryuunyuu ELSE NULL END), " +
                                "SUM(wv_syochin), SUM(wv_houryuu), " +
                                "SUM(wv_jia), SUM(ug_solid_jia), SUM(wv_sairiyou), SUM(wv_ido), SUM(wv_jousui), SUM(wv_kani) " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(42, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("uryou")));
                                    worksheet1.Cell(42, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));

                                    worksheet1.Cell(42, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));

                                    worksheet1.Cell(42, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("wv_syochin")));
                                    worksheet1.Cell(42, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("wv_houryuu")));

                                    worksheet1.Cell(42, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("wv_jia")));
                                    worksheet1.Cell(42, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ug_solid_jia")));

                                    worksheet1.Cell(42, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("wv_sairiyou")));
                                    worksheet1.Cell(42, 51).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("wv_ido")));
                                    worksheet1.Cell(42, 52).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("wv_jousui")));
                                    worksheet1.Cell(42, 53).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("wv_kani")));
                                }
                            }
                        }
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt,tenkou, kion, uryou, wv_ryuunyuu, CASE WHEN seiten='○' THEN wv_ryuunyuu ELSE NULL END, wv_syochin, wv_houryuu, " +
                                "h_syochin, ov_nama, " +
                                "av_hannoutank_all, ar_hannoutank_all, st_hannoutank_all, srt_hannoutank_all, " +
                                "h_syuuchin, ov_hensou, or_hensou, ov_yojou, " +
                                "min_enkon, wv_jia, mr_jia, ug_solid_jia," + 
                                "wv_sairiyou, wv_ido, wv_jousui, " +
                                "wv_kani " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 29).SetValue(FunctionForDatabase.getStringOrNull(reader, 1));
                                    worksheet1.Cell(R, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("kion")));
                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("uryou")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("wv_syochin")));
                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("wv_houryuu")));

                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("h_syochin")));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("ov_nama")));

                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("av_hannoutank_all")));
                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("ar_hannoutank_all")));
                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("st_hannoutank_all")));
                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("srt_hannoutank_all")));

                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("h_syuuchin")));
                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("ov_hensou")));
                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("or_hensou")));
                                    worksheet1.Cell(R, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("ov_yojou")));

                                    worksheet1.Cell(R, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("min_enkon")));
                                    worksheet1.Cell(R, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("wv_jia")));
                                    worksheet1.Cell(R, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("mr_jia")));
                                    worksheet1.Cell(R, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("ug_solid_jia")));

                                    worksheet1.Cell(R, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 22), ColumnsConfigSet.nameToFormat("wv_sairiyou")));
                                    worksheet1.Cell(R, 51).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 23), ColumnsConfigSet.nameToFormat("wv_ido")));
                                    worksheet1.Cell(R, 52).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 24), ColumnsConfigSet.nameToFormat("wv_jousui")));

                                    worksheet1.Cell(R, 53).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 25), ColumnsConfigSet.nameToFormat("wv_kani")));
                                }
                            }
                        }
                    }

                    //処理場ごとの計算条件
                    string calcKijun = "備考：";
                    calcKijun += "初沈（沈殿時間）基準水量=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 1) + "、";
                    calcKijun += "反応タンク（空気倍率・反応時間・SRT）基準水量=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 2) + "、";
                    calcKijun += "SRT基準水質=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 5) + "、";
                    calcKijun += "終沈（沈殿時間・返送汚泥率）基準水量=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 3) + "、";
                    calcKijun += "塩混（接触時間・次亜注入率）基準水量=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 4);

                    worksheet1.Cell(43, 1).SetValue(calcKijun);

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //水質試験月報1シートの記入
            void makeSheet3(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("水質試験月報1", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 27).SetValue(cDtY);
                    worksheet1.Cell(1, 29).SetValue(cDtM);
                    worksheet1.Cell(1, 24).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, wv_ryuunyuu, " +
                                "wt_ryuunyuu, pr_ryuunyuu, ph_ryuunyuu, cod_ryuunyuu, bod_ryuunyuu, ss_ryuunyuu, nd_ryuunyuu, gaikan_ryuunyuu, syuuki_ryuunyuu, tn_ryuunyuu, organicn_ryuunyuu, nh4n_ryuunyuu, no2n_ryuunyuu, no3n_ryuunyuu, tp_ryuunyuu, clion_ryuunyuu, iodineconsum_ryuunyuu, " +
                                "cod_jokyo_sougou, bod_jokyo_sougou, ss_jokyo_sougou " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 26).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));

                                    worksheet1.Cell(R, 27).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("wt_ryuunyuu")));
                                    worksheet1.Cell(R, 28).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("pr_ryuunyuu")));
                                    worksheet1.Cell(R, 29).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ph_ryuunyuu")));
                                    worksheet1.Cell(R, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("cod_ryuunyuu")));
                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("bod_ryuunyuu")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("ss_ryuunyuu")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("nd_ryuunyuu")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.getStringOrNull(reader, 9));
                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.getStringOrNull(reader, 10));
                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("tn_ryuunyuu")));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("organicn_ryuunyuu")));
                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("nh4n_ryuunyuu")));
                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("no2n_ryuunyuu")));
                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("no3n_ryuunyuu")));
                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("tp_ryuunyuu")));
                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("clion_ryuunyuu")));
                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("iodineconsum_ryuunyuu")));

                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("cod_jokyo_sougou")));
                                    worksheet1.Cell(R, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("bod_jokyo_sougou")));
                                    worksheet1.Cell(R, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("ss_jokyo_sougou")));
                                }
                            }
                        }
                    }

                    //処理場ごとの計算条件
                    string calcKijun = "備考：";
                    calcKijun += "総合除去率（COD・BOD・SS）基準水質=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 5);

                    worksheet1.Cell(42, 1).SetValue(calcKijun);

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //水質試験月報2シートの記入
            void makeSheet4(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("水質試験月報2", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 31).SetValue(cDtY);
                    worksheet1.Cell(1, 33).SetValue(cDtM);
                    worksheet1.Cell(1, 28).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, wv_ryuunyuu, " +
                                "pr_syo_ryuunyuu, ph_syo_ryuunyuu, cod_syo_ryuunyuu, bod_syo_ryuunyuu, ss_syo_ryuunyuu, gaikan_syo_ryuunyuu, syuuki_syo_ryuunyuu, " +
                                "wt_syo_etsuryuu, pr_syo_etsuryuu, ph_syo_etsuryuu, cod_syo_etsuryuu, bod_syo_etsuryuu, cbod_syo_etsuryuu, ss_syo_etsuryuu, nd_syo_etsuryuu, gaikan_syo_etsuryuu, syuuki_syo_etsuryuu, tn_syo_etsuryuu, organicn_syo_etsuryuu, nh4n_syo_etsuryuu, no2n_syo_etsuryuu, no3n_syo_etsuryuu, tp_syo_etsuryuu, orthop_syo_etsuryuu, ss_jokyo_syo_etsuryuu " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));

                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("pr_syo_ryuunyuu")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ph_syo_ryuunyuu")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("cod_syo_ryuunyuu")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("bod_syo_ryuunyuu")));
                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ss_syo_ryuunyuu")));
                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.getStringOrNull(reader, 7));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.getStringOrNull(reader, 8));

                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("wt_syo_etsuryuu")));
                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("pr_syo_etsuryuu")));
                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("ph_syo_etsuryuu")));
                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("cod_syo_etsuryuu")));
                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("bod_syo_etsuryuu")));
                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("cbod_syo_etsuryuu")));
                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("ss_syo_etsuryuu")));
                                    worksheet1.Cell(R, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("nd_syo_etsuryuu")));
                                    worksheet1.Cell(R, 46).SetValue(FunctionForDatabase.getStringOrNull(reader, 17));
                                    worksheet1.Cell(R, 47).SetValue(FunctionForDatabase.getStringOrNull(reader, 18));
                                    worksheet1.Cell(R, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("tn_syo_etsuryuu")));
                                    worksheet1.Cell(R, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("organicn_syo_etsuryuu")));
                                    worksheet1.Cell(R, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("nh4n_syo_etsuryuu")));
                                    worksheet1.Cell(R, 51).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 22), ColumnsConfigSet.nameToFormat("no2n_syo_etsuryuu")));
                                    worksheet1.Cell(R, 52).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 23), ColumnsConfigSet.nameToFormat("no3n_syo_etsuryuu")));
                                    worksheet1.Cell(R, 53).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 24), ColumnsConfigSet.nameToFormat("tp_syo_etsuryuu")));
                                    worksheet1.Cell(R, 54).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 25), ColumnsConfigSet.nameToFormat("orthop_syo_etsuryuu")));
                                    worksheet1.Cell(R, 55).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 26), ColumnsConfigSet.nameToFormat("ss_jokyo_syo_etsuryuu")));
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //水質試験月報3シートの記入
            void makeSheet5(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("水質試験月報3", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 29).SetValue(cDtY);
                    worksheet1.Cell(1, 31).SetValue(cDtM);
                    worksheet1.Cell(1, 26).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, wv_ryuunyuu, " +
                                "wt_syuu_etsuryuu, pr_syuu_etsuryuu, ph_syuu_etsuryuu, cod_syuu_etsuryuu, bod_syuu_etsuryuu, cbod_syuu_etsuryuu, ss_syuu_etsuryuu, nd_syuu_etsuryuu, gaikan_syuu_etsuryuu, syuuki_syuu_etsuryuu, jouhatsuzan_syuu_etsuryuu, kyounetsuzan_syuu_etsuryuu, kyounetsugen_syuu_etsuryuu, youkaisei_syuu_etsuryuu, tn_syuu_etsuryuu, organicn_syuu_etsuryuu, nh4n_syuu_etsuryuu, no2n_syuu_etsuryuu, no3n_syuu_etsuryuu, tp_syuu_etsuryuu, orthop_syuu_etsuryuu, clion_syuu_etsuryuu " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 28).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));

                                    worksheet1.Cell(R, 29).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("wt_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("pr_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ph_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("cod_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("bod_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("cbod_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ss_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("nd_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.getStringOrNull(reader, 10));
                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.getStringOrNull(reader, 11));
                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("jouhatsuzan_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("kyounetsuzan_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("kyounetsugen_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("youkaisei_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("tn_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("organicn_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("nh4n_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("no2n_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("no3n_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("tp_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 22), ColumnsConfigSet.nameToFormat("orthop_syuu_etsuryuu")));
                                    worksheet1.Cell(R, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 23), ColumnsConfigSet.nameToFormat("clion_syuu_etsuryuu")));
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //水質試験月報4シートの記入
            void makeSheet6(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("水質試験月報4", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 29).SetValue(cDtY);
                    worksheet1.Cell(1, 31).SetValue(cDtM);
                    worksheet1.Cell(1, 26).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, wv_ryuunyuu, " +
                                "pr_enkondeguchi, ph_enkondeguchi, cod_enkondeguchi, bod_enkondeguchi, cbod_enkondeguchi, ss_enkondeguchi, nd_enkondeguchi, ac_enkondeguchi, fc_enkondeguchi, gaikan_enkondeguchi, syuuki_enkondeguchi, " +
                                "pr_houryuukou, ph_houryuukou, cod_houryuukou, bod_houryuukou, cbod_houryuukou, ss_houryuukou, nd_houryuukou, ac_houryuukou, fc_houryuukou, gaikan_houryuukou, syuuki_houryuukou " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 28).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wv_ryuunyuu")));

                                    worksheet1.Cell(R, 29).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("pr_enkondeguchi")));
                                    worksheet1.Cell(R, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ph_enkondeguchi")));
                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("cod_enkondeguchi")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("bod_enkondeguchi")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("cbod_enkondeguchi")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("ss_enkondeguchi")));
                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("nd_enkondeguchi")));
                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("ac_enkondeguchi")));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("fc_enkondeguchi")));
                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.getStringOrNull(reader, 11));
                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.getStringOrNull(reader, 12));

                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("pr_houryuukou")));
                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("ph_houryuukou")));
                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("cod_houryuukou")));
                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("bod_houryuukou")));
                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("cbod_houryuukou")));
                                    worksheet1.Cell(R, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("ss_houryuukou")));
                                    worksheet1.Cell(R, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("nd_houryuukou")));
                                    worksheet1.Cell(R, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("ac_houryuukou")));
                                    worksheet1.Cell(R, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("fc_houryuukou")));
                                    worksheet1.Cell(R, 49).SetValue(FunctionForDatabase.getStringOrNull(reader, 22));
                                    worksheet1.Cell(R, 50).SetValue(FunctionForDatabase.getStringOrNull(reader, 23));
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //反応タンク試験月報1シートの記入
            void makeSheet7(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("反応タンク試験月報1", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 26).SetValue(cDtY);
                    worksheet1.Cell(1, 28).SetValue(cDtM);
                    worksheet1.Cell(1, 23).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, " +
                                "wt_hannoutank_all, ph_hannoutank_all, sv_hannoutank_all, mldo_p_hannoutank_all, mldo_a_hannoutank_all, mlss_hannoutank_all, mlvss_hannoutank_all, vsspss_hannoutank_all, svi_hannoutank_all, bodpmlss_hannoutank_all, " +
                                "ph_hensou, sv_hensou, do_hensou, rsss_hensou, rsvss_hensou, vsspss_hensou, svi_hensou, " +
                                "ts_nama, " +
                                "st_hannoutank_all, " +
                                "ar_hannoutank_all " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 25).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("wt_hannoutank_all")));
                                    worksheet1.Cell(R, 26).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("ph_hannoutank_all")));
                                    worksheet1.Cell(R, 27).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("sv_hannoutank_all")));
                                    worksheet1.Cell(R, 28).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("mldo_p_hannoutank_all")));
                                    worksheet1.Cell(R, 29).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("mldo_a_hannoutank_all")));
                                    worksheet1.Cell(R, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("mlss_hannoutank_all")));
                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("mlvss_hannoutank_all")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("vsspss_hannoutank_all")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("svi_hannoutank_all")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("bodpmlss_hannoutank_all")));

                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("ph_hensou")));
                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("sv_hensou")));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("do_hensou")));
                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("rsss_hensou")));
                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("rsvss_hensou")));
                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("vsspss_hensou")));
                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("svi_hensou")));

                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("ts_nama")));

                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("st_hannoutank_all")));

                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("ar_hannoutank_all")));
                                }
                            }
                        }
                    }
                    
                    //処理場ごとの計算条件
                    string calcKijun = "備考：";
                    calcKijun += "反応タンク（BOD/MLSS負荷・滞留時間・空気倍率）基準水量=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 2) + "、反応タンク（BOD/MLSS負荷）基準水質=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 6);

                    worksheet1.Cell(42, 1).SetValue(calcKijun);

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //反応タンク試験月報2シートの記入
            void makeSheet8(XLWorkbook workbook1) {
                //名称が存在する最大のタンク番号の取得
                int maxTankNo = 0;

                using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                    con.Open();

                    using(var cmd = con.CreateCommand()) {
                        cmd.CommandText = "SELECT " +
                            "MAX(";


                        for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                            if(i > 1) {
                                cmd.CommandText += " + ";
                            }

                            cmd.CommandText += "(CASE WHEN COUNT(name_hannoutank_" + i.ToString() + ")>0 THEN " + i.ToString() + " ELSE 0 END)";

                            if(i < MaxSizeConfig.maxHannoutank) {
                                cmd.CommandText += ", ";
                            }
                        }

                        cmd.CommandText += ") " +
                            "FROM suishitsu_kanri " +
                            "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "'";

                        using(var reader = cmd.ExecuteReader()) {
                            if(reader.Read()) {
                                maxTankNo = reader.GetInt32(0);
                            }
                        }
                    }
                }

                //ワークシートのコピー
                int maxSheetNo = (int)(Math.Floor((decimal)((maxTankNo - 1) / 2)) + 1);

                IXLWorksheet worksheet0;

                if(!workbook1.TryGetWorksheet("反応タンク試験月報2_1", out worksheet0)) {
                    throw new Exception();
                }

                for(int i = 1; i <= maxSheetNo - 1; i++) {
                    worksheet0.CopyTo("反応タンク試験月報2_" + (i + 1).ToString(), worksheet0.Position + i);
                }

                //各シートへの記入
                int Coffset = 13;

                for(int i = 1; i <= maxSheetNo; i++) {
                    IXLWorksheet worksheet1;

                    if(workbook1.TryGetWorksheet("反応タンク試験月報2_" + i.ToString(), out worksheet1)) {
                        //年月、管理者・処理場
                        worksheet1.Cell(1, 32).SetValue(cDtY);
                        worksheet1.Cell(1, 34).SetValue(cDtM);
                        worksheet1.Cell(1, 36).SetValue(i);
                        worksheet1.Cell(1, 29).SetValue(kanrisya + "　" + syorijou);

                        //月にない日を消去
                        if(monthLastDay < 31) {
                            worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                            worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                        }

                        if(monthLastDay < 30) {
                            worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                            worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                        }

                        if(monthLastDay < 29) {
                            worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                            worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                        }

                        for(int j = 0; j < 2; j++) {
                            int tN = (i - 1) * 2 + j + 1;
                            int Co = Coffset * j;

                            if(tN > maxTankNo) {
                                worksheet1.Range(3, 3 + Co, 6, 15 + Co).SetValue("");
                                worksheet1.Range(3, 31 + Co, 6, 43 + Co).SetValue("");
                                break;
                            }

                            //名称は最新値を採用
                            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                                con.Open();

                                using(var cmd = con.CreateCommand()) {
                                    cmd.CommandText = "SELECT " +
                                        "name_hannoutank_" + tN + " " +
                                        "FROM suishitsu_kanri " +
                                        "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                        "AND name_hannoutank_" + tN + " IS NOT NULL AND name_hannoutank_" + tN + "<>''" +
                                        "ORDER BY sokutei_dt DESC LIMIT 1";

                                    using(var reader = cmd.ExecuteReader()) {
                                        if(reader.Read()) {
                                            worksheet1.Cell(2, 31 + Co).SetValue("反応タンク" + tN.ToString() + "（" + FunctionForDatabase.getStringOrNull(reader, 0) + "）");
                                        }
                                    }
                                }
                            }

                            //月集計
                            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                                con.Open();

                                using(var cmd = con.CreateCommand()) {
                                    cmd.CommandText = "SELECT " +
                                        "sokutei_dt, " +
                                        "ph_hannoutank_" + tN + ", sv_hannoutank_" + tN + ", mldo_p_hannoutank_" + tN + ", mldo_a_hannoutank_" + tN + ", mlss_hannoutank_" + tN + ", mlvss_hannoutank_" + tN + ", vsspss_hannoutank_" + tN + ", svi_hannoutank_" + tN + ", orp_ken_hannoutank_" + tN + ", orp_kou_hannoutank_" + tN + ", rr_zen_hannoutank_" + tN + ", rr_kou_hannoutank_" + tN + ", bodpmlss_hannoutank_" + tN + " " +
                                        "FROM suishitsu_kanri " +
                                        "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                        "ORDER BY sokutei_dt";

                                    using(var reader = cmd.ExecuteReader()) {
                                        int d = 0;

                                        while(reader.Read()) {
                                            while(reader.GetDateTime(0).Day != d) {
                                                d++;
                                            }

                                            int R = d + 6;

                                            worksheet1.Cell(R, 31 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ph_hannoutank_1")));
                                            worksheet1.Cell(R, 32 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("sv_hannoutank_1")));
                                            worksheet1.Cell(R, 33 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("mldo_p_hannoutank_1")));
                                            worksheet1.Cell(R, 34 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("mldo_a_hannoutank_1")));
                                            worksheet1.Cell(R, 35 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("mlss_hannoutank_1")));
                                            worksheet1.Cell(R, 36 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("mlvss_hannoutank_1")));
                                            worksheet1.Cell(R, 37 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("vsspss_hannoutank_1")));
                                            worksheet1.Cell(R, 38 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("svi_hannoutank_")));
                                            worksheet1.Cell(R, 39 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("orp_ken_hannoutank_1")));
                                            worksheet1.Cell(R, 40 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("orp_kou_hannoutank_1")));
                                            worksheet1.Cell(R, 41 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("rr_zen_hannoutank_1")));
                                            worksheet1.Cell(R, 42 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("rr_kou_hannoutank_1")));
                                            worksheet1.Cell(R, 43 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("bodpmlss_hannoutank_1")));
                                        }
                                    }
                                }
                            }
                        }

                        //処理場ごとの計算条件
                        string calcKijun = "備考：";
                        calcKijun += "反応タンク（BOD/MLSS負荷）基準水量=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 2) + "、反応タンク（BOD/MLSS負荷）基準水質=" + SyorijouConfigSet.getConfigString(kanrisya, syorijou, 6);

                        worksheet1.Cell(42, 1).SetValue(calcKijun);

                        worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                    }
                    else {
                        throw new Exception();
                    }
                }

                return;
            }




            //消化槽月報シートの記入
            void makeSheet9(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("消化槽月報", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 32).SetValue(cDtY);
                    worksheet1.Cell(1, 34).SetValue(cDtM);
                    worksheet1.Cell(1, 29).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //名称は最新値を採用
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        for(int i = 1; i <= 4; i++) {
                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_syoukatank_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_syoukatank_" + i + " IS NOT NULL AND name_syoukatank_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(3, 38 + i).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(3, 10 + i, 3, 10 + i).SetValue("");
                                        worksheet1.Range(3, 38 + i, 3, 38 + i).SetValue("");
                                    }
                                }
                            }
                        }
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(ov_syouka_input), " +
                                "SUM(ov_syouka_output), " +
                                "SUM(v_syoukag_hassei), SUM(v_syoukag_soukaon), SUM(v_syoukag_yogas), SUM(v_syoukag_baiden), SUM(m_datsuryuu), " +
                                "SUM(v_hao_syouka), SUM(v_lo_syouka), SUM(v_kerosene_syouka), SUM(v_gasoline_syouka), SUM(v_pgas_syouka) " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(42, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("ov_syouka_input")));

                                    worksheet1.Cell(42, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ov_syouka_output")));

                                    worksheet1.Cell(42, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("v_syoukag_hassei")));
                                    worksheet1.Cell(42, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("v_syoukag_soukaon")));
                                    worksheet1.Cell(42, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("v_syoukag_yogas")));
                                    worksheet1.Cell(42, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("v_syoukag_baiden")));
                                    worksheet1.Cell(42, 52).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("m_datsuryuu")));

                                    worksheet1.Cell(42, 53).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("v_hao_syouka")));
                                    worksheet1.Cell(42, 54).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("v_lo_syouka")));
                                    worksheet1.Cell(42, 55).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("v_kerosene_syouka")));
                                    worksheet1.Cell(42, 56).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("v_gasoline_syouka")));
                                    worksheet1.Cell(42, 57).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("v_pgas_syouka")));
                                }
                            }
                        }
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, " +
                                "ov_syouka_input, oc_syouka_input, oy_syouka_input, ph_syouka_input, " +
                                "ov_syouka_output, oc_syouka_output, oy_syouka_output, ph_syouka_output, " +
                                "tt_syoukatank_1, tt_syoukatank_2, tt_syoukatank_3, tt_syoukatank_4, " +
                                "r_syouka, " +
                                "d_syouka, " +
                                "v_syoukag_hassei, v_syoukag_soukaon, v_syoukag_yogas, v_syoukag_baiden, gr_syoukagas, r_syoukagas_ch3, r_syoukagas_co2, m_datsuryuu, " +
                                "v_hao_syouka, v_lo_syouka, v_kerosene_syouka, v_gasoline_syouka, v_pgas_syouka " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ov_syouka_input")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("oc_syouka_input")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("oy_syouka_input")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ph_syouka_input")));

                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("ov_syouka_output")));
                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("oc_syouka_output")));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("oy_syouka_output")));
                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ph_syouka_output")));

                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("tt_syoukatank_1")));
                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("tt_syoukatank_2")));
                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("tt_syoukatank_3")));
                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("tt_syoukatank_4")));

                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("r_syouka")));

                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("d_syouka")));

                                    worksheet1.Cell(R, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("v_syoukag_hassei")));
                                    worksheet1.Cell(R, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("v_syoukag_soukaon")));
                                    worksheet1.Cell(R, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("v_syoukag_yogas")));
                                    worksheet1.Cell(R, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("v_syoukag_baiden")));
                                    worksheet1.Cell(R, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("gr_syoukagas")));
                                    worksheet1.Cell(R, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("r_syoukagas_ch3")));
                                    worksheet1.Cell(R, 51).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("r_syoukagas_co2")));
                                    worksheet1.Cell(R, 52).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 22), ColumnsConfigSet.nameToFormat("m_datsuryuu")));

                                    worksheet1.Cell(R, 53).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 23), ColumnsConfigSet.nameToFormat("v_hao_syouka")));
                                    worksheet1.Cell(R, 54).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 24), ColumnsConfigSet.nameToFormat("v_lo_syouka")));
                                    worksheet1.Cell(R, 55).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 25), ColumnsConfigSet.nameToFormat("v_kerosene_syouka")));
                                    worksheet1.Cell(R, 56).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 26), ColumnsConfigSet.nameToFormat("v_gasoline_syouka")));
                                    worksheet1.Cell(R, 57).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 27), ColumnsConfigSet.nameToFormat("v_pgas_syouka")));
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //汚泥処理月報1シートの記入
            void makeSheet10(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("汚泥処理月報1", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 31).SetValue(cDtY);
                    worksheet1.Cell(1, 33).SetValue(cDtM);
                    worksheet1.Cell(1, 28).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(ov_dassuiki_input_enshin), SUM(ov_dassuiki_input_bp), SUM(ov_dassuiki_input_ex), SUM(ov_dassuiki_input_all), SUM(sg_dassuiki_input_enshin), SUM(sg_dassuiki_input_bp), SUM(sg_dassuiki_input_ex), SUM(sg_dassuiki_input_all), " +
                                "SUM(ov_dassui_output_enshin), SUM(ov_dassui_output_bp), SUM(ov_dassui_output_ex), SUM(ov_dassui_output_all), SUM(sg_dassui_output_enshin), SUM(sg_dassui_output_bp), SUM(sg_dassui_output_ex), SUM(sg_dassui_output_all) " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(42, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_enshin")));
                                    worksheet1.Cell(42, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_bp")));
                                    worksheet1.Cell(42, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_ex")));
                                    worksheet1.Cell(42, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_all")));
                                    worksheet1.Cell(42, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_enshin")));
                                    worksheet1.Cell(42, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_bp")));
                                    worksheet1.Cell(42, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_ex")));
                                    worksheet1.Cell(42, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_all")));

                                    worksheet1.Cell(42, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ov_dassui_output_enshin")));
                                    worksheet1.Cell(42, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("ov_dassui_output_bp")));
                                    worksheet1.Cell(42, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("ov_dassui_output_ex")));
                                    worksheet1.Cell(42, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("ov_dassui_output_all")));
                                    worksheet1.Cell(42, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("sg_dassui_output_enshin")));
                                    worksheet1.Cell(42, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("sg_dassui_output_bp")));
                                    worksheet1.Cell(42, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("sg_dassui_output_ex")));
                                    worksheet1.Cell(42, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("sg_dassui_output_all")));
                                }
                            }
                        }
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, " +
                                "ov_dassuiki_input_enshin, ov_dassuiki_input_bp, ov_dassuiki_input_ex, ov_dassuiki_input_all, sg_dassuiki_input_enshin, sg_dassuiki_input_bp, sg_dassuiki_input_ex, sg_dassuiki_input_all, " +
                                "ov_dassui_output_enshin, ov_dassui_output_bp, ov_dassui_output_ex, ov_dassui_output_all, owr_dassui_output_enshin, owr_dassui_output_bp, owr_dassui_output_ex, sg_dassui_output_enshin, sg_dassui_output_bp, sg_dassui_output_ex, sg_dassui_output_all, sgcr_dassui_output_enshin, sgcr_dassui_output_bp, sgcr_dassui_output_ex, sgcr_dassui_output_all, oy_dassui_output_enshin, oy_dassui_output_bp, oy_dassui_output_ex " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_enshin")));
                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_bp")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_ex")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ov_dassuiki_input_all")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_enshin")));
                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_bp")));
                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_ex")));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_all")));

                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("ov_dassui_output_enshin")));
                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("ov_dassui_output_bp")));
                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("ov_dassui_output_ex")));
                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("ov_dassui_output_all")));
                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("owr_dassui_output_enshin")));
                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("owr_dassui_output_bp")));
                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("owr_dassui_output_ex")));
                                    worksheet1.Cell(R, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("sg_dassui_output_enshin")));
                                    worksheet1.Cell(R, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("sg_dassui_output_bp")));
                                    worksheet1.Cell(R, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("sg_dassui_output_ex")));
                                    worksheet1.Cell(R, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("sg_dassui_output_all")));
                                    worksheet1.Cell(R, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("sgcr_dassui_output_enshin")));
                                    worksheet1.Cell(R, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("sgcr_dassui_output_bp")));
                                    worksheet1.Cell(R, 51).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 22), ColumnsConfigSet.nameToFormat("sgcr_dassui_output_ex")));
                                    worksheet1.Cell(R, 52).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 23), ColumnsConfigSet.nameToFormat("sgcr_dassui_output_all")));
                                    worksheet1.Cell(R, 53).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 24), ColumnsConfigSet.nameToFormat("oy_dassui_output_enshin")));
                                    worksheet1.Cell(R, 54).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 25), ColumnsConfigSet.nameToFormat("oy_dassui_output_bp")));
                                    worksheet1.Cell(R, 55).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 26), ColumnsConfigSet.nameToFormat("oy_dassui_output_ex")));
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //汚泥処理月報2シートの記入
            void makeSheet11(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("汚泥処理月報2", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 33).SetValue(cDtY);
                    worksheet1.Cell(1, 35).SetValue(cDtM);
                    worksheet1.Cell(1, 30).SetValue(kanrisya + "　" + syorijou);

                    //他薬品1の名称(最新値)
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            string excName = "";

                            for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                                cmd.CommandText = "SELECT type_exc1_dassuiki_" + i.ToString() + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE type_exc1_dassuiki_" + i.ToString() + " IS NOT NULL AND type_exc1_dassuiki_" + i.ToString() + " IS NOT '' " +
                                    "AND del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        excName = reader.GetString(0);
                                        break;
                                    }
                                }
                            }

                            worksheet1.Cell(2, 40).SetValue("他薬品1（" + excName + "）");
                        }
                    }

                    //他薬品2の名称(最新値)
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            string excName = "";

                            for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                                cmd.CommandText = "SELECT type_exc2_dassuiki_" + i.ToString() + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE type_exc2_dassuiki_" + i.ToString() + " IS NOT NULL AND type_exc2_dassuiki_" + i.ToString() + " IS NOT '' " +
                                    "AND del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        excName = reader.GetString(0);
                                        break;
                                    }
                                }
                            }

                            worksheet1.Cell(2, 48).SetValue("他薬品2（" + excName + "）");
                        }
                    }

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(ug_gyousyuuzai_dassuiki_enshin), SUM(ug_gyousyuuzai_dassuiki_bp), SUM(ug_gyousyuuzai_dassuiki_ex), SUM(ug_gyousyuuzai_dassuiki_all), " +
                                "SUM(ug_exc1_dassuiki_enshin), SUM(ug_exc1_dassuiki_bp), SUM(ug_exc1_dassuiki_ex), SUM(ug_exc1_dassuiki_all), " +
                                "SUM(ug_exc2_dassuiki_enshin), SUM(ug_exc2_dassuiki_bp), SUM(ug_exc2_dassuiki_ex), SUM(ug_exc2_dassuiki_all), " +
                                "SUM(dt_dassuiki_enshin), SUM(dt_dassuiki_bp), SUM(dt_dassuiki_ex), SUM(dt_dassuiki_all) " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(42, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_enshin")));
                                    worksheet1.Cell(42, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_bp")));
                                    worksheet1.Cell(42, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_ex")));
                                    worksheet1.Cell(42, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_all")));

                                    worksheet1.Cell(42, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_enshin")));
                                    worksheet1.Cell(42, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_bp")));
                                    worksheet1.Cell(42, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_ex")));
                                    worksheet1.Cell(42, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_all")));

                                    worksheet1.Cell(42, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_enshin")));
                                    worksheet1.Cell(42, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_bp")));
                                    worksheet1.Cell(42, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_ex")));
                                    worksheet1.Cell(42, 51).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_all")));

                                    worksheet1.Cell(42, 56).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("dt_dassuiki_enshin")));
                                    worksheet1.Cell(42, 57).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("dt_dassuiki_bp")));
                                    worksheet1.Cell(42, 58).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("dt_dassuiki_ex")));
                                    worksheet1.Cell(42, 59).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("dt_dassuiki_all")));
                                }
                            }
                        }
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, " +
                                "ug_gyousyuuzai_dassuiki_enshin, ug_gyousyuuzai_dassuiki_bp, ug_gyousyuuzai_dassuiki_ex, ug_gyousyuuzai_dassuiki_all, " +
                                "ug_exc1_dassuiki_enshin, ug_exc1_dassuiki_bp, ug_exc1_dassuiki_ex, ug_exc1_dassuiki_all, " +
                                "ug_exc2_dassuiki_enshin, ug_exc2_dassuiki_bp, ug_exc2_dassuiki_ex, ug_exc2_dassuiki_all, " +
                                "sg_dassuiki_input_enshin, sg_dassuiki_input_bp, sg_dassuiki_input_ex, sg_dassuiki_input_all, " +
                                "dt_dassuiki_enshin, dt_dassuiki_bp, dt_dassuiki_ex, dt_dassuiki_all " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    decimal? ugGyEn = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_enshin"));
                                    decimal? ugGyBp = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_bp"));
                                    decimal? ugGyEx = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_ex"));
                                    decimal? ugGyAl = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_all"));

                                    decimal? ugE1En = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_enshin"));
                                    decimal? ugE1Bp = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_bp"));
                                    decimal? ugE1Ex = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_ex"));
                                    decimal? ugE1Al = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_all"));

                                    decimal? ugE2En = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_enshin"));
                                    decimal? ugE2Bp = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_bp"));
                                    decimal? ugE2Ex = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_ex"));
                                    decimal? ugE2Al = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_all"));

                                    decimal? sgEn = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_enshin"));
                                    decimal? sgBp = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_bp"));
                                    decimal? sgEx = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_ex"));
                                    decimal? sgAl = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("sg_dassuiki_input_all"));

                                    worksheet1.Cell(R, 32).SetValue(ugGyEn);
                                    worksheet1.Cell(R, 33).SetValue(ugGyBp);
                                    worksheet1.Cell(R, 34).SetValue(ugGyEx);
                                    worksheet1.Cell(R, 35).SetValue(ugGyAl);

                                    if(ugGyEn != null && sgEn != null && sgEn > 0) {
                                        worksheet1.Cell(R, 36).SetValue(ugGyEn / sgEn * 100);
                                    }

                                    if(ugGyBp != null && sgBp != null && sgBp > 0) {
                                        worksheet1.Cell(R, 37).SetValue(ugGyBp / sgBp * 100);
                                    }

                                    if(ugGyEx != null && sgEx != null && sgEx > 0) {
                                        worksheet1.Cell(R, 38).SetValue(ugGyEx / sgEx * 100);
                                    }

                                    if(ugGyAl != null && sgAl != null && sgAl > 0) {
                                        worksheet1.Cell(R, 39).SetValue(ugGyAl / sgAl * 100);
                                    }

                                    worksheet1.Cell(R, 40).SetValue(ugE1En);
                                    worksheet1.Cell(R, 41).SetValue(ugE1Bp);
                                    worksheet1.Cell(R, 42).SetValue(ugE1Ex);
                                    worksheet1.Cell(R, 43).SetValue(ugE1Al);

                                    if(ugE1En != null && sgEn != null && sgEn > 0) {
                                        worksheet1.Cell(R, 44).SetValue(ugE1En / sgEn * 100);
                                    }

                                    if(ugE1Bp != null && sgBp != null && sgBp > 0) {
                                        worksheet1.Cell(R, 45).SetValue(ugE1Bp / sgBp * 100);
                                    }

                                    if(ugE1Ex != null && sgEx != null && sgEx > 0) {
                                        worksheet1.Cell(R, 46).SetValue(ugE1Ex / sgEx * 100);
                                    }

                                    if(ugE1Al != null && sgAl != null && sgAl > 0) {
                                        worksheet1.Cell(R, 47).SetValue(ugE1Al / sgAl * 100);
                                    }

                                    worksheet1.Cell(R, 48).SetValue(ugE2En);
                                    worksheet1.Cell(R, 49).SetValue(ugE2Bp);
                                    worksheet1.Cell(R, 50).SetValue(ugE2Ex);
                                    worksheet1.Cell(R, 51).SetValue(ugE2Al);

                                    if(ugE2En != null && sgEn != null && sgEn > 0) {
                                        worksheet1.Cell(R, 52).SetValue(ugE2En / sgEn * 100);
                                    }

                                    if(ugE2Bp != null && sgBp != null && sgBp > 0) {
                                        worksheet1.Cell(R, 53).SetValue(ugE2Bp / sgBp * 100);
                                    }

                                    if(ugE2Ex != null && sgEx != null && sgEx > 0) {
                                        worksheet1.Cell(R, 54).SetValue(ugE2Ex / sgEx * 100);
                                    }

                                    if(ugE2Al != null && sgAl != null && sgAl > 0) {
                                        worksheet1.Cell(R, 55).SetValue(ugE2Al / sgAl * 100);
                                    }

                                    worksheet1.Cell(R, 56).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("dt_dassuiki_enshin")));
                                    worksheet1.Cell(R, 57).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("dt_dassuiki_bp")));
                                    worksheet1.Cell(R, 58).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("dt_dassuiki_ex")));
                                    worksheet1.Cell(R, 59).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("dt_dassuiki_all")));
                                }
                            }
                        }
                    }

                    //備考
                    string calcKijun = "注1 他薬品1・他薬品2の集計は、設備ごとの使用薬品に違いがあっても単純に加算している　注2 他薬品1・他薬品2の名称は、脱水設備1の月内で最新のもの→脱水設備2…の順で採用";

                    worksheet1.Cell(43, 1).SetValue(calcKijun);

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //汚泥処理月報3シートの記入
            void makeSheet12(XLWorkbook workbook1) {
                //名称が存在する最大の脱水設備番号の取得
                int maxDassuiNo = 0;

                using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                    con.Open();

                    using(var cmd = con.CreateCommand()) {
                        cmd.CommandText = "SELECT " +
                            "MAX(";


                        for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                            if(i > 1) {
                                cmd.CommandText += " + ";
                            }

                            cmd.CommandText += "(CASE WHEN COUNT(name_dassuiki_" + i.ToString() + ")>0 THEN " + i.ToString() + " ELSE 0 END)";

                            if(i < MaxSizeConfig.maxDassuiki) {
                                cmd.CommandText += ", ";
                            }
                        }

                        cmd.CommandText += ") " +
                            "FROM suishitsu_kanri " +
                            "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "'";

                        using(var reader = cmd.ExecuteReader()) {
                            if(reader.Read()) {
                                maxDassuiNo = reader.GetInt32(0);
                            }
                        }
                    }
                }

                //ワークシートのコピー
                int maxSheetNo = (int)(Math.Floor((decimal)((maxDassuiNo - 1) / 2)) + 1);

                IXLWorksheet worksheet0;

                if(!workbook1.TryGetWorksheet("汚泥処理月報3_1", out worksheet0)) {
                    throw new Exception();
                }

                for(int i = 1; i <= maxSheetNo - 1; i++) {
                    worksheet0.CopyTo("汚泥処理月報3_" + (i + 1).ToString(), worksheet0.Position + i);
                }

                //各シートへの記入
                int Coffset = 14;

                for(int i = 1; i <= maxSheetNo; i++) {
                    IXLWorksheet worksheet1;

                    if(workbook1.TryGetWorksheet("汚泥処理月報3_" + i.ToString(), out worksheet1)) {
                        //年月、管理者・処理場
                        worksheet1.Cell(1, 33).SetValue(cDtY);
                        worksheet1.Cell(1, 35).SetValue(cDtM);
                        worksheet1.Cell(1, 37).SetValue(i);
                        worksheet1.Cell(1, 30).SetValue(kanrisya + "　" + syorijou);

                        //月にない日を消去
                        if(monthLastDay < 31) {
                            worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                            worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                        }

                        if(monthLastDay < 30) {
                            worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                            worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                        }

                        if(monthLastDay < 29) {
                            worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                            worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                        }

                        for(int j = 0; j < 2; j++) {
                            int dN = (i - 1) * 2 + j + 1;
                            int Co = Coffset * j;

                            if(dN > maxDassuiNo) {
                                worksheet1.Range(3, 3 + Co, 6, 16 + Co).SetValue("");
                                worksheet1.Range(3, 32 + Co, 6, 45 + Co).SetValue("");
                                break;
                            }

                            //他薬品1の名称(最新値)
                            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                                con.Open();

                                using(var cmd = con.CreateCommand()) {
                                    cmd.CommandText = "SELECT type_exc1_dassuiki_" + dN + " " +
                                        "FROM suishitsu_kanri " +
                                        "WHERE type_exc1_dassuiki_" + dN + " IS NOT NULL AND type_exc1_dassuiki_" + dN + " IS NOT '' " +
                                        "AND del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                        "ORDER BY sokutei_dt DESC LIMIT 1";

                                    using(var reader = cmd.ExecuteReader()) {
                                        if(reader.Read()) {
                                            worksheet1.Cell(43, 39 + Co).SetValue(reader.GetString(0));
                                        }
                                    }
                                }
                            }

                            //他薬品2の名称(最新値)
                            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                                con.Open();

                                using(var cmd = con.CreateCommand()) {
                                    cmd.CommandText = "SELECT type_exc2_dassuiki_" + dN + " " +
                                        "FROM suishitsu_kanri " +
                                        "WHERE type_exc2_dassuiki_" + dN + " IS NOT NULL AND type_exc2_dassuiki_" + dN + " IS NOT '' " +
                                        "AND del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                        "ORDER BY sokutei_dt DESC LIMIT 1";

                                    using(var reader = cmd.ExecuteReader()) {
                                        if(reader.Read()) {
                                            worksheet1.Cell(43, 41 + Co).SetValue(reader.GetString(0));
                                        }
                                    }
                                }
                            }

                            //名称、形式は最新値を採用
                            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                                con.Open();

                                using(var cmd = con.CreateCommand()) {
                                    cmd.CommandText = "SELECT " +
                                        "name_dassuiki_" + dN + ", type_dassuiki_" + dN + " " +
                                        "FROM suishitsu_kanri " +
                                        "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                        "AND name_dassuiki_" + dN + " IS NOT NULL AND name_dassuiki_" + dN + "<>''" +
                                        "ORDER BY sokutei_dt DESC LIMIT 1";

                                    using(var reader = cmd.ExecuteReader()) {
                                        if(reader.Read()) {
                                            worksheet1.Cell(2, 32 + Co).SetValue("脱水設備" + dN.ToString() + "　" + FunctionForDatabase.getStringOrNull(reader, 1) + "(" + FunctionForDatabase.getStringOrNull(reader, 0) + ")");

                                            if(FunctionForDatabase.getStringOrNull(reader, 1) != "ベルトプレス") {
                                                worksheet1.Range(3, 15 + Co, 6, 16 + Co).SetValue("");
                                                worksheet1.Range(3, 44 + Co, 6, 45 + Co).SetValue("");
                                            }
                                        }
                                    }
                                }
                            }

                            //年度集計
                            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                                con.Open();

                                using(var cmd = con.CreateCommand()) {
                                    cmd.CommandText = "SELECT " +
                                        "SUM(ov_dassuiki_" + dN + "), SUM(sg_dassuiki_" + dN + "), SUM(v_gyousyuuzai_dassuiki_" + dN + "), SUM(ug_gyousyuuzai_dassuiki_" + dN + "), SUM(ug_exc1_dassuiki_" + dN + "), SUM(ug_exc2_dassuiki_" + dN + "), SUM(dt_dassuiki_" + dN + ") " +
                                        "FROM suishitsu_kanri " +
                                        "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                                    using(var reader = cmd.ExecuteReader()) {
                                        if(reader.Read()) {
                                            worksheet1.Cell(42, 32 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("ov_dassuiki_1")));
                                            worksheet1.Cell(42, 34 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("sg_dassuiki_1")));
                                            worksheet1.Cell(42, 35 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("v_gyousyuuzai_dassuiki_1")));
                                            worksheet1.Cell(42, 37 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_1")));
                                            worksheet1.Cell(42, 39 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_1")));
                                            worksheet1.Cell(42, 41 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_1")));
                                            worksheet1.Cell(42, 43 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("dt_dassuiki_1")));
                                        }
                                    }
                                }
                            }

                            //月集計
                            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                                con.Open();

                                using(var cmd = con.CreateCommand()) {
                                    cmd.CommandText = "SELECT " +
                                        "sokutei_dt, " +
                                        "ov_dassuiki_" + dN + ", oci_dassuiki_" + dN + ", sg_dassuiki_" + dN + ", v_gyousyuuzai_dassuiki_" + dN + ", oc_gyousyuuzai_dassuiki_" + dN + ", ug_gyousyuuzai_dassuiki_" + dN + ", r_gyousyuuzai_dassuiki_" + dN + ", ug_exc1_dassuiki_" + dN + ", r_exc1_dassuiki_" + dN + ", ug_exc2_dassuiki_" + dN + ", r_exc2_dassuiki_" + dN + ", dt_dassuiki_" + dN + ", rofusp_dassuiki_" + dN + ", rokasp_dassuiki_" + dN + " " +
                                        "FROM suishitsu_kanri " +
                                        "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                        "ORDER BY sokutei_dt";

                                    using(var reader = cmd.ExecuteReader()) {
                                        int d = 0;

                                        while(reader.Read()) {
                                            while(reader.GetDateTime(0).Day != d) {
                                                d++;
                                            }

                                            int R = d + 6;

                                            worksheet1.Cell(R, 32 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ov_dassuiki_1")));
                                            worksheet1.Cell(R, 33 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("oci_dassuiki_1")));
                                            worksheet1.Cell(R, 34 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("sg_dassuiki_1")));
                                            worksheet1.Cell(R, 35 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("v_gyousyuuzai_dassuiki_1")));
                                            worksheet1.Cell(R, 36 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("oc_gyousyuuzai_dassuiki_1")));
                                            worksheet1.Cell(R, 37 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_dassuiki_1")));
                                            worksheet1.Cell(R, 38 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("r_gyousyuuzai_dassuiki_1")));
                                            worksheet1.Cell(R, 39 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ug_exc1_dassuiki_1")));
                                            worksheet1.Cell(R, 40 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("r_exc1_dassuiki_1")));
                                            worksheet1.Cell(R, 41 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("ug_exc2_dassuiki_1")));
                                            worksheet1.Cell(R, 42 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("r_exc2_dassuiki_1")));
                                            worksheet1.Cell(R, 43 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("dt_dassuiki_1")));
                                            worksheet1.Cell(R, 44 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("rofusp_dassuiki_1")));
                                            worksheet1.Cell(R, 45 + Co).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("rokasp_dassuiki_1")));
                                        }
                                    }
                                }
                            }

                            //備考
                            string calcKijun = "注1 他薬品1・他薬品2の名称は、月内で最新のものを採用";

                            worksheet1.Cell(43, 1).SetValue(calcKijun);

                            worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                        }
                    }
                    else {
                        throw new Exception();
                    }
                }

                return;
            }




            //汚泥処理月報4シートの記入
            void makeSheet13(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("汚泥処理月報4", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 32).SetValue(cDtY);
                    worksheet1.Cell(1, 34).SetValue(cDtM);
                    worksheet1.Cell(1, 29).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(ov_hansyutsu_chinsa_pumpjou_all), SUM(ov_hansyutsu_shisa_pumpjou_all), " +
                                "SUM(ov_hansyutsu_chinsa), SUM(ov_hansyutsu_shisa), " +
                                "SUM(ov_nama), SUM(sg_nama), " +
                                "SUM(ov_yojou), SUM(sg_yojou), " +
                                "SUM(ov_nousyuku_output_juuryoku), SUM(sg_nousyuku_output_juuryoku), " +
                                "SUM(ov_nousyuku_output_ex), SUM(sg_nousyuku_output_ex), " +
                                "SUM(ov_syouka_output), SUM(sg_syouka_output), ";

                            string q1 = "";
                            string q2 = "";

                            for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                                if(i > 1) {
                                    q1 += " AND ";
                                    q2 += " + ";
                                }

                                q1 += "v_gyousyuuzai_nousyukuki_" + i.ToString() + " IS NULL";
                                q2 += "IFNULL(v_gyousyuuzai_nousyukuki_" + i.ToString() + ", 0)";
                            }

                            cmd.CommandText += "SUM(CASE WHEN " + q1 + " THEN NULL ELSE " + q2 + " END), ";

                            cmd.CommandText += "SUM(ug_gyousyuuzai_nousyukuki_all), ";

                            q1 = "";
                            q2 = "";

                            for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                                if(i > 1) {
                                    q1 += " AND ";
                                    q2 += " + ";
                                }

                                q1 += "v_kihoujozai_nousyukuki_" + i.ToString() + " IS NULL";
                                q2 += "IFNULL(v_kihoujozai_nousyukuki_" + i.ToString() + ", 0)";
                            }

                            cmd.CommandText += "SUM(CASE WHEN " + q1 + " THEN NULL ELSE " + q2 + " END), ";

                            cmd.CommandText += "SUM(ug_kihoujozai_nousyukuki_all), " +
                                "SUM(CASE WHEN ov_hansyutsu_namaodei IS NULL AND ov_hansyutsu_yojouodei IS NULL AND ov_hansyutsu_nousyukuodei IS NULL AND ov_hansyutsu_syoukaodei IS NULL AND ov_hansyutsu_dassuiodei IS NULL THEN NULL ELSE IFNULL(ov_hansyutsu_namaodei, 0) + IFNULL(ov_hansyutsu_yojouodei, 0) + IFNULL(ov_hansyutsu_nousyukuodei, 0) + IFNULL(ov_hansyutsu_syoukaodei, 0) + IFNULL(ov_hansyutsu_dassuiodei, 0) END) " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(42, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("ov_hansyutsu_chinsa_pumpjou_all")));
                                    worksheet1.Cell(42, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ov_hansyutsu_shisa_pumpjou_all")));

                                    decimal? ovCS = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("ov_hansyutsu_chinsa"));

                                    if(ovCS != null) {
                                        worksheet1.Cell(42, 33).SetValue(ovCS * 1000);
                                    }

                                    decimal? ovSS = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ov_hansyutsu_shisa"));

                                    if(ovCS != null) {
                                        worksheet1.Cell(42, 34).SetValue(ovSS * 1000);
                                    }

                                    worksheet1.Cell(42, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ov_nama")));
                                    worksheet1.Cell(42, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("sg_nama")));

                                    worksheet1.Cell(42, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ov_yojou")));
                                    worksheet1.Cell(42, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("sg_yojou")));

                                    worksheet1.Cell(42, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ov_nousyuku_output_juuryoku")));
                                    worksheet1.Cell(42, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("sg_nousyuku_output_juuryoku")));

                                    worksheet1.Cell(42, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("ov_nousyuku_output_ex")));
                                    worksheet1.Cell(42, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("sg_nousyuku_output_ex")));

                                    worksheet1.Cell(42, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("ov_syouka_output")));
                                    worksheet1.Cell(42, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("sg_syouka_output")));

                                    worksheet1.Cell(42, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("v_gyousyuuzai_nousyukuki_1")));
                                    worksheet1.Cell(42, 52).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_nousyukuki_all")));

                                    worksheet1.Cell(42, 53).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("v_kihoujozai_nousyukuki_1")));
                                    worksheet1.Cell(42, 55).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("ug_kihoujozai_nousyukuki_all")));

                                    worksheet1.Cell(42, 56).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("ov_hansyutsu_all")));
                                }
                            }
                        }
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, " +
                                "ov_hansyutsu_chinsa_pumpjou_all, ov_hansyutsu_shisa_pumpjou_all, " +
                                "ov_hansyutsu_chinsa, ov_hansyutsu_shisa, " +
                                "ov_nama, oc_nama, sg_nama, " +
                                "ov_yojou, oc_yojou, sg_yojou, " +
                                "ov_nousyuku_output_juuryoku, oc_nousyuku_output_juuryoku, sg_nousyuku_output_juuryoku, " +
                                "ov_nousyuku_output_ex, oc_nousyuku_output_ex, sg_nousyuku_output_ex, " +
                                "ov_syouka_output, oc_syouka_output, sg_syouka_output, ";

                            string q1 = "";
                            string q2 = "";

                            for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                                if(i > 1) {
                                    q1 += " AND ";
                                    q2 += " + ";
                                }

                                q1 += "v_gyousyuuzai_nousyukuki_" + i.ToString() + " IS NULL";
                                q2 += "IFNULL(v_gyousyuuzai_nousyukuki_" + i.ToString() + ", 0)";
                            }

                            cmd.CommandText += "CASE WHEN " + q1 + " THEN NULL ELSE " + q2 + " END, ";

                            cmd.CommandText += "ug_gyousyuuzai_nousyukuki_all, ";

                            q1 = "";
                            q2 = "";

                            for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                                if(i > 1) {
                                    q1 += " AND ";
                                    q2 += " + ";
                                }

                                q1 += "v_kihoujozai_nousyukuki_" + i.ToString() + " IS NULL";
                                q2 += "IFNULL(v_kihoujozai_nousyukuki_" + i.ToString() + ", 0)";
                            }

                            cmd.CommandText += "CASE WHEN " + q1 + " THEN NULL ELSE " + q2 + " END, ";

                            cmd.CommandText += "ug_kihoujozai_nousyukuki_all, " +
                                "CASE WHEN ov_hansyutsu_namaodei IS NULL AND ov_hansyutsu_yojouodei IS NULL AND ov_hansyutsu_nousyukuodei IS NULL AND ov_hansyutsu_syoukaodei IS NULL AND ov_hansyutsu_dassuiodei IS NULL THEN NULL ELSE IFNULL(ov_hansyutsu_namaodei, 0) + IFNULL(ov_hansyutsu_yojouodei, 0) + IFNULL(ov_hansyutsu_nousyukuodei, 0) + IFNULL(ov_hansyutsu_syoukaodei, 0) + IFNULL(ov_hansyutsu_dassuiodei, 0) END " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ov_hansyutsu_chinsa_pumpjou_all")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("ov_hansyutsu_shisa_pumpjou_all")));

                                    decimal? ovCS = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ov_hansyutsu_chinsa"));

                                    if(ovCS != null) {
                                        worksheet1.Cell(R, 33).SetValue(ovCS * 1000);
                                    }

                                    decimal? ovSS = FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ov_hansyutsu_shisa"));

                                    if(ovSS != null) {
                                        worksheet1.Cell(R, 34).SetValue(ovSS * 1000);
                                    }

                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("ov_nama")));
                                    worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("oc_nama")));
                                    worksheet1.Cell(R, 37).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("sg_nama")));

                                    worksheet1.Cell(R, 38).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ov_yojou")));
                                    worksheet1.Cell(R, 39).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 9), ColumnsConfigSet.nameToFormat("oc_yojou")));
                                    worksheet1.Cell(R, 40).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 10), ColumnsConfigSet.nameToFormat("sg_yojou")));

                                    worksheet1.Cell(R, 41).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 11), ColumnsConfigSet.nameToFormat("ov_nousyuku_output_juuryoku")));
                                    worksheet1.Cell(R, 42).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 12), ColumnsConfigSet.nameToFormat("oc_nousyuku_output_juuryoku")));
                                    worksheet1.Cell(R, 43).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 13), ColumnsConfigSet.nameToFormat("sg_nousyuku_output_juuryoku")));

                                    worksheet1.Cell(R, 44).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 14), ColumnsConfigSet.nameToFormat("ov_nousyuku_output_ex")));
                                    worksheet1.Cell(R, 45).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 15), ColumnsConfigSet.nameToFormat("oc_nousyuku_output_ex")));
                                    worksheet1.Cell(R, 46).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 16), ColumnsConfigSet.nameToFormat("sg_nousyuku_output_ex")));

                                    worksheet1.Cell(R, 47).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 17), ColumnsConfigSet.nameToFormat("ov_syouka_output")));
                                    worksheet1.Cell(R, 48).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 18), ColumnsConfigSet.nameToFormat("oc_syouka_output")));
                                    worksheet1.Cell(R, 49).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 19), ColumnsConfigSet.nameToFormat("sg_syouka_output")));

                                    worksheet1.Cell(R, 50).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 20), ColumnsConfigSet.nameToFormat("v_gyousyuuzai_nousyukuki_1")));
                                    worksheet1.Cell(R, 52).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 21), ColumnsConfigSet.nameToFormat("ug_gyousyuuzai_nousyukuki_all")));

                                    worksheet1.Cell(R, 53).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 22), ColumnsConfigSet.nameToFormat("v_kihoujozai_nousyukuki_1")));
                                    worksheet1.Cell(R, 55).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 23), ColumnsConfigSet.nameToFormat("ug_kihoujozai_nousyukuki_all")));

                                    worksheet1.Cell(R, 56).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 24), ColumnsConfigSet.nameToFormat("ov_hansyutsu_all")));
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //施設管理月報1シートの記入
            void makeSheet14(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("施設管理月報1", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 29).SetValue(cDtY);
                    worksheet1.Cell(1, 31).SetValue(cDtM);
                    worksheet1.Cell(1, 26).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //名称は最新値を採用
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        for(int i = 1; i <= 16; i++) {
                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_pumpjou_" + i.ToString() + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_pumpjou_" + i.ToString() + " IS NOT NULL AND name_pumpjou_" + i.ToString() + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name_pumpjou = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(3, 35 + i).SetValue(name_pumpjou);
                                    }
                                    else {
                                        worksheet1.Range(3, 10 + i, 6, 10 + i).SetValue("");
                                        worksheet1.Range(3, 35 + i, 6, 35 + i).SetValue("");
                                    }
                                }
                            }
                        }
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(ee_juden), SUM(ee_jikahatsu), SUM(ee_mizusyori), SUM(ee_mizusyori_blower), SUM(ee_odeisyori), SUM(ee_extra), " +
                                "SUM(ee_kakuhanki)";

                            for(int i = 1; i <= 16; i++) {
                                cmd.CommandText += ", SUM(ee_pumpjou_" + i.ToString() + ")";
                            }

                            cmd.CommandText += " " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(42, 28).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("ee_juden")));
                                    worksheet1.Cell(42, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ee_jikahatsu")));
                                    worksheet1.Cell(42, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("ee_mizusyori")));
                                    worksheet1.Cell(42, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ee_mizusyori_blower")));
                                    worksheet1.Cell(42, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ee_odeisyori")));
                                    worksheet1.Cell(42, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("ee_extra")));

                                    worksheet1.Cell(42, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ee_kakuhanki")));

                                    int C = 35;
                                    int r = 6;

                                    for(int i = 1; i <= 16; i++) {
                                        worksheet1.Cell(42, C + i).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r + i), ColumnsConfigSet.nameToFormat("ee_pumpjou_1")));
                                    }
                                }
                            }
                        }
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, " +
                                "ee_juden, maxep_juden, ee_jikahatsu, ee_mizusyori, ee_mizusyori_blower, ee_odeisyori, ee_extra, " +
                                "ee_kakuhanki";

                            for(int i = 1; i <= 16; i++) {
                                cmd.CommandText += ", ee_pumpjou_" + i.ToString();
                            }

                            cmd.CommandText += " " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;

                                    worksheet1.Cell(R, 28).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("ee_juden")));
                                    worksheet1.Cell(R, 29).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 2), ColumnsConfigSet.nameToFormat("maxep_juden")));
                                    worksheet1.Cell(R, 30).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 3), ColumnsConfigSet.nameToFormat("ee_jikahatsu")));
                                    worksheet1.Cell(R, 31).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 4), ColumnsConfigSet.nameToFormat("ee_mizusyori")));
                                    worksheet1.Cell(R, 32).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 5), ColumnsConfigSet.nameToFormat("ee_mizusyori_blower")));
                                    worksheet1.Cell(R, 33).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 6), ColumnsConfigSet.nameToFormat("ee_odeisyori")));
                                    worksheet1.Cell(R, 34).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 7), ColumnsConfigSet.nameToFormat("ee_extra")));

                                    worksheet1.Cell(R, 35).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 8), ColumnsConfigSet.nameToFormat("ee_kakuhanki")));

                                    int C = 35;
                                    int r = 8;

                                    for(int i = 1; i <= 16; i++) {
                                        worksheet1.Cell(R, C + i).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r + i), ColumnsConfigSet.nameToFormat("ee_pumpjou_1")));
                                    }
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //施設管理月報2シートの記入
            void makeSheet15(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("施設管理月報2", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 29).SetValue(cDtY);
                    worksheet1.Cell(1, 31).SetValue(cDtM);
                    worksheet1.Cell(1, 26).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(37, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(37, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(36, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(36, 2).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(35, 1).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(35, 2).Clear(XLClearOptions.Contents);
                    }

                    //名称は最新値を採用
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        for(int i = 1; i <= 16; i++) {
                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_pumpjou_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_pumpjou_" + i + " IS NOT NULL AND name_pumpjou_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name_pumpjou = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(3, 27 + i).SetValue(name_pumpjou);
                                    }
                                    else {
                                        worksheet1.Range(3, 2 + i, 6, 2 + i).SetValue("");
                                        worksheet1.Range(3, 27 + i, 6, 27 + i).SetValue("");
                                    }
                                }
                            }
                        }

                        for(int i = 1; i <= 8; i++) {
                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_kansen_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_kansen_" + i + " IS NOT NULL AND name_kansen_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name_kansen = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(3, 43 + i).SetValue(name_kansen);
                                    }
                                    else {
                                        worksheet1.Range(3, 18 + i, 6, 18 + i).SetValue("");
                                        worksheet1.Range(3, 43 + i, 6, 43 + i).SetValue("");
                                    }
                                }
                            }
                        }
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT ";

                            for(int i = 1; i <= 16; i++) {
                                if(i > 1) {
                                    cmd.CommandText += ", ";
                                }

                                cmd.CommandText += "SUM(wv_pumpjou_" + i.ToString() + ")";
                            }

                            for(int i = 1; i <= 8; i++) {
                                cmd.CommandText += ", SUM(wv_kansen_" + i.ToString() + ")";
                            }

                            cmd.CommandText += " " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    int C = 27;
                                    int r = -1;

                                    for(int i = 1; i <= 16; i++) {
                                        worksheet1.Cell(42, C + i).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r + i), ColumnsConfigSet.nameToFormat("wv_pumpjou_1")));
                                    }

                                    C = 43;
                                    r = 15;

                                    for(int i = 1; i <= 8; i++) {
                                        worksheet1.Cell(42, C + i).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r + i), ColumnsConfigSet.nameToFormat("wv_kansen_1")));
                                    }
                                }
                            }
                        }
                    }

                    //月集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt";

                            for(int i = 1; i <= 16; i++) {
                                cmd.CommandText += ", wv_pumpjou_" + i.ToString();
                            }

                            for(int i = 1; i <= 8; i++) {
                                cmd.CommandText += ", wv_kansen_" + i.ToString();
                            }

                            cmd.CommandText += " " +
                                "FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int R = d + 6;
                                    int C = 27;
                                    int r = 0;

                                    for(int i = 1; i <= 16; i++) {
                                        worksheet1.Cell(R, C + i).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r + i), ColumnsConfigSet.nameToFormat("wv_pumpjou_1")));
                                    }

                                    C = 43;
                                    r = 16;

                                    for(int i = 1; i <= 8; i++) {
                                        worksheet1.Cell(R, C + i).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r + i), ColumnsConfigSet.nameToFormat("wv_kansen_1")));
                                    }

                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //運転時間月報1シートの記入
            void makeSheet16(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("運転時間月報1", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 39).SetValue(cDtY);
                    worksheet1.Cell(1, 41).SetValue(cDtM);
                    worksheet1.Cell(1, 36).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(2, 34).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 34).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(2, 33).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 33).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(2, 32).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 32).Clear(XLClearOptions.Contents);
                    }

                    //名称は最新値を採用
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        int R = 4;

                        for(int i = 1; i <= 12; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_chinsachi_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_chinsachi_" + i + " IS NOT NULL AND name_chinsachi_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(R, 2).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }

                        for(int i = 1; i <= 12; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_syupump_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_syupump_" + i + " IS NOT NULL AND name_syupump_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(R, 2).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }

                        for(int i = 1; i <= 12; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_syochin_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_syochin_" + i + " IS NOT NULL AND name_syochin_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(R, 2).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(uryou)";

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", SUM(dt_chinsachi_" + i + ")";
                            }

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", SUM(dt_syupump_" + i + ")";
                            }

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", SUM(dt_syochin_" + i + ")";
                            }

                            cmd.CommandText += " FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(4, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("uryou")));

                                    int R = 4;
                                    int r = 0;

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_chinsachi_1")));
                                    }

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_syupump_1")));
                                    }

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_syochin_1")));
                                    }
                                }
                            }
                        }
                    }

                    //日毎の値
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, uryou";

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", dt_chinsachi_" + i;
                            }

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", dt_syupump_" + i;
                            }

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", dt_syochin_" + i;
                            }

                            cmd.CommandText += " FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int C = d + 3;

                                    worksheet1.Cell(4, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("uryou")));

                                    int R = 4;
                                    int r = 1;

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_chinsachi_1")));
                                    }

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_syupump_1")));
                                    }

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_syochin_1")));
                                    }
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //運転時間月報2シートの記入
            void makeSheet17(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("運転時間月報2", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 39).SetValue(cDtY);
                    worksheet1.Cell(1, 41).SetValue(cDtM);
                    worksheet1.Cell(1, 36).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(2, 34).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 34).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(2, 33).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 33).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(2, 32).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 32).Clear(XLClearOptions.Contents);
                    }

                    //名称は最新値を採用
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        int R = 4;

                        for(int i = 1; i <= 12; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_hannoutank_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_hannoutank_" + i + " IS NOT NULL AND name_hannoutank_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(R, 2).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }

                        for(int i = 1; i <= 12; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_soufuuki_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_soufuuki_" + i + " IS NOT NULL AND name_soufuuki_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(R, 2).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }

                        for(int i = 1; i <= 12; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_syuuchin_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_syuuchin_" + i + " IS NOT NULL AND name_syuuchin_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(R, 2).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(uryou)";

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", SUM(dt_hannoutank_" + i + ")";
                            }

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", SUM(dt_soufuuki_" + i + ")";
                            }

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", SUM(dt_syuuchin_" + i + ")";
                            }

                            cmd.CommandText += " FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(4, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("uryou")));

                                    int R = 4;
                                    int r = 0;

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_hannoutank_1")));
                                    }

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_soufuuki_1")));
                                    }

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_syuuchin_1")));
                                    }
                                }
                            }
                        }
                    }

                    //日毎の値
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, uryou";

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", dt_hannoutank_" + i;
                            }

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", dt_soufuuki_" + i;
                            }

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", dt_syuuchin_" + i;
                            }

                            cmd.CommandText += " FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int C = d + 3;

                                    worksheet1.Cell(4, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("uryou")));

                                    int R = 4;
                                    int r = 1;

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_hannoutank_1")));
                                    }

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_soufuuki_1")));
                                    }

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_syuuchin_1")));
                                    }
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }




            //運転時間月報3シートの記入
            void makeSheet18(XLWorkbook workbook1) {
                IXLWorksheet worksheet1;

                if(workbook1.TryGetWorksheet("運転時間月報3", out worksheet1)) {
                    //年月、管理者・処理場
                    worksheet1.Cell(1, 39).SetValue(cDtY);
                    worksheet1.Cell(1, 41).SetValue(cDtM);
                    worksheet1.Cell(1, 36).SetValue(kanrisya + "　" + syorijou);

                    //月にない日を消去
                    if(monthLastDay < 31) {
                        worksheet1.Cell(2, 34).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 34).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 30) {
                        worksheet1.Cell(2, 33).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 33).Clear(XLClearOptions.Contents);
                    }

                    if(monthLastDay < 29) {
                        worksheet1.Cell(2, 32).Clear(XLClearOptions.Contents);
                        worksheet1.Cell(3, 32).Clear(XLClearOptions.Contents);
                    }

                    //名称は最新値を採用
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        int R = 4;

                        for(int i = 1; i <= 12; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_houryuupump_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_houryuupump_" + i + " IS NOT NULL AND name_houryuupump_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(R, 2).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }

                        for(int i = 1; i <= 8; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_nousyukuki_" + i + ", " +
                                    "type_nousyukuki_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_nousyukuki_" + i + " IS NOT NULL AND name_nousyukuki_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);
                                        string type = FunctionForDatabase.getStringOrNull(reader, 1);

                                        worksheet1.Cell(R, 2).SetValue(type + "(" + name + ")");
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }

                        for(int i = 1; i <= 4; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_syoukatank_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_syoukatank_" + i + " IS NOT NULL AND name_syoukatank_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);

                                        worksheet1.Cell(R, 2).SetValue(name);
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }

                        for(int i = 1; i <= 8; i++) {
                            R++;

                            using(var cmd = con.CreateCommand()) {
                                cmd.CommandText = "SELECT " +
                                    "name_dassuiki_" + i + ", " +
                                    "type_dassuiki_" + i + " " +
                                    "FROM suishitsu_kanri " +
                                    "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                    "AND name_dassuiki_" + i + " IS NOT NULL AND name_dassuiki_" + i + "<>''" +
                                    "ORDER BY sokutei_dt DESC LIMIT 1";

                                using(var reader = cmd.ExecuteReader()) {
                                    if(reader.Read()) {
                                        string name = FunctionForDatabase.getStringOrNull(reader, 0);
                                        string type = FunctionForDatabase.getStringOrNull(reader, 1);

                                        worksheet1.Cell(R, 2).SetValue(type + "(" + name + ")");
                                    }
                                    else {
                                        worksheet1.Range(R, 2, R, 3).SetValue("");
                                    }
                                }
                            }
                        }
                    }

                    //年度集計
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "SUM(uryou)";

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", SUM(dt_houryuupump_" + i + ")";
                            }

                            for(int i = 1; i <= 8; i++) {
                                cmd.CommandText += ", SUM(dt_nousyukuki_" + i + ")";
                            }

                            for(int i = 1; i <= 4; i++) {
                                cmd.CommandText += ", SUM(dt_syoukatank_" + i + ")";
                            }

                            for(int i = 1; i <= 8; i++) {
                                cmd.CommandText += ", SUM(dt_dassuiki_" + i + ")";
                            }

                            cmd.CommandText += " FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stBYSTR + "' AND sokutei_dt<'" + edBYSTR + "'";

                            using(var reader = cmd.ExecuteReader()) {
                                if(reader.Read()) {
                                    worksheet1.Cell(4, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 0), ColumnsConfigSet.nameToFormat("uryou")));

                                    int R = 4;
                                    int r = 0;

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_houryuupump_1")));
                                    }

                                    for(int i = 1; i <= 8; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_nousyukuki_1")));
                                    }

                                    for(int i = 1; i <= 4; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_syoukatank_1")));
                                    }

                                    for(int i = 1; i <= 8; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, 36).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_dassuiki_1")));
                                    }
                                }
                            }
                        }
                    }

                    //日毎の値
                    using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                        con.Open();

                        using(var cmd = con.CreateCommand()) {
                            cmd.CommandText = "SELECT " +
                                "sokutei_dt, uryou";

                            for(int i = 1; i <= 12; i++) {
                                cmd.CommandText += ", dt_houryuupump_" + i;
                            }

                            for(int i = 1; i <= 8; i++) {
                                cmd.CommandText += ", dt_nousyukuki_" + i;
                            }

                            for(int i = 1; i <= 4; i++) {
                                cmd.CommandText += ", dt_syoukatank_" + i;
                            }

                            for(int i = 1; i <= 8; i++) {
                                cmd.CommandText += ", dt_dassuiki_" + i;
                            }

                            cmd.CommandText += " FROM suishitsu_kanri " +
                                "WHERE del_flg IS NULL AND kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "' AND sokutei_dt>='" + stDtSTR + "' AND sokutei_dt<'" + edDtSTR + "' " +
                                "ORDER BY sokutei_dt";

                            using(var reader = cmd.ExecuteReader()) {
                                int d = 0;

                                while(reader.Read()) {
                                    while(reader.GetDateTime(0).Day != d) {
                                        d++;
                                    }

                                    int C = d + 3;

                                    worksheet1.Cell(4, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, 1), ColumnsConfigSet.nameToFormat("uryou")));

                                    int R = 4;
                                    int r = 1;

                                    for(int i = 1; i <= 12; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_houryuupump_1")));
                                    }

                                    for(int i = 1; i <= 8; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_nousyukuki_1")));
                                    }

                                    for(int i = 1; i <= 4; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_syoukatank_1")));
                                    }

                                    for(int i = 1; i <= 8; i++) {
                                        R++;
                                        r++;
                                        worksheet1.Cell(R, C).SetValue(FunctionForDatabase.decConvFromDB(FunctionForDatabase.getDecimalOrNull(reader, r), ColumnsConfigSet.nameToFormat("dt_dassuiki_1")));
                                    }
                                }
                            }
                        }
                    }

                    worksheet1.SheetView.View = XLSheetViewOptions.PageBreakPreview;
                }
                else {
                    throw new Exception();
                }

                return;
            }
        }

    }
}
