﻿// 下水データ管理システム
// Copyright (C) 2018 公益財団法人とちぎ建設技術センター(https://www.tochigictc.or.jp/)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.If not, see<http://www.gnu.org/licenses/>.

using System;
using System.Collections.Generic;
using System.Data.SQLite;
using System.Reflection;
using System.Windows.Forms;

namespace WindowsFormsApplication1 {
    /// <summary>
    /// このプログラム自体のプロパティ
    /// </summary>
    public static class ThisProgramConfig {
        public static string version { get; } = Assembly.GetExecutingAssembly().GetName().Version.ToString();
    }




    /// <summary>
    /// データベース設定のクラス
    /// </summary>
    public static class DataBaseConfig {
        public static string ConnectionString { get; } = @"Data Source=savedata.db";
    }




    /// <summary>
    /// このプログラムで取り扱う規模の最大を設定するクラス
    /// </summary>
    public static class MaxSizeConfig {
        //データの最大数
        public static int maxPumpjou { get; } = 8;
        public static int maxKansen { get; } = 8;
        public static int maxChinsachi { get; } = 8;
        public static int maxSyupump { get; } = 8;
        public static int maxSyochin { get; } = 8;
        public static int maxSoufuuki { get; } = 8;
        public static int maxHannoutank { get; } = 12;
        public static int maxSyuuchin { get; } = 8;
        public static int maxHouryuupump { get; } = 8;
        public static int maxNousyukuki { get; } = 8;
        public static int maxJuuryokunousyuku { get; } = 4;
        public static int maxKikainousyuku { get; } = 4;
        public static int maxSyoukatank { get; } = 4;
        public static int maxDassuiki { get; } = 8;
        public static int maxSpecialAnalysis { get; } = 4;

        //プログラムの上限
        public static int maxPrintForm { get; } = 10;
    }




    /// <summary>
    /// 入力欄をグループ化するタブの設定のクラス
    /// </summary>
    public static class InputTab {
        static int hannouTabNo = (MaxSizeConfig.maxHannoutank - 1) / 4 + 1;
        static int dassuiTabNo = (MaxSizeConfig.maxDassuiki - 1) / 4 + 1;

        public static List<string> title {
            get {
                List<string> tabTitle = new List<string>();

                tabTitle.Add("基本");
                tabTitle.Add("P場・幹線");

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i = i + 8) {
                    string tabName = "ポンプ場" + i.ToString();

                    if(MaxSizeConfig.maxPumpjou < i + 7) {
                        tabName += "-" + MaxSizeConfig.maxPumpjou.ToString();
                    }
                    else {
                        tabName += "-" + (i + 7).ToString();
                    }

                    tabTitle.Add(tabName);
                }

                tabTitle.Add("処理場");
                tabTitle.Add("流入・沈砂");
                tabTitle.Add("初沈");
                tabTitle.Add("反応T共通");

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i = i + 4) {
                    string tabName = "反応T" + i.ToString();

                    if(MaxSizeConfig.maxHannoutank < i + 3) {
                        tabName += "-" + MaxSizeConfig.maxHannoutank.ToString();
                    }
                    else {
                        tabName += "-" + (i + 3).ToString();
                    }

                    tabTitle.Add(tabName);
                }

                tabTitle.Add("終沈");
                tabTitle.Add("塩混・放流");
                tabTitle.Add("汚泥濃縮");

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i = i + 4) {
                    string tabName = "濃縮設備" + i.ToString();

                    if(MaxSizeConfig.maxNousyukuki < i + 3) {
                        tabName += "-" + MaxSizeConfig.maxNousyukuki.ToString();
                    }
                    else {
                        tabName += "-" + (i + 3).ToString();
                    }

                    tabTitle.Add(tabName);
                }

                tabTitle.Add("汚泥消化");
                tabTitle.Add("汚泥脱水");

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i = i + 4) {
                    string tabName = "脱水設備" + i.ToString();

                    if(MaxSizeConfig.maxDassuiki < i + 3) {
                        tabName += "-" + MaxSizeConfig.maxDassuiki.ToString();
                    }
                    else {
                        tabName += "-" + (i + 3).ToString();
                    }

                    tabTitle.Add(tabName);
                }

                tabTitle.Add("汚泥等搬出");

                for(int i = 1; i <= MaxSizeConfig.maxSpecialAnalysis; i++) {
                    tabTitle.Add("別途分析" + i.ToString());
                }

                return tabTitle;
            }
        }
    }




    /// <summary>
    /// データベースとdataGridViewの列の設定のクラス
    /// </summary>
    public class ColumnConfig {
        public int index { get; set; }
        public string HeaderText { get; set; }
        public string Name { get; set; }
        public string unit { get; set; } = "";
        public int putPlaceNo { get; set; } = -1;
        public int Width { get; set; }
        public DataGridViewContentAlignment Alignment { get; set; }
        public string Format { get; set; } = "";
        public bool Visible { get; set; } = true;
        public string inputType { get; set; } = "";
        public string Items { get; set; }
        public bool limit { get; set; } = false;
        public decimal Minimum { get; set; }
        public decimal Maximum { get; set; }
        public decimal? validMin { get; set; }
        public decimal? validMax { get; set; }
        public bool validNull { get; set; }
        public bool statical { get; set; } = false;
        public string dbColumnType { get; set; } = "";
        public bool calculatedDecimal { get; set; }
		public bool calculatedString { get; set; }
        public decimal? tempValueDec { get; set; }
        public string tempValueStr { get; set; }
        public decimal? previousValueDec { get; set; }
        public string previousValueStr { get; set; }
        public bool calculateSource { get; set; } = false;
        public string calculateFormula { get; set; }
        public string[] forCalc { get; set; }
    }




    /// <summary>
    /// データベースとdataGridViewの列の設定を格納するクラス
    /// </summary>
    public static class ColumnsConfigSet {
        //全列を取得
        public static List<ColumnConfig> getSet {
            get {
                List<ColumnConfig> columnConfigs = new List<ColumnConfig>();
                ColumnConfig[] c = new ColumnConfig[2000];

                int panelNo = 4;
                int currentTab = 10;

                int i = 0;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "id";
                c[i].Name = "id";
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].Visible = false;
                c[i].dbColumnType = "mk";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "del_flg";
                c[i].Name = "del_flg";
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].Visible = false;
                columnConfigs.Add(c[i]);


                #region 基本データ定義

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "管理者";
                c[i].Name = "kanrisya";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Width = 65;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].Visible = false;
                c[i].inputType = "TextBox";
                c[i].validNull = true;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "処理場";
                c[i].Name = "syorijou";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Width = 130;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].Visible = false;
                c[i].inputType = "TextBox";
                c[i].validNull = true;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "測定日";
                c[i].Name = "sokutei_dt";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Width = 95;
                c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                c[i].Format = "yyyy/MM/dd ddd";
                c[i].inputType = "DateTimePicker";
                c[i].dbColumnType = "ix";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "天候";
                c[i].Name = "tenkou";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                c[i].inputType = "ComboBox";
                c[i].Items = @",晴,晴曇,晴雨,晴雪,曇,曇晴,曇雨,曇雪,雨,雨晴,雨曇,雨雪,雪,雪晴,雪曇,雪雨";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "気温";
                c[i].Name = "kion";
                c[i].unit = "[℃]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Width = 38;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99.9M;
                c[i].Minimum = -99.9M;
                c[i].validMax = 40.0M;
                c[i].validMin = -20.0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "雨量";
                c[i].Name = "uryou";
                c[i].unit = "[mm]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 500M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "晴天";
                c[i].Name = "seiten";
                c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                c[i].calculatedString = true;
                c[i].calculateFormula = "晴天:(雨量[測定日]<雨天条件雨量[0日] ∧ 雨量[測定日-1]<雨天条件雨量[-1日] ∧ 雨量[測定日-2]<雨天条件雨量[-2日])　雨天:￢晴天";
                columnConfigs.Add(c[i]);

                #endregion


                #region P場・幹線データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "P場排水量計";
                c[i].Name = "wv_pumpjou_all";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "ΣP場n排水量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "P場電力量計";
                c[i].Name = "ee_pumpjou_all";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "ΣP場n電力量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "P場A重油計";
                c[i].Name = "v_hao_pumpjou_all";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "ΣP場nA重油";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "P場軽油計";
                c[i].Name = "v_lo_pumpjou_all";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "ΣP場n軽油";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "P場沈砂発生量計";
                c[i].Name = "v_chinsa_pumpjou_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "ΣP場n沈砂発生量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "P場しさ発生量計";
                c[i].Name = "v_shisa_pumpjou_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "ΣP場nしさ発生量";
                columnConfigs.Add(c[i]);

                for(int j = 1; j <= MaxSizeConfig.maxKansen; j++) {
                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "幹線" + j.ToString() + "名称";
                    c[i].Name = "name_kansen_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "幹線" + j.ToString() + "流量";
                    c[i].Name = "wv_kansen_" + j.ToString();
                    c[i].unit = "[m3]";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 9999999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }

                #endregion


                #region P場個別データ定義

                currentTab += 10;

                int pumpjouPlace = 0;

                for(int j = 1; j <= MaxSizeConfig.maxPumpjou; j++) {
                    if((j - 1) % 2 == 0) {
                        pumpjouPlace++;
                    }

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "P場" + j.ToString() + "名称";
                    c[i].Name = "name_pumpjou_" + j.ToString();
                    c[i].putPlaceNo = currentTab + pumpjouPlace;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "P場" + j.ToString() + "排水量";
                    c[i].Name = "wv_pumpjou_" + j.ToString();
                    c[i].unit = "[m3]";
                    c[i].putPlaceNo = currentTab + pumpjouPlace;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 9999999M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "wv_pumpjou_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "P場" + j.ToString() + "電力量";
                    c[i].Name = "ee_pumpjou_" + j.ToString();
                    c[i].unit = "[kWh]";
                    c[i].putPlaceNo = currentTab + pumpjouPlace;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 9999999M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ee_pumpjou_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "P場" + j.ToString() + "A重油";
                    c[i].Name = "v_hao_pumpjou_" + j.ToString();
                    c[i].unit = "[L]";
                    c[i].putPlaceNo = currentTab + pumpjouPlace;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999.9M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "v_hao_pumpjou_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "P場" + j.ToString() + "軽油";
                    c[i].Name = "v_lo_pumpjou_" + j.ToString();
                    c[i].unit = "[L]";
                    c[i].putPlaceNo = currentTab + pumpjouPlace;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999.9M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "v_lo_pumpjou_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "P場" + j.ToString() + "沈砂発生量";
                    c[i].Name = "v_chinsa_pumpjou_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = currentTab + pumpjouPlace;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "v_chinsa_pumpjou_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "P場" + j.ToString() + "しさ発生量";
                    c[i].Name = "v_shisa_pumpjou_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = currentTab + pumpjouPlace;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "v_shisa_pumpjou_all" };
                    columnConfigs.Add(c[i]);
                }

                #endregion


                #region 処理場データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "受電電力量";
                c[i].Name = "ee_juden";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ee_extra" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "最大需要電力";
                c[i].Name = "maxep_juden";
                c[i].unit = "[kW]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "自家発電電力量";
                c[i].Name = "ee_jikahatsu";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ee_extra" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "場内P場電力量";
                c[i].Name = "ee_jounaipump";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ee_extra" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "水処理電力量";
                c[i].Name = "ee_mizusyori";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ee_extra" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "うちブロア";
                c[i].Name = "ee_mizusyori_blower";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "汚泥処理電力量";
                c[i].Name = "ee_odeisyori";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ee_extra" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "その他電力量";
                c[i].Name = "ee_extra";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "受電電力量+自家発電電力量-場内P場電力量-水処理電力量-汚泥処理電力量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "撹拌機電力量";
                c[i].Name = "ee_kakuhanki";
                c[i].unit = "[kWh]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用A重油→管理";
                c[i].Name = "v_hao_kanri";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用A重油→消化槽";
                c[i].Name = "v_hao_syouka";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用A重油→自家発";
                c[i].Name = "v_hao_jikahatsu";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用A重油→その他";
                c[i].Name = "v_hao_sonota";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用軽油→管理";
                c[i].Name = "v_lo_kanri";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用軽油→消化槽";
                c[i].Name = "v_lo_syouka";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用軽油→自家発";
                c[i].Name = "v_lo_jikahatsu";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用軽油→その他";
                c[i].Name = "v_lo_sonota";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用灯油→管理";
                c[i].Name = "v_kerosene_kanri";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用灯油→消化槽";
                c[i].Name = "v_kerosene_syouka";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用灯油→自家発";
                c[i].Name = "v_kerosene_jikahatsu";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用灯油→その他";
                c[i].Name = "v_kerosene_sonota";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用ガソリン→管理";
                c[i].Name = "v_gasoline_kanri";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用ガソリン→消化槽";
                c[i].Name = "v_gasoline_syouka";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用ガソリン→自家発";
                c[i].Name = "v_gasoline_jikahatsu";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用ガソリン→その他";
                c[i].Name = "v_gasoline_sonota";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用LPG→管理";
                c[i].Name = "v_pgas_kanri";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用LPG→消化槽";
                c[i].Name = "v_pgas_syouka";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用LPG→自家発";
                c[i].Name = "v_pgas_jikahatsu";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用LPG→その他";
                c[i].Name = "v_pgas_sonota";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "用水(再利用水)";
                c[i].Name = "wv_sairiyou";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "用水(井戸)";
                c[i].Name = "wv_ido";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "用水(上水道)";
                c[i].Name = "wv_jousui";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                #endregion


                #region 流入・沈砂データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "現有水処理能力";
                c[i].Name = "wv_genyuumizusyori";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].statical = true;
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水量";
                c[i].Name = "wv_ryuunyuu";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "処理場沈砂発生量";
                c[i].Name = "v_chinsa_syorijou";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "処理場しさ発生量";
                c[i].Name = "v_shisa_syorijou";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水水温";
                c[i].Name = "wt_ryuunyuu";
                c[i].unit = "[℃]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99.9M;
                c[i].Minimum = -99.9M;
                c[i].validMax = 30.0M;
                c[i].validMin = 10.0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水外観";
                c[i].Name = "gaikan_ryuunyuu";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水臭気";
                c[i].Name = "syuuki_ryuunyuu";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水透視度";
                c[i].Name = "pr_ryuunyuu";
                c[i].unit = "[cm]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].validMax = 25M;
                c[i].validMin = 2M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水pH";
                c[i].Name = "ph_ryuunyuu";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                c[i].validMax = 8M;
                c[i].validMin = 6.5M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水COD";
                c[i].Name = "cod_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 300M;
                c[i].validMin = 50M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "cod_jokyo_sougou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水BOD";
                c[i].Name = "bod_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 300M;
                c[i].validMin = 50M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "bod_jokyo_sougou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水SS";
                c[i].Name = "ss_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 300M;
                c[i].validMin = 50M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ss_jokyo_sougou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水大腸菌群";
                c[i].Name = "nd_ryuunyuu";
                c[i].unit = "[個/mL]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].validMax = 1500000M;
                c[i].validMin = 20000M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水T-N";
                c[i].Name = "tn_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水T-P";
                c[i].Name = "tp_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水塩素イオン";
                c[i].Name = "clion_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "流入水よう素消費量";
                c[i].Name = "iodineconsum_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                for(int j = 1; j <= MaxSizeConfig.maxChinsachi; j++) {
                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "沈砂地" + j.ToString() + "名称";
                    c[i].Name = "name_chinsachi_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "沈砂地" + j.ToString() + "運転時間";
                    c[i].Name = "dt_chinsachi_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }

                for(int j = 1; j <= MaxSizeConfig.maxSyupump; j++) {
                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "主ポンプ" + j.ToString() + "名称";
                    c[i].Name = "name_syupump_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "主ポンプ" + j.ToString() + "運転時間";
                    c[i].Name = "dt_syupump_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }

                #endregion


                #region 初沈データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈池容量";
                c[i].Name = "pv_syochin";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].statical = true;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "h_syochin" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈流入量";
                c[i].Name = "wv_syochin";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ar_hannoutank_all", "bodpmlss_hannoutank_@tN", "bodpmlss_hannoutank_all", "h_syochin", "h_syuuchin", "or_hensou", "srt_hannoutank_all", "st_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈沈殿時間";
                c[i].Name = "h_syochin";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "初沈池容量/初沈流入量*24";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥→濃縮";
                c[i].Name = "ov_nama_nousyuku";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_nama", "sg_nama" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥→消化";
                c[i].Name = "ov_nama_syouka";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_nama", "sg_nama" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥→脱水";
                c[i].Name = "ov_nama_dassui";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_nama", "sg_nama" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥→場外";
                c[i].Name = "ov_nama_jougai";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_nama", "sg_nama" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥引抜量計";
                c[i].Name = "ov_nama";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "生汚泥→濃縮+生汚泥→消化+生汚泥→脱水+生汚泥→場外";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥濃度";
                c[i].Name = "oc_nama";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_nama" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥固形物量";
                c[i].Name = "sg_nama";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(生汚泥→濃縮+生汚泥→消化+生汚泥→脱水+生汚泥→場外)*生汚泥濃度/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥TS";
                c[i].Name = "ts_nama";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].validMax = 16000M;
                c[i].validMin = 1000M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈流入外観";
                c[i].Name = "gaikan_syo_ryuunyuu";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈流入臭気";
                c[i].Name = "syuuki_syo_ryuunyuu";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈流入透視度";
                c[i].Name = "pr_syo_ryuunyuu";
                c[i].unit = "[cm]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 2M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈流入pH";
                c[i].Name = "ph_syo_ryuunyuu";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                c[i].validMax = 8M;
                c[i].validMin = 6.5M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈流入COD";
                c[i].Name = "cod_syo_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 300M;
                c[i].validMin = 50M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈流入BOD";
                c[i].Name = "bod_syo_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 300M;
                c[i].validMin = 50M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈流入SS";
                c[i].Name = "ss_syo_ryuunyuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 300M;
                c[i].validMin = 50M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ss_jokyo_syo_etsuryuu" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流水温";
                c[i].Name = "wt_syo_etsuryuu";
                c[i].unit = "[℃]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99.9M;
                c[i].Minimum = -99.9M;
                c[i].validMax = 30.0M;
                c[i].validMin = 10.0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流外観";
                c[i].Name = "gaikan_syo_etsuryuu";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流臭気";
                c[i].Name = "syuuki_syo_etsuryuu";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流透視度";
                c[i].Name = "pr_syo_etsuryuu";
                c[i].unit = "[cm]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].validMax = 25M;
                c[i].validMin = 6M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流pH";
                c[i].Name = "ph_syo_etsuryuu";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                c[i].validMax = 8M;
                c[i].validMin = 6.5M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流COD";
                c[i].Name = "cod_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 100M;
                c[i].validMin = 20M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流BOD";
                c[i].Name = "bod_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 150M;
                c[i].validMin = 30M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[MaxSizeConfig.maxHannoutank + 1];
                c[i].forCalc[0] = "bodpmlss_hannoutank_all";

                for(int j = 1; j <= MaxSizeConfig.maxHannoutank; j++) {
                    c[i].forCalc[j] = "bodpmlss_hannoutank_" + j.ToString();
                }

                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流C-BOD";
                c[i].Name = "cbod_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 150M;
                c[i].validMin = 30M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流SS";
                c[i].Name = "ss_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 100M;
                c[i].validMin = 20M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ss_jokyo_syo_etsuryuu" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越除去率SS";
                c[i].Name = "ss_jokyo_syo_etsuryuu";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(初沈越流SS-初沈流入SS)/初沈流入SS*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越大腸菌群";
                c[i].Name = "nd_syo_etsuryuu";
                c[i].unit = "[個/mL]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].validMax = 1500000M;
                c[i].validMin = 20000M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流T-N";
                c[i].Name = "tn_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流有機性N";
                c[i].Name = "organicn_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流NH4-N";
                c[i].Name = "nh4n_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流NO2-N";
                c[i].Name = "no2n_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流NO3-N";
                c[i].Name = "no3n_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流T-P";
                c[i].Name = "tp_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "初沈越流オルトリン";
                c[i].Name = "orthop_syo_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                for(int j = 1; j <= MaxSizeConfig.maxSyochin; j++) {
                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "初沈池" + j.ToString() + "名称";
                    c[i].Name = "name_syochin_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "初沈池" + j.ToString() + "運転時間";
                    c[i].Name = "dt_syochin_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }

                #endregion


                #region 反応T共通データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応Ｔ容量";
                c[i].Name = "pv_hannoutank";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].statical = true;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "bodpmlss_hannoutank_@tN", "bodpmlss_hannoutank_all", "srt_hannoutank_all", "st_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T空気量";
                c[i].Name = "av_hannoutank_all";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ar_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T空気倍率";
                c[i].Name = "ar_hannoutank_all";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "反応T空気量/(初沈流入量-簡易処理水量)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T滞留時間";
                c[i].Name = "st_hannoutank_all";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "反応Ｔ容量/(初沈流入量-簡易処理水量)*24";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T_SRT";
                c[i].Name = "srt_hannoutank_all";
                c[i].unit = "[d]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "反応T容量*(Σ反応TnMLSS/n)/((余剰汚泥→初沈+余剰汚泥→濃縮+余剰汚泥→消化+余剰汚泥→脱水+余剰汚泥→場外)*返送汚泥SS+(初沈流入量-(余剰汚泥→初沈+余剰汚泥→濃縮+余剰汚泥→消化+余剰汚泥→脱水+余剰汚泥→場外))*塩混出口SS)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均水温";
                c[i].Name = "wt_hannoutank_all";
                c[i].unit = "[℃]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応Tn水温/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均pH";
                c[i].Name = "ph_hannoutank_all";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnpH/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均SV";
                c[i].Name = "sv_hannoutank_all";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnSV/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均MLDO前";
                c[i].Name = "mldo_p_hannoutank_all";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnMLDO前/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均MLDO後";
                c[i].Name = "mldo_a_hannoutank_all";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnMLDO後/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均MLSS";
                c[i].Name = "mlss_hannoutank_all";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnMLSS/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均MLVSS";
                c[i].Name = "mlvss_hannoutank_all";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnMLVSS/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均VSS/SS";
                c[i].Name = "vsspss_hannoutank_all";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ(反応TnMLVSS/反応TnMLSS*100)/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均SVI";
                c[i].Name = "svi_hannoutank_all";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ(反応TnSV*10000/反応TnMLSS)/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反T平均BOD/MLSS";
                c[i].Name = "bodpmlss_hannoutank_all";
                c[i].unit = "[/d]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ((初沈流入量-簡易処理水量)*初沈越流BOD/(反応T容量*反応TnMLSS))/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均ORP嫌気";
                c[i].Name = "orp_ken_hannoutank_all";
                c[i].unit = "[mV]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnORP嫌気/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反応T平均ORP好気";
                c[i].Name = "orp_kou_hannoutank_all";
                c[i].unit = "[mV]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnORP好気/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反T平均Rr前段";
                c[i].Name = "rr_zen_hannoutank_all";
                c[i].unit = "[mg/L/h]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnRr前段/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "反T平均Rr後段";
                c[i].Name = "rr_kou_hannoutank_all";
                c[i].unit = "[mg/L/h]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ反応TnRr後段/n (無効な値は除外)";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥pH";
                c[i].Name = "ph_hensou";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                c[i].validMax = 7M;
                c[i].validMin = 6M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥SV";
                c[i].Name = "sv_hensou";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].validMax = 100M;
                c[i].validMin = 70M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "svi_hensou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥DO";
                c[i].Name = "do_hensou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 2M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥SS";
                c[i].Name = "rsss_hensou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].validMax = 6000M;
                c[i].validMin = 3000M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "srt_hannoutank_all", "svi_hensou", "vsspss_hensou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥VSS";
                c[i].Name = "rsvss_hensou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].validMax = 5500M;
                c[i].validMin = 2500M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "vsspss_hensou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥VSS/SS";
                c[i].Name = "vsspss_hensou";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "返送汚泥VSS/返送汚泥SS*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥SVI";
                c[i].Name = "svi_hensou";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "返送汚泥SV*10000/返送汚泥SS";
                columnConfigs.Add(c[i]);

                for(int j = 1; j <= MaxSizeConfig.maxSoufuuki; j++) {
                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "送風機" + j.ToString() + "名称";
                    c[i].Name = "name_soufuuki_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "送風機" + j.ToString() + "運転時間";
                    c[i].Name = "dt_soufuuki_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }

                #endregion


                #region 反応T個別データ定義

                currentTab += 10;

                for(int j = 1; j <= MaxSizeConfig.maxHannoutank; j++) {
                    int putPlaceNo1 = (int)(Math.Floor((j - 1M) / panelNo)) * 10 + currentTab;
                    int putPlaceNo2 = (j - 1) % panelNo + 1;
                    int putPlaceNo = putPlaceNo1 + putPlaceNo2;

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "名称";
                    c[i].Name = "name_hannoutank_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "運転時間";
                    c[i].Name = "dt_hannoutank_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "水温";
                    c[i].Name = "wt_hannoutank_" + j.ToString();
                    c[i].unit = "[℃]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99.9M;
                    c[i].Minimum = -99.9M;
                    c[i].validMax = 30.0M;
                    c[i].validMin = 10.0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "wt_hannoutank_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "外観";
                    c[i].Name = "gaikan_hannoutank_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "臭気";
                    c[i].Name = "syuuki_hannoutank_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "pH";
                    c[i].Name = "ph_hannoutank_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 14M;
                    c[i].Minimum = 0M;
                    c[i].validMax = 7M;
                    c[i].validMin = 6M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ph_hannoutank_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "SV";
                    c[i].Name = "sv_hannoutank_" + j.ToString();
                    c[i].unit = "[%]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 100M;
                    c[i].Minimum = 0M;
                    c[i].validMax = 80M;
                    c[i].validMin = 10M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "sv_hannoutank_all", "svi_hannoutank_all", "svi_hannoutank_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "MLDO前";
                    c[i].Name = "mldo_p_hannoutank_" + j.ToString();
                    c[i].unit = "[mg/L]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    c[i].validMax = 2M;
                    c[i].validMin = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "mldo_p_hannoutank_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "MLDO後";
                    c[i].Name = "mldo_a_hannoutank_" + j.ToString();
                    c[i].unit = "[mg/L]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    c[i].validMax = 3M;
                    c[i].validMin = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "mldo_a_hannoutank_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "MLSS";
                    c[i].Name = "mlss_hannoutank_" + j.ToString();
                    c[i].unit = "[mg/L]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = 0M;
                    c[i].validMax = 3000M;
                    c[i].validMin = 900M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "bodpmlss_hannoutank_all", "mlss_hannoutank_all", "srt_hannoutank_all", "svi_hannoutank_all", "vsspss_hannoutank_all", "bodpmlss_hannoutank_" + j.ToString(), "svi_hannoutank_" + j.ToString(), "vsspss_hannoutank_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "MLVSS";
                    c[i].Name = "mlvss_hannoutank_" + j.ToString();
                    c[i].unit = "[mg/L]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = 0M;
                    c[i].validMax = 2500M;
                    c[i].validMin = 800M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "mlvss_hannoutank_all", "vsspss_hannoutank_all", "vsspss_hannoutank_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "VSS/SS";
                    c[i].Name = "vsspss_hannoutank_" + j.ToString();
                    c[i].unit = "[%]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "反応T" + j.ToString() + "MLVSS/反応T" + j.ToString() + "MLSS*100";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "SVI";
                    c[i].Name = "svi_hannoutank_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "反応T" + j.ToString() + "SV*10000/反応T" + j.ToString() + "MLSS";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "ORP嫌気";
                    c[i].Name = "orp_ken_hannoutank_" + j.ToString();
                    c[i].unit = "[mV]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = -999999M;
                    c[i].validMax = 200M;
                    c[i].validMin = -350M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "orp_ken_hannoutank_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "ORP好気";
                    c[i].Name = "orp_kou_hannoutank_" + j.ToString();
                    c[i].unit = "[mV]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = -999999M;
                    c[i].validMax = 300M;
                    c[i].validMin = 90M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "orp_kou_hannoutank_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "Rr前段";
                    c[i].Name = "rr_zen_hannoutank_" + j.ToString();
                    c[i].unit = "[mg/L/h]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = -999999M;
                    c[i].validMax = 60M;
                    c[i].validMin = 10M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "rr_zen_hannoutank_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "反応T" + j.ToString() + "Rr後段";
                    c[i].Name = "rr_kou_hannoutank_" + j.ToString();
                    c[i].unit = "[mg/L/h]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = -999999M;
                    c[i].validMax = 60M;
                    c[i].validMin = 10M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "rr_kou_hannoutank_all" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "T" + j.ToString() + "BOD/MLSS";
                    c[i].Name = "bodpmlss_hannoutank_" + j.ToString();
                    c[i].unit = "[/d]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "(初沈流入量-簡易処理水量)*初沈越流BOD/(反応T容量*反応T" + j.ToString() + "MLSS)";
                    columnConfigs.Add(c[i]);
                }

                currentTab = (int)(Math.Floor((MaxSizeConfig.maxHannoutank - 1M) / panelNo)) * 10 + currentTab;

                #endregion


                #region 終沈データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈池容量";
                c[i].Name = "pv_syuuchin";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].statical = true;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "h_syuuchin" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈沈殿時間";
                c[i].Name = "h_syuuchin";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "終沈池容量/(初沈流入量-簡易処理水量)*24";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥量";
                c[i].Name = "ov_hensou";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "or_hensou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "返送汚泥率";
                c[i].Name = "or_hensou";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "返送汚泥量/(初沈流入量-簡易処理水量)*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥→初沈";
                c[i].Name = "ov_yj_syochin";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_yojou", "sg_yojou", "srt_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥→濃縮";
                c[i].Name = "ov_yj_nousyuku";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_yojou", "sg_yojou", "srt_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥→消化";
                c[i].Name = "ov_yj_syouka";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_yojou", "sg_yojou", "srt_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥→脱水";
                c[i].Name = "ov_yj_dassui";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_yojou", "sg_yojou", "srt_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥→場外";
                c[i].Name = "ov_yj_jougai";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_yojou", "sg_yojou", "srt_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥量計";
                c[i].Name = "ov_yojou";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "余剰汚泥→初沈+余剰汚泥→濃縮+余剰汚泥→消化+余剰汚泥→脱水+余剰汚泥→場外";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥濃度";
                c[i].Name = "oc_yojou";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_yojou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥固形物量";
                c[i].Name = "sg_yojou";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(余剰汚泥→初沈+余剰汚泥→濃縮+余剰汚泥→消化+余剰汚泥→脱水+余剰汚泥→場外)*余剰汚泥濃度/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流水温";
                c[i].Name = "wt_syuu_etsuryuu";
                c[i].unit = "[℃]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99.9M;
                c[i].Minimum = -99.9M;
                c[i].validMax = 30.0M;
                c[i].validMin = 10.0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流外観";
                c[i].Name = "gaikan_syuu_etsuryuu";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流臭気";
                c[i].Name = "syuuki_syuu_etsuryuu";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流透視度";
                c[i].Name = "pr_syuu_etsuryuu";
                c[i].unit = "[cm]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].validMax = 100M;
                c[i].validMin = 50M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流pH";
                c[i].Name = "ph_syuu_etsuryuu";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                c[i].validMax = 7M;
                c[i].validMin = 6.2M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流COD";
                c[i].Name = "cod_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流BOD";
                c[i].Name = "bod_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 25M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流C-BOD";
                c[i].Name = "cbod_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流SS";
                c[i].Name = "ss_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 40M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越大腸菌群";
                c[i].Name = "nd_syuu_etsuryuu";
                c[i].unit = "[個/mL]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].validMax = 10000M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越蒸発残留物";
                c[i].Name = "jouhatsuzan_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越強熱残留物";
                c[i].Name = "kyounetsuzan_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越強熱減量";
                c[i].Name = "kyounetsugen_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越溶解性物質";
                c[i].Name = "youkaisei_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流T-N";
                c[i].Name = "tn_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流有機性N";
                c[i].Name = "organicn_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流NH4-N";
                c[i].Name = "nh4n_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流NO2-N";
                c[i].Name = "no2n_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流NO3-N";
                c[i].Name = "no3n_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流T-P";
                c[i].Name = "tp_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流オルトリン";
                c[i].Name = "orthop_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "終沈越流塩素イオン";
                c[i].Name = "clion_syuu_etsuryuu";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                for(int j = 1; j <= MaxSizeConfig.maxSyuuchin; j++) {
                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "終沈池" + j.ToString() + "名称";
                    c[i].Name = "name_syuuchin_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "終沈池" + j.ToString() + "運転時間";
                    c[i].Name = "dt_syuuchin_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }

                #endregion


                #region 塩混・放流データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混池容量";
                c[i].Name = "pv_enkon";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].Format = "N0";
                c[i].statical = true;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "min_enkon" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "次亜比重";
                c[i].Name = "gs_jia";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N3";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 2M;
                c[i].Minimum = 0M;
                c[i].statical = true;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "mr_jia" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "次亜有効塩素";
                c[i].Name = "er_jia";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].statical = true;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "mr_jia" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "次亜注入量";
                c[i].Name = "wv_jia";
                c[i].unit = "[L]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "mr_jia" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "次亜注入率";
                c[i].Name = "mr_jia";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "次亜比重*次亜有効塩素*次亜注入量*10/放流水量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混接触時間";
                c[i].Name = "min_enkon";
                c[i].unit = "[min]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "塩混池容量/(放流水量-簡易処理水量)*24*60";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流水量";
                c[i].Name = "wv_houryuu";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "min_enkon", "mr_jia" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "簡易処理水量";
                c[i].Name = "wv_kani";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ar_hannoutank_all", "bodpmlss_hannoutank_@tN", "bodpmlss_hannoutank_all", "h_syuuchin", "min_enkon", "or_hensou", "st_hannoutank_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口外観";
                c[i].Name = "gaikan_enkondeguchi";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口臭気";
                c[i].Name = "syuuki_enkondeguchi";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口透視度";
                c[i].Name = "pr_enkondeguchi";
                c[i].unit = "[cm]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].validMax = 100M;
                c[i].validMin = 50M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口pH";
                c[i].Name = "ph_enkondeguchi";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                c[i].validMax = 7M;
                c[i].validMin = 6.2M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口COD";
                c[i].Name = "cod_enkondeguchi";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "cod_jokyo_sougou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口BOD";
                c[i].Name = "bod_enkondeguchi";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "bod_jokyo_sougou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口C-BOD";
                c[i].Name = "cbod_enkondeguchi";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口SS";
                c[i].Name = "ss_enkondeguchi";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 40M;
                c[i].validMin = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "srt_hannoutank_all", "ss_jokyo_sougou" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出大腸菌群";
                c[i].Name = "nd_enkondeguchi";
                c[i].unit = "[個/mL]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].validMax = 3000M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出口全塩素";
                c[i].Name = "ac_enkondeguchi";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].validMax = 2.1M;
                c[i].validMin = 0.1M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "塩混出遊離塩素";
                c[i].Name = "fc_enkondeguchi";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].validMax = 2.1M;
                c[i].validMin = 0.1M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口外観";
                c[i].Name = "gaikan_houryuukou";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口臭気";
                c[i].Name = "syuuki_houryuukou";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                c[i].inputType = "TextBox";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口透視度";
                c[i].Name = "pr_houryuukou";
                c[i].unit = "[cm]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].validMax = 100M;
                c[i].validMin = 50M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口pH";
                c[i].Name = "ph_houryuukou";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                c[i].validMax = 7M;
                c[i].validMin = 6.2M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口COD";
                c[i].Name = "cod_houryuukou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口BOD";
                c[i].Name = "bod_houryuukou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口C-BOD";
                c[i].Name = "cbod_houryuukou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 15M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口SS";
                c[i].Name = "ss_houryuukou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].validMax = 40M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口大腸菌群";
                c[i].Name = "nd_houryuukou";
                c[i].unit = "[個/mL]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].validMax = 3000M;
                c[i].validMin = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口全塩素";
                c[i].Name = "ac_houryuukou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].validMax = 2.1M;
                c[i].validMin = 0.1M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "放流口遊離塩素";
                c[i].Name = "fc_houryuukou";
                c[i].unit = "[mg/L]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].validMax = 2.1M;
                c[i].validMin = 0.1M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "総合除去率COD";
                c[i].Name = "cod_jokyo_sougou";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(流入水COD-塩混出口COD)/流入COD*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "総合除去率BOD";
                c[i].Name = "bod_jokyo_sougou";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(流入水BOD-塩混出口BOD)/流入BOD*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "総合除去率SS";
                c[i].Name = "ss_jokyo_sougou";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(流入水SS-塩混出口SS)/流入SS*100";
                columnConfigs.Add(c[i]);

                for(int j = 1; j <= MaxSizeConfig.maxHouryuupump; j++) {
                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "放流ポンプ" + j.ToString() + "名称";
                    c[i].Name = "name_houryuupump_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "放流ポンプ" + j.ToString() + "運転時間";
                    c[i].Name = "dt_houryuupump_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }

                #endregion


                #region 汚泥濃縮データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投入汚泥量重力濃計";
                c[i].Name = "ov_nousyukuki_input_juuryoku";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n投入汚泥量[濃縮設備n形式=重力式]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投入汚泥量他濃計";
                c[i].Name = "ov_nousyukuki_input_ex";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n投入汚泥量[濃縮設備n形式≠重力式]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投入汚泥濃縮設備計";
                c[i].Name = "ov_nousyukuki_input_all";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n投入汚泥量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投汚固形物量重力濃計";
                c[i].Name = "sg_nousyukuki_input_juuryoku";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n投入汚泥量[濃縮設備n形式=重力式]*濃n投入汚泥濃度[濃縮設備n形式=重力式]/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投汚固形物量他濃計";
                c[i].Name = "sg_nousyukuki_input_ex";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n投入汚泥量[濃縮設備n形式≠重力式]*濃n投入汚泥濃度[濃縮設備n形式≠重力式]/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投汚固形物濃縮設備計";
                c[i].Name = "sg_nousyukuki_input_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n投入汚泥量*濃n投入汚泥濃度/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "高分子使用量重力濃計";
                c[i].Name = "ug_gyousyuuzai_nousyukuki_juuryoku";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n高分子凝集剤注入量[濃縮設備n形式=重力式]*濃n高分子剤溶解濃度[濃縮設備n形式=重力式]/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "高分子使用量他濃計";
                c[i].Name = "ug_gyousyuuzai_nousyukuki_ex";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n高分子凝集剤注入量[濃縮設備n形式≠重力式]*濃n高分子剤溶解濃度[濃縮設備n形式≠重力式]/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "高分子使用濃縮設備計";
                c[i].Name = "ug_gyousyuuzai_nousyukuki_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n高分子凝集剤注入量*濃n高分子剤溶解濃度/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "起泡剤使用量重力濃計";
                c[i].Name = "ug_kihoujozai_nousyukuki_juuryoku";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n起泡助剤注入量[濃縮設備n形式=重力式]*濃n起泡助剤溶解濃度[濃縮設備n形式=重力式]/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "起泡剤使用量他濃計";
                c[i].Name = "ug_kihoujozai_nousyukuki_ex";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n起泡助剤注入量[濃縮設備n形式≠重力式]*濃n起泡助剤溶解濃度[濃縮設備n形式≠重力式]/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "起泡剤使用濃縮設備計";
                c[i].Name = "ug_kihoujozai_nousyukuki_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n起泡助剤注入量*濃n起泡助剤溶解濃度/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥発生量重力";
                c[i].Name = "ov_nousyuku_output_juuryoku";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_nousyuku_output_all", "sg_nousyuku_output_all", "sg_nousyuku_output_juuryoku" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥発生量他";
                c[i].Name = "ov_nousyuku_output_ex";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_nousyuku_output_all", "sg_nousyuku_output_all", "sg_nousyuku_output_ex" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥発生量計";
                c[i].Name = "ov_nousyuku_output_all";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "濃縮汚泥発生量重力+濃縮汚泥発生量他";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥濃度重力";
                c[i].Name = "oc_nousyuku_output_juuryoku";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_nousyuku_output_all", "sg_nousyuku_output_juuryoku" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥濃度他";
                c[i].Name = "oc_nousyuku_output_ex";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_nousyuku_output_all", "sg_nousyuku_output_ex" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥固形物量重力";
                c[i].Name = "sg_nousyuku_output_juuryoku";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "濃縮汚泥発生量重力*濃縮汚泥濃度重力/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥固形物量他";
                c[i].Name = "sg_nousyuku_output_ex";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "濃縮汚泥発生量他*濃縮汚泥濃度他/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥固形物量計";
                c[i].Name = "sg_nousyuku_output_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "濃縮汚泥発生量重力*濃縮汚泥濃度重力/100*1000+濃縮汚泥発生量他*濃縮汚泥濃度他/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "運転時間重力濃計";
                c[i].Name = "dt_nousyukuki_juuryoku";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n運転時間[濃縮設備n形式=重力式]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "運転時間他濃計";
                c[i].Name = "dt_nousyukuki_ex";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n運転時間[濃縮設備n形式≠重力式]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "運転時間濃縮設備計";
                c[i].Name = "dt_nousyukuki_all";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ濃n運転時間";
                columnConfigs.Add(c[i]);

                #endregion


                #region 濃縮設備個別データ定義

                currentTab += 10;

                for(int j = 1; j <= MaxSizeConfig.maxNousyukuki; j++) {
                    int putPlaceNo1 = (int)(Math.Floor((j - 1M) / panelNo)) * 10 + currentTab;
                    int putPlaceNo2 = (j - 1) % panelNo + 1;
                    int putPlaceNo = putPlaceNo1 + putPlaceNo2;

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃縮設備" + j.ToString() + "名称";
                    c[i].Name = "name_nousyukuki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃縮設備" + j.ToString() + "形式";
                    c[i].Name = "type_nousyukuki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",重力式,加圧浮上式,常圧浮上式,遠心式,スクリーン式,ベルト式,スクリュー式,造粒式,その他濃縮方式";
                    c[i].statical = true;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "dt_nousyukuki_ex", "dt_nousyukuki_juuryoku", "ov_nousyukuki_input_ex", "ov_nousyukuki_input_juuryoku", "sg_nousyukuki_input_ex", "sg_nousyukuki_input_juuryoku", "ug_gyousyuuzai_nousyukuki_ex", "ug_gyousyuuzai_nousyukuki_juuryoku", "ug_kihoujozai_nousyukuki_ex", "ug_kihoujozai_nousyukuki_juuryoku" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "投入汚泥種別";
                    c[i].Name = "otype_nousyukuki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",初沈,終沈,混合,し尿,その他";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "投入汚泥量";
                    c[i].Name = "ov_nousyukuki_" + j.ToString();
                    c[i].unit = "[m3]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ov_nousyukuki_input_all", "ov_nousyukuki_input_ex", "ov_nousyukuki_input_juuryoku", "sg_nousyukuki_input_all", "sg_nousyukuki_input_ex", "sg_nousyukuki_input_juuryoku", "sg_nousyukuki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "投入汚泥濃度";
                    c[i].Name = "oci_nousyukuki_" + j.ToString();
                    c[i].unit = "[%]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 100M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "sg_nousyukuki_input_all", "sg_nousyukuki_input_ex", "sg_nousyukuki_input_juuryoku", "sg_nousyukuki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "投入汚泥固形物量";
                    c[i].Name = "sg_nousyukuki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "濃" + j.ToString() + "投入汚泥量*濃" + j.ToString() + "投入汚泥濃度/100*1000";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "高分子凝集剤注入量";
                    c[i].Name = "v_gyousyuuzai_nousyukuki_" + j.ToString();
                    c[i].unit = "[L]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ug_gyousyuuzai_nousyukuki_all", "ug_gyousyuuzai_nousyukuki_ex", "ug_gyousyuuzai_nousyukuki_juuryoku", "ug_gyousyuuzai_nousyukuki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "高分子剤溶解濃度";
                    c[i].Name = "oc_gyousyuuzai_nousyukuki_" + j.ToString();
                    c[i].unit = "[%]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 100M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ug_gyousyuuzai_nousyukuki_all", "ug_gyousyuuzai_nousyukuki_ex", "ug_gyousyuuzai_nousyukuki_juuryoku", "ug_gyousyuuzai_nousyukuki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "高分子剤使用量";
                    c[i].Name = "ug_gyousyuuzai_nousyukuki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "濃" + j.ToString() + "高分子凝集剤注入量*濃" + j.ToString() + "高分子剤溶解濃度/100";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "起泡助剤注入量";
                    c[i].Name = "v_kihoujozai_nousyukuki_" + j.ToString();
                    c[i].unit = "[L]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ug_kihoujozai_nousyukuki_all", "ug_kihoujozai_nousyukuki_ex", "ug_kihoujozai_nousyukuki_juuryoku", "ug_kihoujozai_nousyukuki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "起泡助剤溶解濃度";
                    c[i].Name = "oc_kihoujozai_nousyukuki_" + j.ToString();
                    c[i].unit = "[%]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 100M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ug_kihoujozai_nousyukuki_all", "ug_kihoujozai_nousyukuki_ex", "ug_kihoujozai_nousyukuki_juuryoku", "ug_kihoujozai_nousyukuki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "起泡助剤使用量";
                    c[i].Name = "ug_kihoujozai_nousyukuki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "濃" + j.ToString() + "起泡助剤注入量*濃" + j.ToString() + "起泡助剤溶解濃度/100";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "他薬品1";
                    c[i].Name = "type_exc1_nousyukuki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",塩化第二鉄,ポリ硫酸第二鉄,消石灰,生石灰,ポリ塩化アルミニウム,過酸化水素,その他";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "他薬品1使用量";
                    c[i].Name = "ug_exc1_nousyukuki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999.99M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "他薬品2";
                    c[i].Name = "type_exc2_nousyukuki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",塩化第二鉄,ポリ硫酸第二鉄,消石灰,生石灰,ポリ塩化アルミニウム,過酸化水素,その他";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "他薬品2使用量";
                    c[i].Name = "ug_exc2_nousyukuki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999.99M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "濃" + j.ToString() + "運転時間";
                    c[i].Name = "dt_nousyukuki_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "dt_nousyukuki_all", "dt_nousyukuki_ex", "dt_nousyukuki_juuryoku" };
                    columnConfigs.Add(c[i]);
                }

                currentTab = (int)(Math.Floor((MaxSizeConfig.maxNousyukuki - 1M) / panelNo)) * 10 + currentTab;

                #endregion


                #region 汚泥消化データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化槽容量";
                c[i].Name = "pv_syouka";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999999M;
                c[i].Minimum = 0M;
                c[i].Format = "N0";
                c[i].statical = true;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "d_syouka" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化槽投入汚泥種別";
                c[i].Name = "otype_syouka_input";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                c[i].inputType = "ComboBox";
                c[i].Items = @",生汚泥,濃縮汚泥,し尿,その他";
                c[i].statical = true;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化槽投入汚泥量";
                c[i].Name = "ov_syouka_input";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "d_syouka", "gr_syoukagas", "sg_syouka_input" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化槽投汚濃度";
                c[i].Name = "oc_syouka_input";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_syouka_input" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化槽投汚固形物量";
                c[i].Name = "sg_syouka_input";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "消化槽投入汚泥量*消化槽投汚濃度/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化槽投汚有機分";
                c[i].Name = "oy_syouka_input";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "r_syouka" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化槽投汚pH";
                c[i].Name = "ph_syouka_input";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化日数";
                c[i].Name = "d_syouka";
                c[i].unit = "[d]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "消化槽容量/消化槽投入汚泥量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化汚泥発生量";
                c[i].Name = "ov_syouka_output";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_syouka_output" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化汚泥濃度";
                c[i].Name = "oc_syouka_output";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_syouka_output" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化汚泥固形物量";
                c[i].Name = "sg_syouka_output";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "消化汚泥発生量*消化汚泥濃度/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化汚泥有機分";
                c[i].Name = "oy_syouka_output";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "r_syouka" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化汚泥pH";
                c[i].Name = "ph_syouka_output";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 14M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化率";
                c[i].Name = "r_syouka";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(1-(100-消化槽投汚有機分)*消化汚泥有機分/(消化槽投汚有機分*(100-消化汚泥有機分)))*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化ガス発生量";
                c[i].Name = "v_syoukag_hassei";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "gr_syoukagas" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用消化G→槽加温";
                c[i].Name = "v_syoukag_soukaon";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用消化G→余G燃焼";
                c[i].Name = "v_syoukag_yogas";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用消化G→売電発電";
                c[i].Name = "v_syoukag_baiden";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用消化G→管理部門";
                c[i].Name = "v_syoukag_kanri";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用消化G→自家発電";
                c[i].Name = "v_syoukag_jikahatsu";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "使用消化G→その他";
                c[i].Name = "v_syoukag_sonota";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "ガス発生倍率";
                c[i].Name = "gr_syoukagas";
                c[i].unit = "";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "消化ガス発生量/消化槽投入汚泥量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化ガスメタン";
                c[i].Name = "r_syoukagas_ch3";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化ガス二酸化炭素";
                c[i].Name = "r_syoukagas_co2";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱硫剤";
                c[i].Name = "m_datsuryuu";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 999999M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                for(int j = 1; j <= MaxSizeConfig.maxSyoukatank; j++) {
                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "消化槽" + j.ToString() + "名称";
                    c[i].Name = "name_syoukatank_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "消化槽" + j.ToString() + "温度";
                    c[i].Name = "tt_syoukatank_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 100M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "消化槽" + j.ToString() + "運転時間";
                    c[i].Name = "dt_syoukatank_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "dt_syoukatank_all" };
                    columnConfigs.Add(c[i]);
                }

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "運転時間消化槽計";
                c[i].Name = "dt_syoukatank_all";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ消化槽n運転時間";
                columnConfigs.Add(c[i]);

                #endregion


                #region 汚泥脱水データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "現有汚泥脱水能力";
                c[i].Name = "wv_genyuuodeidassui";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].statical = true;
                c[i].limit = true;
                c[i].Maximum = 99999.9M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投入汚泥量遠心脱計";
                c[i].Name = "ov_dassuiki_input_enshin";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n投入汚泥量[脱水設備n形式=遠心分離]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投入汚泥量BP脱計";
                c[i].Name = "ov_dassuiki_input_bp";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n投入汚泥量[脱水設備n形式=ベルトプレス]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投入汚泥量他脱計";
                c[i].Name = "ov_dassuiki_input_ex";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n投入汚泥量[脱水設備n形式≠遠心分離 ∧ 脱水設備n形式≠ベルトプレス]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投入汚泥脱水設備計";
                c[i].Name = "ov_dassuiki_input_all";
                c[i].unit = "[m3]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n投入汚泥量";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投汚固形物量遠心脱計";
                c[i].Name = "sg_dassuiki_input_enshin";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n投入汚泥量[脱水設備n形式=遠心分離]*脱n投入汚泥濃度[脱水設備n形式=遠心分離]/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投汚固形物量BP脱計";
                c[i].Name = "sg_dassuiki_input_bp";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n投入汚泥量[脱水設備n形式=ベルトプレス]*脱n投入汚泥濃度[脱水設備n形式=ベルトプレス]/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投汚固形物量他脱計";
                c[i].Name = "sg_dassuiki_input_ex";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n投入汚泥量[脱水設備n形式≠遠心分離 ∧ 脱水設備n形式≠ベルトプレス]*脱n投入汚泥濃度[脱水設備n形式≠遠心分離 ∧ 脱水設備n形式≠ベルトプレス]/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "投汚固形物脱水設備計";
                c[i].Name = "sg_dassuiki_input_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n投入汚泥量*脱n投入汚泥濃度/100*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "高分子使用量遠心脱計";
                c[i].Name = "ug_gyousyuuzai_dassuiki_enshin";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n高分子凝集剤注入量[脱水設備n形式=遠心分離]*脱n高分子剤溶解濃度[脱水設備n形式=遠心分離]/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "高分子使用量BP脱計";
                c[i].Name = "ug_gyousyuuzai_dassuiki_bp";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n高分子凝集剤注入量[脱水設備n形式=ベルトプレス]*脱n高分子剤溶解濃度[脱水設備n形式=ベルトプレス]/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "高分子使用量他脱計";
                c[i].Name = "ug_gyousyuuzai_dassuiki_ex";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n高分子凝集剤注入量[脱水設備n形式≠遠心分離 ∧ 脱水設備n形式≠ベルトプレス]*脱n高分子剤溶解濃度[脱水設備n形式≠遠心分離 ∧ 脱水設備n形式≠ベルトプレス]/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "高分子使用脱水設備計";
                c[i].Name = "ug_gyousyuuzai_dassuiki_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 2;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n高分子凝集剤注入量*脱n高分子剤溶解濃度/100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥発生量遠心";
                c[i].Name = "ov_dassui_output_enshin";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_dassui_output_all", "sg_dassui_output_all", "sg_dassui_output_enshin", "sgcr_dassui_output_all", "sgcr_dassui_output_enshin" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥発生量BP";
                c[i].Name = "ov_dassui_output_bp";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_dassui_output_all", "sg_dassui_output_all", "sg_dassui_output_bp", "sgcr_dassui_output_all", "sgcr_dassui_output_bp" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥発生量他";
                c[i].Name = "ov_dassui_output_ex";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_dassui_output_all", "sg_dassui_output_all", "sg_dassui_output_ex", "sgcr_dassui_output_all", "sgcr_dassui_output_ex" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥発生量計";
                c[i].Name = "ov_dassui_output_all";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "脱水汚泥発生量遠心+脱水汚泥発生量BP+脱水汚泥発生量他";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥含水率遠心";
                c[i].Name = "owr_dassui_output_enshin";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_dassui_output_all", "sg_dassui_output_enshin", "sgcr_dassui_output_all", "sgcr_dassui_output_enshin" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥含水率BP";
                c[i].Name = "owr_dassui_output_bp";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_dassui_output_all", "sg_dassui_output_bp", "sgcr_dassui_output_all", "sgcr_dassui_output_bp" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥含水率他";
                c[i].Name = "owr_dassui_output_ex";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "sg_dassui_output_all", "sg_dassui_output_ex", "sgcr_dassui_output_all", "sgcr_dassui_output_ex" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥固形物量遠心";
                c[i].Name = "sg_dassui_output_enshin";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(100-脱水汚泥含水率遠心)/100*脱水汚泥発生量遠心*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥固形物量BP";
                c[i].Name = "sg_dassui_output_bp";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(100-脱水汚泥含水率BP)/100*脱水汚泥発生量BP*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥固形物量他";
                c[i].Name = "sg_dassui_output_ex";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(100-脱水汚泥含水率他)/100*脱水汚泥発生量他*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥固形物量計";
                c[i].Name = "sg_dassui_output_all";
                c[i].unit = "[kg]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N0";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "(100-脱水汚泥含水率遠心)/100*脱水汚泥発生量遠心*1000+(100-脱水汚泥含水率BP)/100*脱水汚泥発生量BP*1000+(100-脱水汚泥含水率他)/100*脱水汚泥発生量他*1000";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱汚固形物回収率遠心";
                c[i].Name = "sgcr_dassui_output_enshin";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "((100-脱水汚泥含水率遠心)/100*脱水汚泥発生量遠心*1000)/(Σ脱n投入汚泥量[脱水設備n形式=遠心分離]*脱n投入汚泥濃度[脱水設備n形式=遠心分離]/100*1000)*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱汚固形物回収率BP";
                c[i].Name = "sgcr_dassui_output_bp";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "((100-脱水汚泥含水率BP)/100*脱水汚泥発生量BP*1000)/(Σ脱n投入汚泥量[脱水設備n形式=ベルトプレス]*脱n投入汚泥濃度[脱水設備n形式=ベルトプレス]/100*1000)*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱汚固形物回収率他";
                c[i].Name = "sgcr_dassui_output_ex";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "((100-脱水汚泥含水率他)/100*脱水汚泥発生量他*1000)/(Σ脱n投入汚泥量[脱水設備n形式≠遠心分離 ∧ 脱水設備n形式≠ベルトプレス]*脱n投入汚泥濃度[脱水設備n形式≠遠心分離 ∧ 脱水設備n形式≠ベルトプレス]/100*1000)*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱汚固形物回収率全体";
                c[i].Name = "sgcr_dassui_output_all";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "((100-脱水汚泥含水率遠心)/100*脱水汚泥発生量遠心*1000+(100-脱水汚泥含水率BP)/100*脱水汚泥発生量BP*1000+(100-脱水汚泥含水率他)/100*脱水汚泥発生量他*1000)/(Σ脱n投入汚泥量*脱n投入汚泥濃度/100*1000)*100";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥有機分遠心";
                c[i].Name = "oy_dassui_output_enshin";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥有機分BP";
                c[i].Name = "oy_dassui_output_bp";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥有機分他";
                c[i].Name = "oy_dassui_output_ex";
                c[i].unit = "[%]";
                c[i].putPlaceNo = currentTab + 3;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 100M;
                c[i].Minimum = 0M;
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "運転時間遠心脱計";
                c[i].Name = "dt_dassuiki_enshin";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n運転時間[脱水設備n形式=遠心分離]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "運転時間BP脱計";
                c[i].Name = "dt_dassuiki_bp";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n運転時間[脱水設備n形式=ベルトプレス]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "運転時間他脱計";
                c[i].Name = "dt_dassuiki_ex";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n運転時間[脱水設備n形式≠遠心分離 ∧ 脱水設備n形式≠ベルトプレス]";
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "運転時間脱水設備計";
                c[i].Name = "dt_dassuiki_all";
                c[i].unit = "[h]";
                c[i].putPlaceNo = currentTab + 4;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N1";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "Σ脱n運転時間";
                columnConfigs.Add(c[i]);

                #endregion


                #region 脱水設備個別データ定義

                currentTab += 10;

                for(int j = 1; j <= MaxSizeConfig.maxDassuiki; j++) {
                    int putPlaceNo1 = (int)(Math.Floor((j - 1M) / panelNo)) * 10 + currentTab;
                    int putPlaceNo2 = (j - 1) % panelNo + 1;
                    int putPlaceNo = putPlaceNo1 + putPlaceNo2;

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱水設備" + j.ToString() + "名称";
                    c[i].Name = "name_dassuiki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱水設備" + j.ToString() + "形式";
                    c[i].Name = "type_dassuiki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",真空ろ過,加圧ろ過,遠心分離,ベルトプレス,スクリュープレス,ロータリープレス,天日,多重板型スクリュープレス,多重円板型,その他脱水方式";
                    c[i].statical = true;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "dt_dassuiki_bp", "dt_dassuiki_enshin", "dt_dassuiki_ex", "ov_dassuiki_input_bp", "ov_dassuiki_input_enshin", "ov_dassuiki_input_ex", "sg_dassuiki_input_bp", "sg_dassuiki_input_enshin", "sg_dassuiki_input_ex", "sgcr_dassui_output_all", "sgcr_dassui_output_bp", "sgcr_dassui_output_enshin", "sgcr_dassui_output_ex", "ug_gyousyuuzai_dassuiki_bp", "ug_gyousyuuzai_dassuiki_enshin", "ug_gyousyuuzai_dassuiki_ex" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "投入汚泥種別";
                    c[i].Name = "otype_dassuiki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",生汚泥,濃縮汚泥,消化汚泥,洗浄汚泥,熱処理汚泥,湿式酸化汚泥,その他汚泥";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "投入汚泥量";
                    c[i].Name = "ov_dassuiki_" + j.ToString();
                    c[i].unit = "[m3]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ov_dassuiki_input_all", "ov_dassuiki_input_bp", "ov_dassuiki_input_enshin", "ov_dassuiki_input_ex", "sg_dassuiki_input_all", "sg_dassuiki_input_bp", "sg_dassuiki_input_enshin", "sg_dassuiki_input_ex", "sgcr_dassui_output_all", "sgcr_dassui_output_bp", "sgcr_dassui_output_enshin", "sgcr_dassui_output_ex", "r_gyousyuuzai_dassuiki_" + j.ToString(), "sg_dassuiki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "投入汚泥濃度";
                    c[i].Name = "oci_dassuiki_" + j.ToString();
                    c[i].unit = "[%]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 100M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "sg_dassuiki_input_all", "sg_dassuiki_input_bp", "sg_dassuiki_input_enshin", "sg_dassuiki_input_ex", "sgcr_dassui_output_all", "sgcr_dassui_output_bp", "sgcr_dassui_output_enshin", "sgcr_dassui_output_ex", "r_gyousyuuzai_dassuiki_" + j.ToString(), "sg_dassuiki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "投入汚泥固形物量";
                    c[i].Name = "sg_dassuiki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "脱" + j.ToString() + "投入汚泥量*脱" + j.ToString() + "投入汚泥濃度/100*1000";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "高分子凝集剤注入量";
                    c[i].Name = "v_gyousyuuzai_dassuiki_" + j.ToString();
                    c[i].unit = "[L]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ug_gyousyuuzai_dassuiki_all", "ug_gyousyuuzai_dassuiki_bp", "ug_gyousyuuzai_dassuiki_enshin", "ug_gyousyuuzai_dassuiki_ex", "r_gyousyuuzai_dassuiki_" + j.ToString(), "ug_gyousyuuzai_dassuiki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "高分子剤溶解濃度";
                    c[i].Name = "oc_gyousyuuzai_dassuiki_" + j.ToString();
                    c[i].unit = "[%]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 100M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "ug_gyousyuuzai_dassuiki_all", "ug_gyousyuuzai_dassuiki_bp", "ug_gyousyuuzai_dassuiki_enshin", "ug_gyousyuuzai_dassuiki_ex", "r_gyousyuuzai_dassuiki_" + j.ToString(), "ug_gyousyuuzai_dassuiki_" + j.ToString() };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "高分子剤使用量";
                    c[i].Name = "ug_gyousyuuzai_dassuiki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "脱" + j.ToString() + "高分子凝集剤注入量*脱" + j.ToString() + "高分子剤溶解濃度/100";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "高分子剤注入率";
                    c[i].Name = "r_gyousyuuzai_dassuiki_" + j.ToString();
                    c[i].unit = "[%]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].calculatedDecimal = true;
                    c[i].calculateFormula = "(脱" + j.ToString() + "高分子凝集剤注入量*脱" + j.ToString() + "高分子剤溶解濃度/100)/(脱" + j.ToString() + "投入汚泥量*脱" + j.ToString() + "投入汚泥濃度/100*1000)*100";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "他薬品1";
                    c[i].Name = "type_exc1_dassuiki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",塩化第二鉄,ポリ硫酸第二鉄,消石灰,生石灰,ポリ塩化アルミニウム,過酸化水素,その他";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "他薬品1使用量";
                    c[i].Name = "ug_exc1_dassuiki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999.99M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "他薬品2";
                    c[i].Name = "type_exc2_dassuiki_" + j.ToString();
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",塩化第二鉄,ポリ硫酸第二鉄,消石灰,生石灰,ポリ塩化アルミニウム,過酸化水素,その他";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "他薬品2使用量";
                    c[i].Name = "ug_exc2_dassuiki_" + j.ToString();
                    c[i].unit = "[kg]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N2";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 999999.99M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "運転時間";
                    c[i].Name = "dt_dassuiki_" + j.ToString();
                    c[i].unit = "[h]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 24M;
                    c[i].Minimum = 0M;
                    c[i].calculateSource = true;
                    c[i].forCalc = new string[] { "dt_dassuiki_all", "dt_dassuiki_bp", "dt_dassuiki_enshin", "dt_dassuiki_ex" };
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "ろ布速度";
                    c[i].Name = "rofusp_dassuiki_" + j.ToString();
                    c[i].unit = "[m/min]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "脱" + j.ToString() + "ろ過速度";
                    c[i].Name = "rokasp_dassuiki_" + j.ToString();
                    c[i].unit = "[kg/m/h]";
                    c[i].putPlaceNo = putPlaceNo;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }

                currentTab = (int)(Math.Floor((MaxSizeConfig.maxDassuiki - 1M) / panelNo)) * 10 + currentTab;

                #endregion


                #region 汚泥等搬出データ定義

                currentTab += 10;

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "沈砂搬出量";
                c[i].Name = "ov_hansyutsu_chinsa";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_hansyutsu_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "し渣搬出量";
                c[i].Name = "ov_hansyutsu_shisa";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_hansyutsu_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "生汚泥搬出量";
                c[i].Name = "ov_hansyutsu_namaodei";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_hansyutsu_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "余剰汚泥搬出量";
                c[i].Name = "ov_hansyutsu_yojouodei";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_hansyutsu_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "濃縮汚泥搬出量";
                c[i].Name = "ov_hansyutsu_nousyukuodei";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_hansyutsu_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "消化汚泥搬出量";
                c[i].Name = "ov_hansyutsu_syoukaodei";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_hansyutsu_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "脱水汚泥搬出量";
                c[i].Name = "ov_hansyutsu_dassuiodei";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].inputType = "NumericUpDown";
                c[i].limit = true;
                c[i].Maximum = 9999.99M;
                c[i].Minimum = 0M;
                c[i].calculateSource = true;
                c[i].forCalc = new string[] { "ov_hansyutsu_all" };
                columnConfigs.Add(c[i]);

                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "汚泥等搬出量計";
                c[i].Name = "ov_hansyutsu_all";
                c[i].unit = "[t]";
                c[i].putPlaceNo = currentTab + 1;
                c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                c[i].Format = "N2";
                c[i].calculatedDecimal = true;
                c[i].calculateFormula = "沈砂搬出量+し渣搬出量+生汚泥搬出量+余剰汚泥搬出量+濃縮汚泥搬出量+消化汚泥搬出量+脱水汚泥搬出量";
                columnConfigs.Add(c[i]);

                #endregion


                #region 別途分析

                for(int j = 1; j <= MaxSizeConfig.maxSpecialAnalysis; j++) {
                    currentTab += 10;

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "箇所";
                    c[i].Name = "name_exbunseki_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    c[i].statical = true;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "採取時刻(時)";
                    c[i].Name = "h_exbunseki_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 23M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "採取時刻(分)";
                    c[i].Name = "m_exbunseki_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 59M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "水温";
                    c[i].Name = "wt_exbunseki_" + j.ToString();
                    c[i].unit = "[℃]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99.9M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "気温";
                    c[i].Name = "kion_exbunseki_" + j.ToString();
                    c[i].unit = "[℃]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99.9M;
                    c[i].Minimum = -99.9M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "天候";
                    c[i].Name = "tenkou_exbunseki_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                    c[i].inputType = "ComboBox";
                    c[i].Items = @",晴,曇,雨,雪";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "外観";
                    c[i].Name = "gaikan_exbunseki_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "臭気";
                    c[i].Name = "syuuki_exbunseki_" + j.ToString();
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleLeft;
                    c[i].inputType = "TextBox";
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "透視度";
                    c[i].Name = "pr_exbunseki_" + j.ToString();
                    c[i].unit = "[cm]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 100M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "pH温度";
                    c[i].Name = "phwt_exbunseki_" + j.ToString();
                    c[i].unit = "[℃]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99.9M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "pH";
                    c[i].Name = "ph_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 14M;
                    c[i].Minimum = 0M;
                    c[i].validMax = 7M;
                    c[i].validMin = 6.2M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "SS";
                    c[i].Name = "ss_exbunseki_" + j.ToString();
                    c[i].unit = "[mg/L]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "COD";
                    c[i].Name = "cod_exbunseki_" + j.ToString();
                    c[i].unit = "[mg/L]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "BOD";
                    c[i].Name = "bod_exbunseki_" + j.ToString();
                    c[i].unit = "[mg/L]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "C-BOD";
                    c[i].Name = "cbod_exbunseki_" + j.ToString();
                    c[i].unit = "[mg/L]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N1";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "大腸菌群";
                    c[i].Name = "nd_exbunseki_" + j.ToString();
                    c[i].unit = "[個/mL]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 9999999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "クリプトスポリジウム";
                    c[i].Name = "cryptosporidium_exbunseki_" + j.ToString();
                    c[i].unit = "[個/L]";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 9999999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "臭気指数";
                    c[i].Name = "syuukiidx_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 1;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N0";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 9999999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "n-ヘキサン抽出物質";
                    c[i].Name = "nhexane_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "クロム含有量";
                    c[i].Name = "cr_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "六価クロム化合物";
                    c[i].Name = "hcr_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "亜鉛含有量";
                    c[i].Name = "zn_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "鉛及び化合物";
                    c[i].Name = "pb_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "カドミウム及び化合物";
                    c[i].Name = "cd_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "砒素及び化合物";
                    c[i].Name = "as_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "総水銀";
                    c[i].Name = "hg_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "アルキル水銀化合物";
                    c[i].Name = "alkylhg_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "有機燐化合物";
                    c[i].Name = "organicp_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "シアン化合物";
                    c[i].Name = "cyan_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "ふっ素及び化合物";
                    c[i].Name = "f_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "銅含有量";
                    c[i].Name = "cu_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "溶解性マンガン含有量";
                    c[i].Name = "ssmn_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "溶解性鉄含有量";
                    c[i].Name = "ssfe_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "ポリ塩化ビフェニル";
                    c[i].Name = "pcb_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "フェノール類含有量";
                    c[i].Name = "phenol_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 2;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "アンモニア性窒素";
                    c[i].Name = "nh4n_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "亜硝酸性窒素";
                    c[i].Name = "no2n_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "硝酸性窒素";
                    c[i].Name = "no3n_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "窒素含有量";
                    c[i].Name = "tn_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "燐含有量";
                    c[i].Name = "tp_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "トリクロロエチレン";
                    c[i].Name = "trice_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "テトラクロロエチレン";
                    c[i].Name = "tetce_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "ジクロロメタン";
                    c[i].Name = "dcm_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "四塩化炭素";
                    c[i].Name = "ccl4_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "1,2-ジクロロエタン";
                    c[i].Name = "x12dce_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "1,1-ジクロロエチレン";
                    c[i].Name = "x11dce_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "シス1,2-ジクロロエチレン";
                    c[i].Name = "cis12dce_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "1,1,1-トリクロロエタン";
                    c[i].Name = "x111tce_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "1,1,2-トリクロロエタン";
                    c[i].Name = "x112tce_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "1,3-ジクロロプロペン";
                    c[i].Name = "x13dcp_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "チウラム";
                    c[i].Name = "thiuram_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "シマジン";
                    c[i].Name = "simazine_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 3;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "チオベンカルブ";
                    c[i].Name = "thiobencarb_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "ベンゼン";
                    c[i].Name = "benzene_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "セレン及びその化合物";
                    c[i].Name = "se_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "アンモニア他化合物";
                    c[i].Name = "nh3_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "ほう素及びその化合物";
                    c[i].Name = "b_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "ニッケル";
                    c[i].Name = "ni_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "1,4-ジオキサン";
                    c[i].Name = "x14dox_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);

                    i++;
                    c[i] = new ColumnConfig();
                    c[i].index = i;
                    c[i].HeaderText = "別分" + j.ToString() + "LAS";
                    c[i].Name = "las_exbunseki_" + j.ToString();
                    c[i].unit = "";
                    c[i].putPlaceNo = currentTab + 4;
                    c[i].Alignment = DataGridViewContentAlignment.MiddleRight;
                    c[i].Format = "N4";
                    c[i].inputType = "NumericUpDown";
                    c[i].limit = true;
                    c[i].Maximum = 99999.9999M;
                    c[i].Minimum = 0M;
                    columnConfigs.Add(c[i]);
                }


                #endregion


                i++;
                c[i] = new ColumnConfig();
                c[i].index = i;
                c[i].HeaderText = "入力(インポート)日時";
                c[i].Name = "input_dt";
                c[i].Width = 146;
                c[i].Alignment = DataGridViewContentAlignment.MiddleCenter;
                c[i].Format = "yyyy/MM/dd ddd HH:mm:ss";
                columnConfigs.Add(c[i]);

                return columnConfigs;
            }
        }




        //各putPlaceに最初に配置される列のindexを取得
        public static List<int> tabTopColumnNumber {
            get {
                List<int> tabTopCN = new List<int>();

                int cn = 0;
                for(int i = 1; i <= InputTab.title.Count; i++) {
                    for(int j = cn; j < getSet.Count; j++) {
                        if(getSet[j].putPlaceNo == i * 10 + 1) {
                            tabTopCN.Add(j);
                            cn = j;
                            break;
                        }
                    }
                }

                return tabTopCN;
            }
        }




        //二次データの列のみ取得
        public static List<ColumnConfig> getCalculateSet {
            get {
                List<ColumnConfig> columnConfigs = new List<ColumnConfig>(getSet);

                columnConfigs.RemoveAll(x => !x.calculatedDecimal && !x.calculatedString);

                return columnConfigs;
            }
        }




        //ヘッダーテキストからインデックスを取得
        public static int headerTextToIndex(string HeaderText) {
            foreach(ColumnConfig cc in getSet) {
                if(cc.HeaderText == HeaderText) {
                    return cc.index;
                }
            }

            return -1;
        }




        //ヘッダーテキスト＋単位からインデックスを取得
        public static int headerTextAndUnitToIndex(string HeaderTextAndUnit) {
            HeaderTextAndUnit = HeaderTextAndUnit.Replace("\r", "");
            HeaderTextAndUnit = HeaderTextAndUnit.Replace("\n", "");

            foreach(ColumnConfig cc in getSet) {
                if(cc.HeaderText + cc.unit == HeaderTextAndUnit) {
                    return cc.index;
                }
            }

            return -1;
        }




        //データ名からインデックスを取得
        public static int nameToIndex(string Name) {
            foreach(ColumnConfig cc in getSet) {
                if(cc.Name == Name) {
                    return cc.index;
                }
            }

            return -1;
        }




        //データ名からフォーマットを取得
        public static string nameToFormat(string Name) {
            foreach(ColumnConfig cc in getSet) {
                if(cc.Name == Name) {
                    return cc.Format;
                }
            }

            return null;
        }




        //データ名からputPlaceNoを取得
        public static int nameToPutPlaceNo(string Name) {
            foreach(ColumnConfig cc in getSet) {
                if(cc.Name == Name) {
                    return cc.putPlaceNo;
                }
            }

            return -1;
        }




        //データ名を基に丸め処理を行う
        public static decimal roundingByName(decimal value, string Name) {
            foreach(ColumnConfig cc in getSet) {
                if(cc.Name == Name) {
                    if(cc.Format == "N0") {
                        return Math.Round(value, 0, MidpointRounding.AwayFromZero);
                    }
                    else if(cc.Format == "N1") {
                        return Math.Round(value, 1, MidpointRounding.AwayFromZero);
                    }
                    else if(cc.Format == "N2") {
                        return Math.Round(value, 2, MidpointRounding.AwayFromZero);
                    }
                    else if(cc.Format == "N3") {
                        return Math.Round(value, 3, MidpointRounding.AwayFromZero);
                    }
                    else if(cc.Format == "N4") {
                        return Math.Round(value, 4, MidpointRounding.AwayFromZero);
                    }
                    else if(cc.Format == "N5") {
                        return Math.Round(value, 5, MidpointRounding.AwayFromZero);
                    }
                    else {
                        return value;
                    }
                }
            }

            return value;
        }

    }




    /// <summary>
    /// データベースとdataGridViewの列の設定のクラス
    /// </summary>
    public class SeitenConfig {
        public string kanrisya { get; set; }
        public string syorijou { get; set; }
        public decimal? utenLLUryouDay0 { get; set; }
        public decimal? utenLLUryouDay1 { get; set; }
        public decimal? utenLLUryouDay2 { get; set; }
    }




    /// <summary>
    /// 晴天日判定用の雨量の条件を格納するクラス
    /// </summary>
    public static class SeitenConfigSet {
        public static List<SeitenConfig> getSet { get; set; } = new List<SeitenConfig>();

        //リストを初期化しデータベースから設定を読み込む
        public static void presetSeitenConfigSet() {
            getSet.Clear();

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    cmd.CommandText = "SELECT kanrisya, syorijou, uten_ll_uryou_day0, uten_ll_uryou_day1, uten_ll_uryou_day2 FROM uten_judge_uryou_setting;";

                    using(var reader = cmd.ExecuteReader()) {
                        string ccFormat = ColumnsConfigSet.nameToFormat("uryou");

                        while(reader.Read()) {
                            SeitenConfig sc = new SeitenConfig();

                            sc.kanrisya = reader.GetString(0);
                            sc.syorijou = reader.GetString(1);

                            if(!reader.IsDBNull(2)) {
                                sc.utenLLUryouDay0 = FunctionForDatabase.decConvFromDB(reader.GetDecimal(2), ccFormat);
                            }

                            if(!reader.IsDBNull(3)) {
                                sc.utenLLUryouDay1 = FunctionForDatabase.decConvFromDB(reader.GetDecimal(3), ccFormat);
                            }

                            if(!reader.IsDBNull(4)) {
                                sc.utenLLUryouDay2 = FunctionForDatabase.decConvFromDB(reader.GetDecimal(4), ccFormat);
                            }

                            getSet.Add(sc);
                        }
                    }
                }
            }
        }




        //管理者名と処理場からこれ以上を雨天と判定する当日の雨量を取得
        public static decimal? getUtenLLUryouDay0(string kanrisya, string syorijou) {
            if(SeitenConfigSet.getSet != null) {
                List<SeitenConfig> scSet = SeitenConfigSet.getSet;

                foreach(SeitenConfig sc in scSet) {
                    if(sc.kanrisya == kanrisya && sc.syorijou == syorijou) {
                        return sc.utenLLUryouDay0;
                    }
                }
            }

            return 1;
        }




        //管理者名と処理場からこれ以上を雨天と判定する1日前の雨量を取得
        public static decimal? getUtenLLUryouDay1(string kanrisya, string syorijou) {
            if(SeitenConfigSet.getSet != null) {
                List<SeitenConfig> scSet = SeitenConfigSet.getSet;

                foreach(SeitenConfig sc in scSet) {
                    if(sc.kanrisya == kanrisya && sc.syorijou == syorijou) {
                        return sc.utenLLUryouDay1;
                    }
                }

            }

            return 15;
        }




        //管理者名と処理場からこれ以上を雨天と判定する2日前の雨量を取得
        public static decimal? getUtenLLUryouDay2(string kanrisya, string syorijou) {
            if(SeitenConfigSet.getSet != null) {
                List<SeitenConfig> scSet = SeitenConfigSet.getSet;

                foreach(SeitenConfig sc in scSet) {
                    if(sc.kanrisya == kanrisya && sc.syorijou == syorijou) {
                        return sc.utenLLUryouDay2;
                    }
                }
            }

            return 30;
        }




        //管理者名と処理場の設定がなければtrue
        public static bool isNull(string kanrisya, string syorijou) {
            if(SeitenConfigSet.getSet != null) {
                List<SeitenConfig> scSet = SeitenConfigSet.getSet;

                foreach(SeitenConfig sc in scSet) {
                    if(sc.kanrisya == kanrisya && sc.syorijou == syorijou) {
                        return false;
                    }
                }
            }

            return true;
        }

    }




    /// <summary>
    /// 帳票に含めるdataGridViewの列のデフォルト設定を格納するクラス
    /// </summary>
    public static class SettingPrintSet {
        public static List<string[]> getSet {
            get {
                List<string[]> pcs = new List<string[]>();

                string[] printColumn1 = { "ポンプ場運転", "tenkou", "seiten", "wv_pumpjou_all", "ee_pumpjou_all", "v_hao_pumpjou_all", "v_lo_pumpjou_all", "v_chinsa_pumpjou_all", "v_shisa_pumpjou_all" };
                pcs.Add(printColumn1);

                string[] printColumn2 = { "処理場使用エネルギー", "ee_juden", "maxep_juden", "ee_jikahatsu", "ee_jounaipump", "ee_mizusyori", "ee_mizusyori_blower", "ee_odeisyori", "ee_extra", "ee_kakuhanki", "v_hao_kanri", "v_hao_syouka", "v_hao_jikahatsu" };
                pcs.Add(printColumn2);

                string[] printColumn3 = { "処理場水量", "tenkou", "seiten", "wv_ido", "wv_jousui", "wv_ryuunyuu", "wv_syochin", "wv_houryuu", "wv_kani" };
                pcs.Add(printColumn3);

                string[] printColumn4 = { "処理場運用", "pv_syochin", "pv_hannoutank", "pv_syuuchin", "pv_enkon", "dt_nousyukuki_all", "pv_syouka", "dt_syoukatank_all", "dt_dassuiki_all" };
                pcs.Add(printColumn4);

                string[] printColumn5 = { "主ポンプ運転時間", "dt_syupump_1", "name_syupump_2", "dt_syupump_2", "name_syupump_3", "dt_syupump_3", "name_syupump_4", "dt_syupump_4", "name_syupump_5", "dt_syupump_5", "name_syupump_6", "dt_syupump_6", "name_syupump_7", "dt_syupump_7", "name_syupump_8", "dt_syupump_8" };
                pcs.Add(printColumn5);

                string[] printColumn6 = { "初沈総合", "pv_syochin", "wv_syochin", "h_syochin", "ov_nama", "oc_nama", "sg_nama", "ts_nama", "cod_syo_ryuunyuu", "bod_syo_ryuunyuu", "ss_syo_ryuunyuu", "cod_syo_etsuryuu", "bod_syo_etsuryuu", "cbod_syo_etsuryuu", "ss_syo_etsuryuu", "ss_jokyo_syo_etsuryuu" };
                pcs.Add(printColumn6);

                string[] printColumn7 = { "反応タンク総合", "pv_hannoutank", "av_hannoutank_all", "ar_hannoutank_all", "st_hannoutank_all", "srt_hannoutank_all", "wt_hannoutank_all", "ph_hannoutank_all", "sv_hannoutank_all", "mldo_p_hannoutank_all", "mldo_a_hannoutank_all", "mlss_hannoutank_all", "mlvss_hannoutank_all", "vsspss_hannoutank_all", "svi_hannoutank_all", "bodpmlss_hannoutank_all", "orp_ken_hannoutank_all", "orp_kou_hannoutank_all", "rr_zen_hannoutank_all", "rr_kou_hannoutank_all" };
                pcs.Add(printColumn7);

                string[] printColumn8 = { "終沈総合", "pv_syuuchin", "h_syuuchin", "ov_hensou", "or_hensou", "ov_yojou", "oc_yojou", "sg_yojou", "pr_syuu_etsuryuu", "cod_syuu_etsuryuu", "bod_syuu_etsuryuu", "cbod_syuu_etsuryuu", "ss_syuu_etsuryuu" };
                pcs.Add(printColumn8);

                string[] printColumn9 = { "汚泥発生", "v_chinsa_pumpjou_all", "v_shisa_pumpjou_all", "v_chinsa_syorijou", "v_shisa_syorijou", "ov_nama", "oc_nama", "sg_nama", "ov_hensou", "or_hensou", "ov_yojou", "oc_yojou", "sg_yojou" };
                pcs.Add(printColumn9);

                string[] printColumn10 = { "汚泥処理", "ov_nousyukuki_input_all", "ov_nousyuku_output_all", "dt_nousyukuki_all", "ov_syouka_input", "ov_syouka_output", "ov_dassuiki_input_all", "ov_dassui_output_all", "ov_hansyutsu_all" };
                pcs.Add(printColumn10);

                return pcs;
            }
        }

    }

}

