﻿// 下水データ管理システム
// Copyright (C) 2018 公益財団法人とちぎ建設技術センター(https://www.tochigictc.or.jp/)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.If not, see<http://www.gnu.org/licenses/>.

using System;
using System.Collections.Generic;
using System.Data.SQLite;
using System.Windows.Forms;

namespace WindowsFormsApplication1 {
    /// <summary>
    /// 管理者と処理場の組合わせのクラス
    /// </summary>
    public class KanrisyaSyorijou : IEquatable<KanrisyaSyorijou> {
        public string kanrisya { get; set; }
        public string syorijou { get; set; }
        public string selectionName { get { return kanrisya + "：" + syorijou; } }

        public KanrisyaSyorijou(string k, string s) {
            kanrisya = k;
            syorijou = s;
        }

        public string primaryKey(int c) {
            switch(c) {
                case 0:
                    return kanrisya;
                case 1:
                    return syorijou;
                default:
                    return "";
            }
        }

		public bool Equals(KanrisyaSyorijou other) {
			if (this.kanrisya == other.kanrisya && this.syorijou == other.syorijou) {
				return true;
			}
			else {
				return false;
			}
		}
	}




    /// <summary>
    /// 選択可能月のクラス
    /// </summary>
    public class SelectableMonth {
        public int year { get; set; }
        public int month { get; set; }
        public string selectionName {
            get {
                if(year == -1 && month == -1) {
                    return "";
                }
                else if(year == 0 && month == 0) {
                    return "全期間";
                }
                else {
                    return ("0000" + year.ToString()).Substring(year.ToString().Length, 4) + "年" + ("00" + month.ToString()).Substring(month.ToString().Length, 2) + "月";
                }
            }
        }

        public SelectableMonth(int y, int m) {
            year = y;
            month = m;
        }

        public int primaryKey(int c) {
            switch(c) {
                case 0:
                    return year;
                case 1:
                    return month;
                default:
                    return 0;
            }
        }
    }




    /// <summary>
    /// 選択可能年度のクラス
    /// </summary>
    public class SelectableBYear {
        public int bYear { get; set; }
        public string selectionName {
            get {
                if(bYear == -1) {
                    return "";
                }
                else {
                    return ("0000" + bYear.ToString()).Substring(bYear.ToString().Length, 4) + "年度";
                }
            }
        }

        public SelectableBYear(int y) {
            bYear = y;
        }

        public int primaryKey(int c) {
            switch(c) {
                case 0:
                    return bYear;
                default:
                    return 0;
            }
        }
    }




    /// <summary>
    /// 選択可能四半期のクラス
    /// </summary>
    public class SelectableQuarter {
        public int bYear { get; set; }
        public int quarter { get; set; }
        public string selectionName {
            get {
                if(bYear == -1 && quarter == -1) {
                    return "";
                }
                else {
                    return ("0000" + bYear.ToString()).Substring(bYear.ToString().Length, 4) + "年度　第" + quarter.ToString() + "四半期";
                }
            }
        }

        public SelectableQuarter(int y, int q) {
            bYear = y;
            quarter = q;
        }

        public int primaryKey(int c) {
            switch(c) {
                case 0:
                    return bYear;
                case 1:
                    return quarter;
                default:
                    return 0;
            }
        }
    }




    /// <summary>
    /// グラフ化する期間のクラス
    /// </summary>
    public class SelectableSpan {
        public int months { get; set; }
        public string selectionName {
            get {
                if(months < 0) {
                    return "";
                }
                else if(months == 0) {
                    return "　～　";
                }
                else {
                    int year = (int)(months / 12);
                    int month = months % 12;
                    string sYear = "";
                    string sMonth = "";

                    if(year > 0) {
                        sYear = string.Format("{0}年", year);
                    }

                    if(month > 0) {
                        sMonth = string.Format("{0}ヶ月", month);
                    }

                    return (sYear + sMonth);
                }
            }
        }

        public SelectableSpan(int ms) {
            months = ms;
        }
    }




    /// <summary>
    /// グラフの系列のクラス
    /// </summary>
    public class graphSeries {
        public string kanrisya { get; set; }
        public string syorijou { get; set; }
        public int columnsIndex { get; set; } = 0;
        public int scaleNo { get; set; } = 0;

        public void delete() {
            kanrisya = null;
            syorijou = null;
            columnsIndex = 0;
            scaleNo = 0;
        }
    }




    /// <summary>
    /// プリンターの用紙と印刷範囲のクラス
    /// </summary>
    public static class PrintArea {
        // 1/100インチ単位
        public static bool landscape { get; set; }
        public static int paperSizeWSet { get; set; }
        public static int paperSizeHSet { get; set; }
        public static int marginL { get; set; } = 100;
        public static int marginR { get; set; } = 100;
        public static int marginT { get; set; } = 100;
        public static int marginB { get; set; } = 100;
        public static float hardMarginX { get; set; }
        public static float hardMarginY { get; set; }

        public static int paperSizeW {
            get {
                if(landscape) {
                    return paperSizeWSet;
                }
                else {
                    return paperSizeHSet;
                }
            }
        }
        public static int paperSizeH {
            get {
                if(landscape) {
                    return paperSizeHSet;
                }
                else {
                    return paperSizeWSet;
                }
            }
        }

        public static int printAreaW { get { return paperSizeW - marginL - marginR; } }
        public static int printAreaH { get { return paperSizeH - marginT - marginB; } }
        public static float printerMarginL { get { return marginL - (int)hardMarginX; } }
        public static float printerMarginT { get { return marginT - (int)hardMarginY; } }

        //ミリメートル単位
        public static float printAreaWmm { get { return printAreaW * 0.254F; } }
        public static float printAreaHmm { get { return printAreaH * 0.254F; } }
        public static float printerMarginLmm { get { return printerMarginL * 0.254F; } }
        public static float printerMarginTmm { get { return printerMarginT * 0.254F; } }
    }




    /// <summary>
    /// 印刷に使用するデータのコンテナのクラス
    /// </summary>
    public static class PrintContainer {
        public static string printType { get; set; }

        public static DataGridView dataGridView1 { get; set; }

        //帳票印刷
        public static ComboBox comboBox1 { get; set; }
        public static ComboBox comboBox2 { get; set; }
        public static List<int> printColumnsNo { get; set; }
        public static int printWidth { get; set; }

        public static int dataGridView1Width {
            get {
                int allWidth = 0;

                for(int i = 0; i < dataGridView1.ColumnCount; i++) {
                    if(dataGridView1.Columns[i].Visible) {
                        allWidth += dataGridView1.Columns[i].Width;
                    }
                }

                return allWidth;
            }
        }

        //グラフ印刷
        public static DateTime timeSpanSt;
        public static DateTime timeSpanEd;
        public static bool monthAverage = false;
        public static bool colorPrint = true;

        //グラフの系統数(+1)
        public static int graphSeriesNo = 7;
        public static graphSeries[] graphSerieses = new graphSeries[graphSeriesNo];
    }




    /// <summary>
    /// エクセル関係の処理のクラス
    /// </summary>
    public static class FunctionForExcel {
        //列番号からエクセル形式の列表示文字列を返す
        public static string columnNoToLetter(int columnNo) {
            string alphabet = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

            if(columnNo < 1) {
                return "";
            }

            return columnNoToLetter((columnNo - 1) / 26) + alphabet[(columnNo - 1) % 26];
        }
    }




    /// <summary>
    /// データベース関係の処理のクラス
    /// </summary>
    public static class FunctionForDatabase {
        //テーブルsuishitsu_kanri_settingにsuishitsu_kanriの列を行として用意し初期値を設定
        public static void setSuishitsuKanriRow() {
            List<ColumnConfig> ccSet = ColumnsConfigSet.getSet;

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    cmd.CommandText = "";

                    foreach(ColumnConfig cc in ccSet) {
                        if(cc.Name != "id" && cc.Name != "del_flg" && cc.Name != "kanrisya" && cc.Name != "syorijou" && cc.Name != "sokutei_dt") {
                            cmd.CommandText += "INSERT OR IGNORE INTO suishitsu_kanri_setting (column_name, visible, printform1, printform2, printform3, printform4, printform5, printform6, printform7, printform8, printform9, printform10) VALUES('" + cc.Name + "',1 ,0 ,0 ,0 ,0 ,0 ,0 ,0 ,0 ,0 ,0 ); ";
                        }
                    }

                    cmd.ExecuteNonQuery();
                }

                con.Close();
            }
        }




        //テーブルsuishitsu_kanri_settingにデフォルト値を設定
        public static void setDefaultSuishitsuKanri() {
            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var transaction = con.BeginTransaction()) {
                    using(var cmd = con.CreateCommand()) {
                        for(int i = 0; i < SettingPrintSet.getSet.Count; i++) {
                            string[] printColumn = SettingPrintSet.getSet[i];

                            if(printColumn.Length > 1) {
                                cmd.CommandText = "UPDATE suishitsu_kanri_setting SET printform" + (i + 1).ToString() + " = 1 WHERE column_name IN (";

                                int j = 0;
                                foreach(string settingValue in printColumn) {
                                    j++;

                                    if(j == 1) {
                                        continue;
                                    }
                                    else if(j > 2) {
                                        cmd.CommandText += ", ";
                                    }

                                    cmd.CommandText += "'" + settingValue + "'";
                                }

                                cmd.CommandText += ");";
                                cmd.ExecuteNonQuery();
                            }
                        }
                    }

                    transaction.Commit();
                }

                con.Close();
            }
        }




        //テーブルprint_form_settingに初期設定として帳票分の行を用意
        public static void setPrintFormRow() {
            List<ColumnConfig> ccSet = ColumnsConfigSet.getSet;

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    cmd.CommandText = "";

                    for(int i = 0; i < MaxSizeConfig.maxPrintForm; i++) {
                        cmd.CommandText += "INSERT OR IGNORE INTO print_form_setting (form_no, name) VALUES(" + (i + 1) + ", '帳票" + (i + 1).ToString() + "');";
                    }

                    cmd.ExecuteNonQuery();
                }

                con.Close();
            }
        }




        //テーブルprint_form_settingにデフォルト値を設定
        public static void setDefaultPrintForm() {
            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var transaction = con.BeginTransaction()) {
                    using(var cmd = con.CreateCommand()) {
                        for(int i = 0; i < SettingPrintSet.getSet.Count; i++) {
                            string[] printColumn = SettingPrintSet.getSet[i];

                            if(printColumn.Length > 0) {
                                cmd.CommandText = "UPDATE print_form_setting SET name = '" + printColumn[0] + "' WHERE form_no = " + (i + 1).ToString() + ";";
                                cmd.ExecuteNonQuery();
                            }
                        }
                    }

                    transaction.Commit();
                }

                con.Close();
            }
        }



        //テーブル格納時に小数は10のn乗倍し整数で格納する(doubleをDecimalとして利用する)
        public static decimal? decConvForDB(decimal? dec, string format) {
            if(dec == null) {
                return null;
            }
            else {
                if(format == "N1") {
                    return dec * 10;
                }
                else if(format == "N2") {
                    return dec * 100;
                }
                else if(format == "N3") {
                    return dec * 1000;
                }
                else if(format == "N4") {
                    return dec * 10000;
                }
                else if(format == "N5") {
                    return dec * 100000;
                }
                else {
                    return dec;
                }
            }
        }




        //テーブル格納時に小数は10のn乗倍し整数で格納したものを元に戻す(doubleをDecimalとして利用する)
        public static decimal? decConvFromDB(decimal? dec, string format) {
            if(dec == null) {
                return null;
            }
            else {
                if(format == "N1") {
                    return dec / 10;
                }
                else if(format == "N2") {
                    return dec / 100;
                }
                else if(format == "N3") {
                    return dec / 1000;
                }
                else if(format == "N4") {
                    return dec / 10000;
                }
                else if(format == "N5") {
                    return dec / 100000;
                }
                else {
                    return dec;
                }
            }
        }




        //晴天日を判定してデータベースをアップデートする
        public static void judgeSeitenUpdateDB(string kanrisya, string syorijou, DateTime? sokutei_dt = null) {
            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    string sql1 = "UPDATE suishitsu_kanri SET seiten='○' WHERE del_flg IS NULL AND kanrisya = @kanrisya AND syorijou = @syorijou";

                    if(sokutei_dt != null) {
                        sql1 += " AND sokutei_dt IN (@sokutei_dt0, @sokutei_dt1, @sokutei_dt2);";
                    }
                    else {
                        sql1 += ";";
                    }

                    string sql2 = "UPDATE suishitsu_kanri AS s0 SET seiten=null WHERE del_flg IS NULL AND kanrisya = @kanrisya AND syorijou = @syorijou";

                    if(sokutei_dt != null) {
                        sql2 += " AND sokutei_dt IN (@sokutei_dt0, @sokutei_dt1, @sokutei_dt2)";
                    }

                    string sql3 = "";

                    if(SeitenConfigSet.getUtenLLUryouDay0(kanrisya, syorijou) != null) {
                        sql3 += "uryou >= @uten_ll_uryou_day0";
                    }

                    if(SeitenConfigSet.getUtenLLUryouDay1(kanrisya, syorijou) != null) {
                        if(sql3 != "") {
                            sql3 += " OR ";
                        }

                        sql3 += "(SELECT uryou FROM suishitsu_kanri AS s1 WHERE s1.del_flg IS NULL AND s1.kanrisya = @kanrisya AND s1.syorijou = @syorijou AND s1.sokutei_dt = strftime('%Y-%m-%d %H:%M:%S',s0.sokutei_dt, '-1 day')) >= @uten_ll_uryou_day1";
                    }

                    if(SeitenConfigSet.getUtenLLUryouDay2(kanrisya, syorijou) != null) {
                        if(sql3 != "") {
                            sql3 += " OR ";
                        }

                        sql3 += "(SELECT uryou FROM suishitsu_kanri AS s2 WHERE s2.del_flg IS NULL AND s2.kanrisya = @kanrisya AND s2.syorijou = @syorijou AND s2.sokutei_dt = strftime('%Y-%m-%d %H:%M:%S',s0.sokutei_dt, '-2 day')) >= @uten_ll_uryou_day2";
                    }

                    if(sql3 != "") {
                        sql3 = " AND (" + sql3 + ")";
                    }

                    string sql4 = ";";

                    cmd.CommandText = sql1 + sql2 + sql3 + sql4;

                    cmd.Parameters.Add(new SQLiteParameter("@kanrisya", kanrisya));
                    cmd.Parameters.Add(new SQLiteParameter("@syorijou", syorijou));

                    if(sokutei_dt != null) {
                        DateTime cDate = (DateTime)sokutei_dt;
                        cmd.Parameters.Add(new SQLiteParameter("@sokutei_dt0", cDate.ToString("yyyy-MM-dd HH:mm:ss")));
                        cmd.Parameters.Add(new SQLiteParameter("@sokutei_dt1", cDate.AddDays(1).ToString("yyyy-MM-dd HH:mm:ss")));
                        cmd.Parameters.Add(new SQLiteParameter("@sokutei_dt2", cDate.AddDays(2).ToString("yyyy-MM-dd HH:mm:ss")));
                    }

                    string ccFormat = ColumnsConfigSet.nameToFormat("uryou");

                    if(SeitenConfigSet.getUtenLLUryouDay0(kanrisya, syorijou) != null) {
                        cmd.Parameters.Add(new SQLiteParameter("@uten_ll_uryou_day0", FunctionForDatabase.decConvForDB(SeitenConfigSet.getUtenLLUryouDay0(kanrisya, syorijou), ccFormat)));
                    }

                    if(SeitenConfigSet.getUtenLLUryouDay1(kanrisya, syorijou) != null) {
                        cmd.Parameters.Add(new SQLiteParameter("@uten_ll_uryou_day1", FunctionForDatabase.decConvForDB(SeitenConfigSet.getUtenLLUryouDay1(kanrisya, syorijou), ccFormat)));
                    }

                    if(SeitenConfigSet.getUtenLLUryouDay2(kanrisya, syorijou) != null) {
                        cmd.Parameters.Add(new SQLiteParameter("@uten_ll_uryou_day2", FunctionForDatabase.decConvForDB(SeitenConfigSet.getUtenLLUryouDay2(kanrisya, syorijou), ccFormat)));
                    }

                    cmd.ExecuteNonQuery();
                }
            }
        }




        //SQLiteDataReaderからdecimal?を取得する
        public static decimal? getDecimalOrNull(SQLiteDataReader reader, int columnNo) {
            return reader.IsDBNull(columnNo) ? null : (decimal?)reader.GetDecimal(columnNo);
        }




        //SQLiteDataReaderからstring?を取得する
        public static string getStringOrNull(SQLiteDataReader reader, int columnNo) {
            return reader.IsDBNull(columnNo) ? null : reader.GetString(columnNo);
        }
    }




    /// <summary>
    /// 計算によって求める二次データを求めるクラス
    /// </summary>
    public static class Calculate2ndValue {
        //その日が晴天日扱いならtrueを返す（その処理場の前日・前々日の雨量が影響する）
        public static bool judgeSeiten(string kanrisya, string syorijou, DateTime cDate) {
            decimal? utenLLUryouDay0 = SeitenConfigSet.getUtenLLUryouDay0(kanrisya, syorijou);
            decimal? utenLLUryouDay1 = SeitenConfigSet.getUtenLLUryouDay1(kanrisya, syorijou);
            decimal? utenLLUryouDay2 = SeitenConfigSet.getUtenLLUryouDay2(kanrisya, syorijou);

            string D0 = cDate.ToString("yyyy-MM-dd") + " 00:00:00";
            string D1 = cDate.AddDays(-1).ToString("yyyy-MM-dd") + " 00:00:00";
            string D2 = cDate.AddDays(-2).ToString("yyyy-MM-dd") + " 00:00:00";

            using(var con = new SQLiteConnection(DataBaseConfig.ConnectionString)) {
                con.Open();

                using(var cmd = con.CreateCommand()) {
                    string sql = "";
                    string ccFormat = ColumnsConfigSet.nameToFormat("uryou");

                    if(utenLLUryouDay0 != null) {
                        sql += "sokutei_dt='" + D0 + "' AND del_flg IS NULL AND uryou>=" + FunctionForDatabase.decConvForDB(utenLLUryouDay0, ccFormat).ToString();
                    }

                    if(utenLLUryouDay1 != null) {
                        if(sql != "") {
                            sql += " OR ";
                        }

                        sql += "sokutei_dt='" + D1 + "' AND del_flg IS NULL AND uryou>=" + FunctionForDatabase.decConvForDB(utenLLUryouDay1, ccFormat).ToString();
                    }

                    if(utenLLUryouDay2 != null) {
                        if(sql != "") {
                            sql += " OR ";
                        }

                        sql += "sokutei_dt='" + D2 + "' AND del_flg IS NULL AND uryou>=" + FunctionForDatabase.decConvForDB(utenLLUryouDay2, ccFormat).ToString();
                    }

                    if(sql != "") {
                        sql = " AND (" + sql + ")";
                    }
                    else {
                        sql = " AND 1=0";
                    }

                    //雨量が閾値以上の日があれば雨天
                    cmd.CommandText = "SELECT COUNT(*) FROM suishitsu_kanri WHERE kanrisya='" + kanrisya + "' AND syorijou='" + syorijou + "'" + sql + ";";

                    using(var reader = cmd.ExecuteReader()) {
                        reader.Read();

                        if(reader.HasRows) {
                            if(!reader.IsDBNull(0)) {
                                if(reader.GetInt32(0) >= 1) {
                                    return false;
                                }
                            }
                        }
                    }
                }
            }

            //それ以外なら晴天扱いする
            return true;
        }




        //二次データを求める計算（数値戻り）
        public static decimal? calc2ndDecimal(string ccName, DataGridViewRow dgv1Row = null, List<ColumnConfig> ccSet = null) {
            decimal? retValue = null;


            #region P場・幹線計算定義

            if(ccName == "wv_pumpjou_all") {
                //P場排水量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["wv_pumpjou_" + i.ToString()].Value != null) {
                            decimal wv_pumpjou = 0;

                            if(decimal.TryParse(dgv1Row.Cells["wv_pumpjou_" + i.ToString()].Value.ToString(), out wv_pumpjou)) {
                                sum += wv_pumpjou;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? wv_pumpjou2 = ccSet[ColumnsConfigSet.nameToIndex("wv_pumpjou_" + i.ToString())].tempValueDec;

                        if(wv_pumpjou2 != null) {
                            sum += (decimal)wv_pumpjou2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ee_pumpjou_all") {
                //P場電力量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["ee_pumpjou_" + i.ToString()].Value != null) {
                            decimal ee_pumpjou = 0;

                            if(decimal.TryParse(dgv1Row.Cells["ee_pumpjou_" + i.ToString()].Value.ToString(), out ee_pumpjou)) {
                                sum += ee_pumpjou;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? ee_pumpjou2 = ccSet[ColumnsConfigSet.nameToIndex("ee_pumpjou_" + i.ToString())].tempValueDec;

                        if(ee_pumpjou2 != null) {
                            sum += (decimal)ee_pumpjou2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "v_hao_pumpjou_all") {
                //P場A重油計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["v_hao_pumpjou_" + i.ToString()].Value != null) {
                            decimal v_hao_pumpjou = 0;

                            if(decimal.TryParse(dgv1Row.Cells["v_hao_pumpjou_" + i.ToString()].Value.ToString(), out v_hao_pumpjou)) {
                                sum += v_hao_pumpjou;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? v_hao_pumpjou2 = ccSet[ColumnsConfigSet.nameToIndex("v_hao_pumpjou_" + i.ToString())].tempValueDec;

                        if(v_hao_pumpjou2 != null) {
                            sum += (decimal)v_hao_pumpjou2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "v_lo_pumpjou_all") {
                //P場軽油計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["v_lo_pumpjou_" + i.ToString()].Value != null) {
                            decimal v_lo_pumpjou = 0;

                            if(decimal.TryParse(dgv1Row.Cells["v_lo_pumpjou_" + i.ToString()].Value.ToString(), out v_lo_pumpjou)) {
                                sum += v_lo_pumpjou;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? v_lo_pumpjou2 = ccSet[ColumnsConfigSet.nameToIndex("v_lo_pumpjou_" + i.ToString())].tempValueDec;

                        if(v_lo_pumpjou2 != null) {
                            sum += (decimal)v_lo_pumpjou2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "v_chinsa_pumpjou_all") {
                //P場沈砂発生量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["v_chinsa_pumpjou_" + i.ToString()].Value != null) {
                            decimal v_chinsa_pumpjou = 0;

                            if(decimal.TryParse(dgv1Row.Cells["v_chinsa_pumpjou_" + i.ToString()].Value.ToString(), out v_chinsa_pumpjou)) {
                                sum += v_chinsa_pumpjou;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? v_chinsa_pumpjou2 = ccSet[ColumnsConfigSet.nameToIndex("v_chinsa_pumpjou_" + i.ToString())].tempValueDec;

                        if(v_chinsa_pumpjou2 != null) {
                            sum += (decimal)v_chinsa_pumpjou2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "v_shisa_pumpjou_all") {
                //P場しさ発生量計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxPumpjou; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["v_shisa_pumpjou_" + i.ToString()].Value != null) {
                            decimal v_shisa_pumpjou = 0;

                            if(decimal.TryParse(dgv1Row.Cells["v_shisa_pumpjou_" + i.ToString()].Value.ToString(), out v_shisa_pumpjou)) {
                                sum += v_shisa_pumpjou;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? v_shisa_pumpjou2 = ccSet[ColumnsConfigSet.nameToIndex("v_shisa_pumpjou_" + i.ToString())].tempValueDec;

                        if(v_shisa_pumpjou2 != null) {
                            sum += (decimal)v_shisa_pumpjou2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }

            #endregion


            #region 処理場計算定義

            else if(ccName == "ee_extra") {
                //その他電力量
                decimal ee_juden = 0;
                decimal ee_jikahatsu = 0;
                decimal ee_jounaipump = 0;
                decimal ee_mizusyori = 0;
                decimal ee_odeisyori = 0;
                bool nullFlug = true;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ee_juden"].Value == null) {
                        ee_juden = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ee_juden"].Value.ToString(), out ee_juden)) {
                            ee_juden = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ee_jikahatsu"].Value == null) {
                        ee_jikahatsu = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ee_jikahatsu"].Value.ToString(), out ee_jikahatsu)) {
                            ee_jikahatsu = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ee_jounaipump"].Value == null) {
                        ee_jounaipump = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ee_jounaipump"].Value.ToString(), out ee_jounaipump)) {
                            ee_jounaipump = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ee_mizusyori"].Value == null) {
                        ee_mizusyori = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ee_mizusyori"].Value.ToString(), out ee_mizusyori)) {
                            ee_mizusyori = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ee_odeisyori"].Value == null) {
                        ee_odeisyori = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ee_odeisyori"].Value.ToString(), out ee_odeisyori)) {
                            ee_odeisyori = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ee_juden2 = ccSet[ColumnsConfigSet.nameToIndex("ee_juden")].tempValueDec;

                    if(ee_juden2 == null) {
                        ee_juden2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ee_jikahatsu2 = ccSet[ColumnsConfigSet.nameToIndex("ee_jikahatsu")].tempValueDec;

                    if(ee_jikahatsu2 == null) {
                        ee_jikahatsu2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ee_jounaipump2 = ccSet[ColumnsConfigSet.nameToIndex("ee_jounaipump")].tempValueDec;

                    if(ee_jounaipump2 == null) {
                        ee_jounaipump2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ee_mizusyori2 = ccSet[ColumnsConfigSet.nameToIndex("ee_mizusyori")].tempValueDec;

                    if(ee_mizusyori2 == null) {
                        ee_mizusyori2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ee_odeisyori2 = ccSet[ColumnsConfigSet.nameToIndex("ee_odeisyori")].tempValueDec;

                    if(ee_odeisyori2 == null) {
                        ee_odeisyori2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    ee_juden = (decimal)ee_juden2;
                    ee_jikahatsu = (decimal)ee_jikahatsu2;
                    ee_jounaipump = (decimal)ee_jounaipump2;
                    ee_mizusyori = (decimal)ee_mizusyori2;
                    ee_odeisyori = (decimal)ee_odeisyori2;
                }

                if(nullFlug) {
                    return null;
                }

                decimal retValue2 = ee_juden + ee_jikahatsu - ee_jounaipump - ee_mizusyori - ee_odeisyori;
                retValue = retValue2;
            }

            #endregion


            #region 初沈計算定義

            else if(ccName == "h_syochin") {
                //初沈沈殿時間
                decimal pv_syochin = 0;
                decimal wv_syochin = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["pv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["pv_syochin"].Value.ToString(), out pv_syochin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_syochin"].Value.ToString(), out wv_syochin)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? pv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("pv_syochin")].tempValueDec;

                    if(pv_syochin2 == null) {
                        return null;
                    }

                    decimal? wv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("wv_syochin")].tempValueDec;

                    if(wv_syochin2 == null) {
                        return null;
                    }

                    pv_syochin = (decimal)pv_syochin2;
                    wv_syochin = (decimal)wv_syochin2;
                }

                if(wv_syochin <= 0) {
                    return null;
                }

                decimal retValue2 = pv_syochin / wv_syochin * 24;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "ov_nama") {
                //生汚泥引抜量計
                decimal ov_nama_nousyuku = 0;
                decimal ov_nama_syouka = 0;
                decimal ov_nama_dassui = 0;
                decimal ov_nama_jougai = 0;
                bool nullFlug = true;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ov_nama_nousyuku"].Value == null) {
                        ov_nama_nousyuku = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nama_nousyuku"].Value.ToString(), out ov_nama_nousyuku)) {
                            ov_nama_nousyuku = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_nama_syouka"].Value == null) {
                        ov_nama_syouka = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nama_syouka"].Value.ToString(), out ov_nama_syouka)) {
                            ov_nama_syouka = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_nama_dassui"].Value == null) {
                        ov_nama_dassui = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nama_dassui"].Value.ToString(), out ov_nama_dassui)) {
                            ov_nama_dassui = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_nama_jougai"].Value == null) {
                        ov_nama_jougai = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nama_jougai"].Value.ToString(), out ov_nama_jougai)) {
                            ov_nama_jougai = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ov_nama_nousyuku2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nama_nousyuku")].tempValueDec;

                    if(ov_nama_nousyuku2 == null) {
                        ov_nama_nousyuku2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_nama_syouka2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nama_syouka")].tempValueDec;

                    if(ov_nama_syouka2 == null) {
                        ov_nama_syouka2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_nama_dassui2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nama_dassui")].tempValueDec;

                    if(ov_nama_dassui2 == null) {
                        ov_nama_dassui2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_nama_jougai2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nama_jougai")].tempValueDec;

                    if(ov_nama_jougai2 == null) {
                        ov_nama_jougai2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    ov_nama_nousyuku = (decimal)ov_nama_nousyuku2;
                    ov_nama_syouka = (decimal)ov_nama_syouka2;
                    ov_nama_dassui = (decimal)ov_nama_dassui2;
                    ov_nama_jougai = (decimal)ov_nama_jougai2;
                }

                if(nullFlug) {
                    return null;
                }

                decimal retValue2 = ov_nama_nousyuku + ov_nama_syouka + ov_nama_dassui + ov_nama_jougai;
                retValue = retValue2;
            }
            else if(ccName == "sg_nama") {
                //生汚泥固形物量
                decimal ov_nama_nousyuku = 0;
                decimal ov_nama_syouka = 0;
                decimal ov_nama_dassui = 0;
                decimal ov_nama_jougai = 0;
                decimal oc_nama = 0;
                bool nullFlug = true;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ov_nama_nousyuku"].Value == null) {
                        ov_nama_nousyuku = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nama_nousyuku"].Value.ToString(), out ov_nama_nousyuku)) {
                            ov_nama_nousyuku = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_nama_syouka"].Value == null) {
                        ov_nama_syouka = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nama_syouka"].Value.ToString(), out ov_nama_syouka)) {
                            ov_nama_syouka = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_nama_dassui"].Value == null) {
                        ov_nama_dassui = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nama_dassui"].Value.ToString(), out ov_nama_dassui)) {
                            ov_nama_dassui = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_nama_jougai"].Value == null) {
                        ov_nama_jougai = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nama_jougai"].Value.ToString(), out ov_nama_jougai)) {
                            ov_nama_jougai = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["oc_nama"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_nama"].Value.ToString(), out oc_nama)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ov_nama_nousyuku2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nama_nousyuku")].tempValueDec;

                    if(ov_nama_nousyuku2 == null) {
                        ov_nama_nousyuku2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_nama_syouka2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nama_syouka")].tempValueDec;

                    if(ov_nama_syouka2 == null) {
                        ov_nama_syouka2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_nama_dassui2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nama_dassui")].tempValueDec;

                    if(ov_nama_dassui2 == null) {
                        ov_nama_dassui2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_nama_jougai2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nama_jougai")].tempValueDec;

                    if(ov_nama_jougai2 == null) {
                        ov_nama_jougai2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? oc_nama2 = ccSet[ColumnsConfigSet.nameToIndex("oc_nama")].tempValueDec;

                    if(oc_nama2 == null) {
                        return null;
                    }

                    ov_nama_nousyuku = (decimal)ov_nama_nousyuku2;
                    ov_nama_syouka = (decimal)ov_nama_syouka2;
                    ov_nama_dassui = (decimal)ov_nama_dassui2;
                    ov_nama_jougai = (decimal)ov_nama_jougai2;
                    oc_nama = (decimal)oc_nama2;
                }

                if(nullFlug) {
                    return null;
                }

                decimal retValue2 = (ov_nama_nousyuku + ov_nama_syouka + ov_nama_dassui + ov_nama_jougai) * oc_nama / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "ss_jokyo_syo_etsuryuu") {
                //初沈越除去率SS
                decimal ss_syo_ryuunyuu = 0;
                decimal ss_syo_etsuryuu = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ss_syo_ryuunyuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ss_syo_ryuunyuu"].Value.ToString(), out ss_syo_ryuunyuu)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ss_syo_etsuryuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ss_syo_etsuryuu"].Value.ToString(), out ss_syo_etsuryuu)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ss_syo_ryuunyuu2 = ccSet[ColumnsConfigSet.nameToIndex("ss_syo_ryuunyuu")].tempValueDec;

                    if(ss_syo_ryuunyuu2 == null) {
                        return null;
                    }

                    decimal? ss_syo_etsuryuu2 = ccSet[ColumnsConfigSet.nameToIndex("ss_syo_etsuryuu")].tempValueDec;

                    if(ss_syo_etsuryuu2 == null) {
                        return null;
                    }

                    ss_syo_ryuunyuu = (decimal)ss_syo_ryuunyuu2;
                    ss_syo_etsuryuu = (decimal)ss_syo_etsuryuu2;
                }

                if(ss_syo_ryuunyuu <= 0) {
                    return null;
                }

                decimal retValue2 = (ss_syo_ryuunyuu - ss_syo_etsuryuu) / ss_syo_ryuunyuu * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }

            #endregion


            #region 反応T共通計算定義

            else if(ccName == "wt_hannoutank_all") {
                //反応T平均水温
                decimal sum = 0;
                int count = 0;

                for(int i=1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["wt_hannoutank_" + i.ToString()].Value != null) {
                            decimal wt_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["wt_hannoutank_" + i.ToString()].Value.ToString(), out wt_hannoutank)) {
                                sum += wt_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? wt_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("wt_hannoutank_" + i.ToString())].tempValueDec;

                        if(wt_hannoutank2 != null) {
                            sum += (decimal)wt_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "ph_hannoutank_all") {
                //反応T平均pH
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["ph_hannoutank_" + i.ToString()].Value != null) {
                            decimal ph_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["ph_hannoutank_" + i.ToString()].Value.ToString(), out ph_hannoutank)) {
                                sum += ph_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? ph_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("ph_hannoutank_" + i.ToString())].tempValueDec;

                        if(ph_hannoutank2 != null) {
                            sum += (decimal)ph_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sv_hannoutank_all") {
                //反応T平均SV
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["sv_hannoutank_" + i.ToString()].Value != null) {
                            decimal sv_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["sv_hannoutank_" + i.ToString()].Value.ToString(), out sv_hannoutank)) {
                                sum += sv_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? sv_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("sv_hannoutank_" + i.ToString())].tempValueDec;

                        if(sv_hannoutank2 != null) {
                            sum += (decimal)sv_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "mldo_p_hannoutank_all") {
                //反応T平均MLDO前
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["mldo_p_hannoutank_" + i.ToString()].Value != null) {
                            decimal mldo_p_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["mldo_p_hannoutank_" + i.ToString()].Value.ToString(), out mldo_p_hannoutank)) {
                                sum += mldo_p_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? mldo_p_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mldo_p_hannoutank_" + i.ToString())].tempValueDec;

                        if(mldo_p_hannoutank2 != null) {
                            sum += (decimal)mldo_p_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "mldo_a_hannoutank_all") {
                //反応T平均MLDO後
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["mldo_a_hannoutank_" + i.ToString()].Value != null) {
                            decimal mldo_a_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["mldo_a_hannoutank_" + i.ToString()].Value.ToString(), out mldo_a_hannoutank)) {
                                sum += mldo_a_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? mldo_a_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mldo_a_hannoutank_" + i.ToString())].tempValueDec;

                        if(mldo_a_hannoutank2 != null) {
                            sum += (decimal)mldo_a_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "mlss_hannoutank_all") {
                //反応T平均MLSS
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value != null) {
                            decimal mlss_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value.ToString(), out mlss_hannoutank)) {
                                sum += mlss_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? mlss_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mlss_hannoutank_" + i.ToString())].tempValueDec;

                        if(mlss_hannoutank2 != null) {
                            sum += (decimal)mlss_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "mlvss_hannoutank_all") {
                //反応T平均MLVSS
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["mlvss_hannoutank_" + i.ToString()].Value != null) {
                            decimal mlvss_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["mlvss_hannoutank_" + i.ToString()].Value.ToString(), out mlvss_hannoutank)) {
                                sum += mlvss_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? mlvss_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mlvss_hannoutank_" + i.ToString())].tempValueDec;

                        if(mlvss_hannoutank2 != null) {
                            sum += (decimal)mlvss_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "vsspss_hannoutank_all") {
                //反応T平均VSS/SS
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value != null) {
                            decimal mlss_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value.ToString(), out mlss_hannoutank)) {
                                if(mlss_hannoutank > 0) {
                                    if(dgv1Row.Cells["mlvss_hannoutank_" + i.ToString()].Value != null) {
                                        decimal mlvss_hannoutank = 0;

                                        if(decimal.TryParse(dgv1Row.Cells["mlvss_hannoutank_" + i.ToString()].Value.ToString(), out mlvss_hannoutank)) {
                                            sum += mlvss_hannoutank / mlss_hannoutank * 100;
                                            count++;
                                        }
                                    }
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? mlss_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mlss_hannoutank_" + i.ToString())].tempValueDec;

                        if(mlss_hannoutank2 != null) {
                            if(mlss_hannoutank2 > 0) {
                                decimal? mlvss_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mlvss_hannoutank_" + i.ToString())].tempValueDec;

                                if(mlvss_hannoutank2 != null) {
                                    sum += (decimal)mlvss_hannoutank2 / (decimal)mlss_hannoutank2 * 100;
                                    count++;
                                }
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "svi_hannoutank_all") {
                //反応T平均SVI
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value != null) {
                            decimal mlss_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value.ToString(), out mlss_hannoutank)) {
                                if(mlss_hannoutank > 0) {
                                    if(dgv1Row.Cells["sv_hannoutank_" + i.ToString()].Value != null) {
                                        decimal sv_hannoutank = 0;

                                        if(decimal.TryParse(dgv1Row.Cells["sv_hannoutank_" + i.ToString()].Value.ToString(), out sv_hannoutank)) {
                                            sum += sv_hannoutank * 10000 / mlss_hannoutank;
                                            count++;
                                        }
                                    }
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? mlss_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mlss_hannoutank_" + i.ToString())].tempValueDec;

                        if(mlss_hannoutank2 != null) {
                            if(mlss_hannoutank2 > 0) {
                                decimal? sv_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("sv_hannoutank_" + i.ToString())].tempValueDec;

                                if(sv_hannoutank2 != null) {
                                    sum += (decimal)sv_hannoutank2 * 10000 / (decimal)mlss_hannoutank2;
                                    count++;
                                }
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "bodpmlss_hannoutank_all") {
                //反応T平均BOD/MLSS負荷
                decimal wv_syochin = 0;
                decimal wv_kani = 0;
                decimal bod_syo_etsuryuu = 0;
                decimal pv_hannoutank = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["wv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_syochin"].Value.ToString(), out wv_syochin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_kani"].Value == null) {
                        wv_kani = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_kani"].Value.ToString(), out wv_kani)) {
                            wv_kani = 0;
                        }
                    }

                    if(dgv1Row.Cells["bod_syo_etsuryuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["bod_syo_etsuryuu"].Value.ToString(), out bod_syo_etsuryuu)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["pv_hannoutank"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["pv_hannoutank"].Value.ToString(), out pv_hannoutank)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? wv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("wv_syochin")].tempValueDec;

                    if(wv_syochin2 == null) {
                        return null;
                    }

                    decimal? wv_kani2 = ccSet[ColumnsConfigSet.nameToIndex("wv_kani")].tempValueDec;

                    if(wv_kani2 == null) {
                        wv_kani2 = 0;
                    }

                    decimal? bod_syo_etsuryuu2 = ccSet[ColumnsConfigSet.nameToIndex("bod_syo_etsuryuu")].tempValueDec;

                    if(bod_syo_etsuryuu2 == null) {
                        return null;
                    }

                    decimal? pv_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("pv_hannoutank")].tempValueDec;

                    if(pv_hannoutank2 == null) {
                        return null;
                    }

                    wv_syochin = (decimal)wv_syochin2;
                    wv_kani = (decimal)wv_kani2;
                    bod_syo_etsuryuu = (decimal)bod_syo_etsuryuu2;
                    pv_hannoutank = (decimal)pv_hannoutank2;
                }

                if(pv_hannoutank <= 0) {
                    return null;
                }

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value != null) {
                            decimal mlss_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value.ToString(), out mlss_hannoutank)) {
                                if(mlss_hannoutank > 0) {
                                    sum += (wv_syochin - wv_kani) * bod_syo_etsuryuu / (pv_hannoutank * mlss_hannoutank);
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? mlss_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mlss_hannoutank_" + i.ToString())].tempValueDec;

                        if(mlss_hannoutank2 != null) {
                            if(mlss_hannoutank2 > 0) {
                                sum += (wv_syochin - wv_kani) * bod_syo_etsuryuu / (pv_hannoutank * (decimal)mlss_hannoutank2);
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "ar_hannoutank_all") {
                //反応T空気倍率
                decimal av_hannoutank_all = 0;
                decimal wv_syochin = 0;
                decimal wv_kani = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["av_hannoutank_all"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["av_hannoutank_all"].Value.ToString(), out av_hannoutank_all)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_syochin"].Value.ToString(), out wv_syochin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_kani"].Value == null) {
                        wv_kani = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_kani"].Value.ToString(), out wv_kani)) {
                            wv_kani = 0;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? av_hannoutank_all2 = ccSet[ColumnsConfigSet.nameToIndex("av_hannoutank_all")].tempValueDec;

                    if(av_hannoutank_all2 == null) {
                        return null;
                    }

                    decimal? wv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("wv_syochin")].tempValueDec;

                    if(wv_syochin2 == null) {
                        return null;
                    }

                    decimal? wv_kani2 = ccSet[ColumnsConfigSet.nameToIndex("wv_kani")].tempValueDec;

                    if(wv_kani2 == null) {
                        wv_kani2 = 0;
                    }

                    av_hannoutank_all = (decimal)av_hannoutank_all2;
                    wv_syochin = (decimal)wv_syochin2;
                    wv_kani = (decimal)wv_kani2;
                }

                if((wv_syochin - wv_kani) <= 0) {
                    return null;
                }

                decimal retValue2 = av_hannoutank_all / (wv_syochin - wv_kani);
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "st_hannoutank_all") {
                //反応T滞留時間
                decimal pv_hannoutank = 0;
                decimal wv_syochin = 0;
                decimal wv_kani = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["pv_hannoutank"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["pv_hannoutank"].Value.ToString(), out pv_hannoutank)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_syochin"].Value.ToString(), out wv_syochin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_kani"].Value == null) {
                        wv_kani = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_kani"].Value.ToString(), out wv_kani)) {
                            wv_kani = 0;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? pv_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("pv_hannoutank")].tempValueDec;

                    if(pv_hannoutank2 == null) {
                        return null;
                    }

                    decimal? wv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("wv_syochin")].tempValueDec;

                    if(wv_syochin2 == null) {
                        return null;
                    }

                    decimal? wv_kani2 = ccSet[ColumnsConfigSet.nameToIndex("wv_kani")].tempValueDec;

                    if(wv_kani2 == null) {
                        wv_kani2 = 0;
                    }

                    pv_hannoutank = (decimal)pv_hannoutank2;
                    wv_syochin = (decimal)wv_syochin2;
                    wv_kani = (decimal)wv_kani2;
                }

                if((wv_syochin - wv_kani) <= 0) {
                    return null;
                }

                decimal retValue2 = pv_hannoutank / (wv_syochin - wv_kani) * 24;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "srt_hannoutank_all") {
                //反応T_SRT
                decimal pv_hannoutank = 0;
                decimal ov_yojou = 0;
                decimal rsss_hensou = 0;
                decimal wv_syochin = 0;
                decimal ss_enkondeguchi = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["pv_hannoutank"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["pv_hannoutank"].Value.ToString(), out pv_hannoutank)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_syochin"].Value != null) {
                        decimal ov = 0;

                        if(decimal.TryParse(dgv1Row.Cells["ov_yj_syochin"].Value.ToString(), out ov)) {
                            ov_yojou += ov;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_nousyuku"].Value != null) {
                        decimal ov = 0;

                        if(decimal.TryParse(dgv1Row.Cells["ov_yj_nousyuku"].Value.ToString(), out ov)) {
                            ov_yojou += ov;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_syouka"].Value != null) {
                        decimal ov = 0;

                        if(decimal.TryParse(dgv1Row.Cells["ov_yj_syouka"].Value.ToString(), out ov)) {
                            ov_yojou += ov;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_dassui"].Value != null) {
                        decimal ov = 0;

                        if(decimal.TryParse(dgv1Row.Cells["ov_yj_dassui"].Value.ToString(), out ov)) {
                            ov_yojou += ov;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_jougai"].Value != null) {
                        decimal ov = 0;

                        if(decimal.TryParse(dgv1Row.Cells["ov_yj_jougai"].Value.ToString(), out ov)) {
                            ov_yojou += ov;
                        }
                    }

                    if(dgv1Row.Cells["rsss_hensou"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["rsss_hensou"].Value.ToString(), out rsss_hensou)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_syochin"].Value.ToString(), out wv_syochin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ss_enkondeguchi"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ss_enkondeguchi"].Value.ToString(), out ss_enkondeguchi)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? pv_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("pv_hannoutank")].tempValueDec;

                    if(pv_hannoutank2 == null) {
                        return null;
                    }

                    decimal? ov_yj_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_syochin")].tempValueDec;

                    if(ov_yj_syochin2 != null) {
                        ov_yojou += (decimal)ov_yj_syochin2;
                    }

                    decimal? ov_yj_nousyuku2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_nousyuku")].tempValueDec;

                    if(ov_yj_nousyuku2 != null) {
                        ov_yojou += (decimal)ov_yj_nousyuku2;
                    }

                    decimal? ov_yj_syouka2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_syouka")].tempValueDec;

                    if(ov_yj_syouka2 != null) {
                        ov_yojou += (decimal)ov_yj_syouka2;
                    }

                    decimal? ov_yj_dassui2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_dassui")].tempValueDec;

                    if(ov_yj_dassui2 != null) {
                        ov_yojou += (decimal)ov_yj_dassui2;
                    }

                    decimal? ov_yj_jougai2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_jougai")].tempValueDec;

                    if(ov_yj_jougai2 != null) {
                        ov_yojou += (decimal)ov_yj_jougai2;
                    }

                    decimal? rsss_hensou2 = ccSet[ColumnsConfigSet.nameToIndex("rsss_hensou")].tempValueDec;

                    if(rsss_hensou2 == null) {
                        return null;
                    }

                    decimal? wv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("wv_syochin")].tempValueDec;

                    if(wv_syochin2 == null) {
                        return null;
                    }

                    decimal? ss_enkondeguchi2 = ccSet[ColumnsConfigSet.nameToIndex("ss_enkondeguchi")].tempValueDec;

                    if(ss_enkondeguchi2 == null) {
                        return null;
                    }

                    pv_hannoutank = (decimal)pv_hannoutank2;
                    rsss_hensou = (decimal)rsss_hensou2;
                    wv_syochin = (decimal)wv_syochin2;
                    ss_enkondeguchi = (decimal)ss_enkondeguchi2;
                }

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value != null) {
                            decimal mlss_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["mlss_hannoutank_" + i.ToString()].Value.ToString(), out mlss_hannoutank)) {
                                sum += mlss_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? mlss_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mlss_hannoutank_" + i.ToString())].tempValueDec;

                        if(mlss_hannoutank2 != null) {
                            sum += (decimal)mlss_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal mlss_hannoutank_all = sum / count;

                if((ov_yojou * rsss_hensou + (wv_syochin - ov_yojou) * ss_enkondeguchi) <= 0) {
                    return null;
                }

                decimal retValue2 = pv_hannoutank * mlss_hannoutank_all / (ov_yojou * rsss_hensou + (wv_syochin - ov_yojou) * ss_enkondeguchi);
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "orp_ken_hannoutank_all") {
                //反応T平均ORP嫌気
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["orp_ken_hannoutank_" + i.ToString()].Value != null) {
                            decimal orp_ken_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["orp_ken_hannoutank_" + i.ToString()].Value.ToString(), out orp_ken_hannoutank)) {
                                sum += orp_ken_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? orp_ken_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("orp_ken_hannoutank_" + i.ToString())].tempValueDec;

                        if(orp_ken_hannoutank2 != null) {
                            sum += (decimal)orp_ken_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "orp_kou_hannoutank_all") {
                //反応T平均ORP好気
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["orp_kou_hannoutank_" + i.ToString()].Value != null) {
                            decimal orp_kou_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["orp_kou_hannoutank_" + i.ToString()].Value.ToString(), out orp_kou_hannoutank)) {
                                sum += orp_kou_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? orp_kou_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("orp_kou_hannoutank_" + i.ToString())].tempValueDec;

                        if(orp_kou_hannoutank2 != null) {
                            sum += (decimal)orp_kou_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "rr_zen_hannoutank_all") {
                //反応T平均Rr前段
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["rr_zen_hannoutank_" + i.ToString()].Value != null) {
                            decimal rr_zen_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["rr_zen_hannoutank_" + i.ToString()].Value.ToString(), out rr_zen_hannoutank)) {
                                sum += rr_zen_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? rr_zen_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("rr_zen_hannoutank_" + i.ToString())].tempValueDec;

                        if(rr_zen_hannoutank2 != null) {
                            sum += (decimal)rr_zen_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "rr_kou_hannoutank_all") {
                //反応T平均Rr後段
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxHannoutank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["rr_kou_hannoutank_" + i.ToString()].Value != null) {
                            decimal rr_kou_hannoutank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["rr_kou_hannoutank_" + i.ToString()].Value.ToString(), out rr_kou_hannoutank)) {
                                sum += rr_kou_hannoutank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? rr_kou_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("rr_kou_hannoutank_" + i.ToString())].tempValueDec;

                        if(rr_kou_hannoutank2 != null) {
                            sum += (decimal)rr_kou_hannoutank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / count;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "vsspss_hensou") {
                //返送汚泥VSS/SS
                decimal rsvss_hensou = 0;
                decimal rsss_hensou = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["rsvss_hensou"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["rsvss_hensou"].Value.ToString(), out rsvss_hensou)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["rsss_hensou"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["rsss_hensou"].Value.ToString(), out rsss_hensou)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? rsvss_hensou2 = ccSet[ColumnsConfigSet.nameToIndex("rsvss_hensou")].tempValueDec;

                    if(rsvss_hensou2 == null) {
                        return null;
                    }

                    decimal? rsss_hensou2 = ccSet[ColumnsConfigSet.nameToIndex("rsss_hensou")].tempValueDec;

                    if(rsss_hensou2 == null) {
                        return null;
                    }

                    rsvss_hensou = (decimal)rsvss_hensou2;
                    rsss_hensou = (decimal)rsss_hensou2;
                }

                decimal retValue2 = rsvss_hensou / rsss_hensou * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "svi_hensou") {
                //返送汚泥SVI
                decimal sv_hensou = 0;
                decimal rsss_hensou = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["sv_hensou"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["sv_hensou"].Value.ToString(), out sv_hensou)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["rsss_hensou"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["rsss_hensou"].Value.ToString(), out rsss_hensou)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? sv_hensou2 = ccSet[ColumnsConfigSet.nameToIndex("sv_hensou")].tempValueDec;

                    if(sv_hensou2 == null) {
                        return null;
                    }

                    decimal? rsss_hensou2 = ccSet[ColumnsConfigSet.nameToIndex("rsss_hensou")].tempValueDec;

                    if(rsss_hensou2 == null) {
                        return null;
                    }

                    sv_hensou = (decimal)sv_hensou2;
                    rsss_hensou = (decimal)rsss_hensou2;
                }

                decimal retValue2 = sv_hensou * 10000 / rsss_hensou;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }

            #endregion


            #region 終沈計算定義

            else if(ccName == "h_syuuchin") {
                //終沈沈殿時間
                decimal pv_syuuchin = 0;
                decimal wv_syochin = 0;
                decimal wv_kani = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["pv_syuuchin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["pv_syuuchin"].Value.ToString(), out pv_syuuchin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_syochin"].Value.ToString(), out wv_syochin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_kani"].Value == null) {
                        wv_kani = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_kani"].Value.ToString(), out wv_kani)) {
                            wv_kani = 0;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? pv_syuuchin2 = ccSet[ColumnsConfigSet.nameToIndex("pv_syuuchin")].tempValueDec;

                    if(pv_syuuchin2 == null) {
                        return null;
                    }

                    decimal? wv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("wv_syochin")].tempValueDec;

                    if(wv_syochin2 == null) {
                        return null;
                    }

                    decimal? wv_kani2 = ccSet[ColumnsConfigSet.nameToIndex("wv_kani")].tempValueDec;

                    if(wv_kani2 == null) {
                        wv_kani2 = 0;
                    }

                    pv_syuuchin = (decimal)pv_syuuchin2;
                    wv_syochin = (decimal)wv_syochin2;
                    wv_kani = (decimal)wv_kani2;
                }

                if((wv_syochin - wv_kani) <= 0) {
                    return null;
                }

                decimal retValue2 = pv_syuuchin / (wv_syochin - wv_kani) * 24;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "or_hensou") {
                //返送汚泥率
                decimal ov_hensou = 0;
                decimal wv_syochin = 0;
                decimal wv_kani = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ov_hensou"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_hensou"].Value.ToString(), out ov_hensou)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_syochin"].Value.ToString(), out wv_syochin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_kani"].Value == null) {
                        wv_kani = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_kani"].Value.ToString(), out wv_kani)) {
                            wv_kani = 0;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ov_hensou2 = ccSet[ColumnsConfigSet.nameToIndex("ov_hensou")].tempValueDec;

                    if(ov_hensou2 == null) {
                        return null;
                    }

                    decimal? wv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("wv_syochin")].tempValueDec;

                    if(wv_syochin2 == null) {
                        return null;
                    }

                    decimal? wv_kani2 = ccSet[ColumnsConfigSet.nameToIndex("wv_kani")].tempValueDec;

                    if(wv_kani2 == null) {
                        wv_kani2 = 0;
                    }

                    ov_hensou = (decimal)ov_hensou2;
                    wv_syochin = (decimal)wv_syochin2;
                    wv_kani = (decimal)wv_kani2;
                }

                if((wv_syochin - wv_kani) <= 0) {
                    return null;
                }

                decimal retValue2 = ov_hensou / (wv_syochin - wv_kani) * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "ov_yojou") {
                //余剰汚泥計
                decimal ov_yj_syochin = 0;
                decimal ov_yj_nousyuku = 0;
                decimal ov_yj_syouka = 0;
                decimal ov_yj_dassui = 0;
                decimal ov_yj_jougai = 0;
                bool nullFlug = true;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ov_yj_syochin"].Value == null) {
                        ov_yj_syochin = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_syochin"].Value.ToString(), out ov_yj_syochin)) {
                            ov_yj_syochin = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_nousyuku"].Value == null) {
                        ov_yj_nousyuku = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_nousyuku"].Value.ToString(), out ov_yj_nousyuku)) {
                            ov_yj_nousyuku = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_syouka"].Value == null) {
                        ov_yj_syouka = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_syouka"].Value.ToString(), out ov_yj_syouka)) {
                            ov_yj_syouka = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_dassui"].Value == null) {
                        ov_yj_dassui = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_dassui"].Value.ToString(), out ov_yj_dassui)) {
                            ov_yj_dassui = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_jougai"].Value == null) {
                        ov_yj_jougai = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_jougai"].Value.ToString(), out ov_yj_jougai)) {
                            ov_yj_jougai = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ov_yj_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_syochin")].tempValueDec;

                    if(ov_yj_syochin2 == null) {
                        ov_yj_syochin2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_yj_nousyuku2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_nousyuku")].tempValueDec;

                    if(ov_yj_nousyuku2 == null) {
                        ov_yj_nousyuku2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_yj_syouka2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_syouka")].tempValueDec;

                    if(ov_yj_syouka2 == null) {
                        ov_yj_syouka2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_yj_dassui2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_dassui")].tempValueDec;

                    if(ov_yj_dassui2 == null) {
                        ov_yj_dassui2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_yj_jougai2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_jougai")].tempValueDec;

                    if(ov_yj_jougai2 == null) {
                        ov_yj_jougai2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    ov_yj_syochin = (decimal)ov_yj_syochin2;
                    ov_yj_nousyuku = (decimal)ov_yj_nousyuku2;
                    ov_yj_syouka = (decimal)ov_yj_syouka2;
                    ov_yj_dassui = (decimal)ov_yj_dassui2;
                    ov_yj_jougai = (decimal)ov_yj_jougai2;
                }

                if(nullFlug) {
                    return null;
                }

                decimal retValue2 = ov_yj_syochin + ov_yj_nousyuku + ov_yj_syouka + ov_yj_dassui + ov_yj_jougai;
                retValue = retValue2;
            }
            else if(ccName == "sg_yojou") {
                //余剰汚泥固形物量
                decimal ov_yj_syochin = 0;
                decimal ov_yj_nousyuku = 0;
                decimal ov_yj_syouka = 0;
                decimal ov_yj_dassui = 0;
                decimal ov_yj_jougai = 0;
                decimal oc_yojou = 0;
                bool nullFlug = true;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ov_yj_syochin"].Value == null) {
                        ov_yj_syochin = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_syochin"].Value.ToString(), out ov_yj_syochin)) {
                            ov_yj_syochin = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_nousyuku"].Value == null) {
                        ov_yj_nousyuku = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_nousyuku"].Value.ToString(), out ov_yj_nousyuku)) {
                            ov_yj_nousyuku = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_syouka"].Value == null) {
                        ov_yj_syouka = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_syouka"].Value.ToString(), out ov_yj_syouka)) {
                            ov_yj_syouka = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_dassui"].Value == null) {
                        ov_yj_dassui = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_dassui"].Value.ToString(), out ov_yj_dassui)) {
                            ov_yj_dassui = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_yj_jougai"].Value == null) {
                        ov_yj_jougai = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_yj_jougai"].Value.ToString(), out ov_yj_jougai)) {
                            ov_yj_jougai = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["oc_yojou"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_yojou"].Value.ToString(), out oc_yojou)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ov_yj_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_syochin")].tempValueDec;

                    if(ov_yj_syochin2 == null) {
                        ov_yj_syochin2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_yj_nousyuku2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_nousyuku")].tempValueDec;

                    if(ov_yj_nousyuku2 == null) {
                        ov_yj_nousyuku2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_yj_syouka2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_syouka")].tempValueDec;

                    if(ov_yj_syouka2 == null) {
                        ov_yj_syouka2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_yj_dassui2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_dassui")].tempValueDec;

                    if(ov_yj_dassui2 == null) {
                        ov_yj_dassui2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_yj_jougai2 = ccSet[ColumnsConfigSet.nameToIndex("ov_yj_jougai")].tempValueDec;

                    if(ov_yj_jougai2 == null) {
                        ov_yj_jougai2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? oc_yojou2 = ccSet[ColumnsConfigSet.nameToIndex("oc_yojou")].tempValueDec;

                    if(oc_yojou2 == null) {
                        return null;
                    }

                    ov_yj_syochin = (decimal)ov_yj_syochin2;
                    ov_yj_nousyuku = (decimal)ov_yj_nousyuku2;
                    ov_yj_syouka = (decimal)ov_yj_syouka2;
                    ov_yj_dassui = (decimal)ov_yj_dassui2;
                    ov_yj_jougai = (decimal)ov_yj_jougai2;
                    oc_yojou = (decimal)oc_yojou2;
                }

                if(nullFlug) {
                    return null;
                }

                decimal retValue2 = (ov_yj_syochin + ov_yj_nousyuku + ov_yj_syouka + ov_yj_dassui + ov_yj_jougai) * oc_yojou / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }

            #endregion


            #region 塩混・放流計算定義

            else if(ccName == "mr_jia") {
                //次亜注入率
                decimal wv_jia = 0;
                decimal gs_jia = 0;
                decimal er_jia = 0;
                decimal wv_houryuu = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["wv_jia"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_jia"].Value.ToString(), out wv_jia)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["gs_jia"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["gs_jia"].Value.ToString(), out gs_jia)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["er_jia"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["er_jia"].Value.ToString(), out er_jia)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_houryuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_houryuu"].Value.ToString(), out wv_houryuu)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? wv_jia2 = ccSet[ColumnsConfigSet.nameToIndex("wv_jia")].tempValueDec;

                    if(wv_jia2 == null) {
                        return null;
                    }

                    decimal? gs_jia2 = ccSet[ColumnsConfigSet.nameToIndex("gs_jia")].tempValueDec;

                    if(gs_jia2 == null) {
                        return null;
                    }

                    decimal? er_jia2 = ccSet[ColumnsConfigSet.nameToIndex("er_jia")].tempValueDec;

                    if(er_jia2 == null) {
                        er_jia2 = 0;
                    }

                    decimal? wv_houryuu2 = ccSet[ColumnsConfigSet.nameToIndex("wv_houryuu")].tempValueDec;

                    if(wv_houryuu2 == null) {
                        wv_houryuu2 = 0;
                    }

                    wv_jia = (decimal)wv_jia2;
                    gs_jia = (decimal)gs_jia2;
                    er_jia = (decimal)er_jia2;
                    wv_houryuu = (decimal)wv_houryuu2;
                }

                if(wv_houryuu <= 0) {
                    return null;
                }

                decimal retValue2 = wv_jia * gs_jia * er_jia * 10 / wv_houryuu;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "min_enkon") {
                //塩混接触時間
                decimal pv_enkon = 0;
                decimal wv_houryuu = 0;
                decimal wv_kani = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["pv_enkon"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["pv_enkon"].Value.ToString(), out pv_enkon)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_houryuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_houryuu"].Value.ToString(), out wv_houryuu)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_kani"].Value == null) {
                        wv_kani = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_kani"].Value.ToString(), out wv_kani)) {
                            wv_kani = 0;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? pv_enkon2 = ccSet[ColumnsConfigSet.nameToIndex("pv_enkon")].tempValueDec;

                    if(pv_enkon2 == null) {
                        return null;
                    }

                    decimal? wv_houryuu2 = ccSet[ColumnsConfigSet.nameToIndex("wv_houryuu")].tempValueDec;

                    if(wv_houryuu2 == null) {
                        wv_houryuu2 = 0;
                    }

                    decimal? wv_kani2 = ccSet[ColumnsConfigSet.nameToIndex("wv_kani")].tempValueDec;

                    if(wv_kani2 == null) {
                        wv_kani2 = 0;
                    }

                    pv_enkon = (decimal)pv_enkon2;
                    wv_houryuu = (decimal)wv_houryuu2;
                    wv_kani = (decimal)wv_kani2;
                }

                if((wv_houryuu - wv_kani) <= 0) {
                    return null;
                }

                decimal retValue2 = pv_enkon / (wv_houryuu - wv_kani) * 24 * 60;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "cod_jokyo_sougou") {
                //総合除去率COD
                decimal cod_ryuunyuu = 0;
                decimal cod_enkondeguchi = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["cod_ryuunyuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["cod_ryuunyuu"].Value.ToString(), out cod_ryuunyuu)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["cod_enkondeguchi"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["cod_enkondeguchi"].Value.ToString(), out cod_enkondeguchi)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? cod_ryuunyuu2 = ccSet[ColumnsConfigSet.nameToIndex("cod_ryuunyuu")].tempValueDec;

                    if(cod_ryuunyuu2 == null) {
                        return null;
                    }

                    decimal? cod_enkondeguchi2 = ccSet[ColumnsConfigSet.nameToIndex("cod_enkondeguchi")].tempValueDec;

                    if(cod_enkondeguchi2 == null) {
                        return null;
                    }

                    cod_ryuunyuu = (decimal)cod_ryuunyuu2;
                    cod_enkondeguchi = (decimal)cod_enkondeguchi2;
                }

                if(cod_ryuunyuu <= 0) {
                    return null;
                }

                decimal retValue2 = (cod_ryuunyuu - cod_enkondeguchi) / cod_ryuunyuu * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "bod_jokyo_sougou") {
                //総合除去率BOD
                decimal bod_ryuunyuu = 0;
                decimal bod_enkondeguchi = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["bod_ryuunyuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["bod_ryuunyuu"].Value.ToString(), out bod_ryuunyuu)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["bod_enkondeguchi"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["bod_enkondeguchi"].Value.ToString(), out bod_enkondeguchi)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? bod_ryuunyuu2 = ccSet[ColumnsConfigSet.nameToIndex("bod_ryuunyuu")].tempValueDec;

                    if(bod_ryuunyuu2 == null) {
                        return null;
                    }

                    decimal? bod_enkondeguchi2 = ccSet[ColumnsConfigSet.nameToIndex("bod_enkondeguchi")].tempValueDec;

                    if(bod_enkondeguchi2 == null) {
                        return null;
                    }

                    bod_ryuunyuu = (decimal)bod_ryuunyuu2;
                    bod_enkondeguchi = (decimal)bod_enkondeguchi2;
                }

                if(bod_ryuunyuu <= 0) {
                    return null;
                }

                decimal retValue2 = (bod_ryuunyuu - bod_enkondeguchi) / bod_ryuunyuu * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "ss_jokyo_sougou") {
                //総合除去率SS
                decimal ss_ryuunyuu = 0;
                decimal ss_enkondeguchi = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ss_ryuunyuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ss_ryuunyuu"].Value.ToString(), out ss_ryuunyuu)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ss_enkondeguchi"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ss_enkondeguchi"].Value.ToString(), out ss_enkondeguchi)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ss_ryuunyuu2 = ccSet[ColumnsConfigSet.nameToIndex("ss_ryuunyuu")].tempValueDec;

                    if(ss_ryuunyuu2 == null) {
                        return null;
                    }

                    decimal? ss_enkondeguchi2 = ccSet[ColumnsConfigSet.nameToIndex("ss_enkondeguchi")].tempValueDec;

                    if(ss_enkondeguchi2 == null) {
                        return null;
                    }

                    ss_ryuunyuu = (decimal)ss_ryuunyuu2;
                    ss_enkondeguchi = (decimal)ss_enkondeguchi2;
                }

                if(ss_ryuunyuu <= 0) {
                    return null;
                }

                decimal retValue2 = (ss_ryuunyuu - ss_enkondeguchi) / ss_ryuunyuu * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }

            #endregion


            #region 汚泥濃縮計算定義

            else if(ccName == "ov_nousyukuki_input_juuryoku") {
                //投入汚泥量重力濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() == "重力式") {
                            if(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value != null) {
                                decimal ov_nousyukuki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value.ToString(), out ov_nousyukuki)) {
                                    sum += ov_nousyukuki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr == "重力式") {
                            decimal? ov_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyukuki_" + i.ToString())].tempValueDec;

                            if(ov_nousyukuki2 != null) {
                                sum += (decimal)ov_nousyukuki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ov_nousyukuki_input_ex") {
                //投入汚泥量他濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() != "重力式") {
                            if(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value != null) {
                                decimal ov_nousyukuki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value.ToString(), out ov_nousyukuki)) {
                                    sum += ov_nousyukuki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr != "重力式") {
                            decimal? ov_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyukuki_" + i.ToString())].tempValueDec;

                            if(ov_nousyukuki2 != null) {
                                sum += (decimal)ov_nousyukuki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ov_nousyukuki_input_all") {
                //投入汚泥濃縮設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value != null) {
                            decimal ov_nousyukuki = 0;

                            if(decimal.TryParse(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value.ToString(), out ov_nousyukuki)) {
                                sum += ov_nousyukuki;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? ov_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyukuki_" + i.ToString())].tempValueDec;

                        if(ov_nousyukuki2 != null) {
                            sum += (decimal)ov_nousyukuki2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "sg_nousyukuki_input_juuryoku") {
                //投汚固形物量重力濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    decimal oci_nousyukuki = 0;
                    decimal ov_nousyukuki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() == "重力式") {
                            if(dgv1Row.Cells["oci_nousyukuki_" + i.ToString()].Value != null) {
                                if(!decimal.TryParse(dgv1Row.Cells["oci_nousyukuki_" + i.ToString()].Value.ToString(), out oci_nousyukuki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value != null) {
                                if(decimal.TryParse(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value.ToString(), out ov_nousyukuki)) {
                                    sum += ov_nousyukuki * oci_nousyukuki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr == "重力式") {
                            decimal? oci_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_nousyukuki_" + i.ToString())].tempValueDec;

                            if(oci_nousyukuki2 == null) {
                                continue;
                            }

                            decimal? ov_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyukuki_" + i.ToString())].tempValueDec;

                            if(ov_nousyukuki2 != null) {
                                oci_nousyukuki = (decimal)oci_nousyukuki2;
                                ov_nousyukuki = (decimal)ov_nousyukuki2;

                                sum += ov_nousyukuki * oci_nousyukuki;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_nousyukuki_input_ex") {
                //投汚固形物量他濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    decimal oci_nousyukuki = 0;
                    decimal ov_nousyukuki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() != "重力式") {
                            if(dgv1Row.Cells["oci_nousyukuki_" + i.ToString()].Value != null) {
                                if(!decimal.TryParse(dgv1Row.Cells["oci_nousyukuki_" + i.ToString()].Value.ToString(), out oci_nousyukuki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value != null) {
                                if(decimal.TryParse(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value.ToString(), out ov_nousyukuki)) {
                                    sum += ov_nousyukuki * oci_nousyukuki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr != "重力式") {
                            decimal? oci_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_nousyukuki_" + i.ToString())].tempValueDec;

                            if(oci_nousyukuki2 == null) {
                                continue;
                            }

                            decimal? ov_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyukuki_" + i.ToString())].tempValueDec;

                            if(ov_nousyukuki2 != null) {
                                oci_nousyukuki = (decimal)oci_nousyukuki2;
                                ov_nousyukuki = (decimal)ov_nousyukuki2;

                                sum += ov_nousyukuki * oci_nousyukuki;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_nousyukuki_input_all") {
                //投汚固形物濃縮設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    decimal oci_nousyukuki = 0;
                    decimal ov_nousyukuki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["oci_nousyukuki_" + i.ToString()].Value != null) {
                            if(!decimal.TryParse(dgv1Row.Cells["oci_nousyukuki_" + i.ToString()].Value.ToString(), out oci_nousyukuki)) {
                                continue;
                            }
                        }

                        if(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value != null) {
                            if(decimal.TryParse(dgv1Row.Cells["ov_nousyukuki_" + i.ToString()].Value.ToString(), out ov_nousyukuki)) {
                                sum += ov_nousyukuki * oci_nousyukuki;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? oci_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_nousyukuki_" + i.ToString())].tempValueDec;

                        if(oci_nousyukuki2 == null) {
                            continue;
                        }

                        decimal? ov_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyukuki_" + i.ToString())].tempValueDec;

                        if(ov_nousyukuki2 != null) {
                            oci_nousyukuki = (decimal)oci_nousyukuki2;
                            ov_nousyukuki = (decimal)ov_nousyukuki2;

                            sum += ov_nousyukuki * oci_nousyukuki;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "ug_gyousyuuzai_nousyukuki_juuryoku") {
                //高分子使用量重力濃計
                decimal v_gyousyuuzai_nousyukuki = 0;
                decimal oc_gyousyuuzai_nousyukuki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() == "重力式") {
                            if(dgv1Row.Cells["v_gyousyuuzai_nousyukuki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_nousyukuki_" + i.ToString()].Value.ToString(), out v_gyousyuuzai_nousyukuki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["oc_gyousyuuzai_nousyukuki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_nousyukuki_" + i.ToString()].Value.ToString(), out oc_gyousyuuzai_nousyukuki)) {
                                    continue;
                                }
                            }

                            sum += v_gyousyuuzai_nousyukuki * oc_gyousyuuzai_nousyukuki / 100;
                            count++;
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr == "重力式") {
                            decimal? v_gyousyuuzai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_nousyukuki_" + i.ToString())].tempValueDec;

                            if(v_gyousyuuzai_nousyukuki2 == null) {
                                continue;
                            }

                            decimal? oc_gyousyuuzai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_nousyukuki_" + i.ToString())].tempValueDec;

                            if(oc_gyousyuuzai_nousyukuki2 == null) {
                                continue;
                            }

                            v_gyousyuuzai_nousyukuki = (decimal)v_gyousyuuzai_nousyukuki2;
                            oc_gyousyuuzai_nousyukuki = (decimal)oc_gyousyuuzai_nousyukuki2;

                            sum += v_gyousyuuzai_nousyukuki * oc_gyousyuuzai_nousyukuki / 100;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ug_gyousyuuzai_nousyukuki_ex") {
                //高分子使用量他濃計
                decimal v_gyousyuuzai_nousyukuki = 0;
                decimal oc_gyousyuuzai_nousyukuki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() != "重力式") {
                            if(dgv1Row.Cells["v_gyousyuuzai_nousyukuki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_nousyukuki_" + i.ToString()].Value.ToString(), out v_gyousyuuzai_nousyukuki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["oc_gyousyuuzai_nousyukuki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_nousyukuki_" + i.ToString()].Value.ToString(), out oc_gyousyuuzai_nousyukuki)) {
                                    continue;
                                }
                            }

                            sum += v_gyousyuuzai_nousyukuki * oc_gyousyuuzai_nousyukuki / 100;
                            count++;
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr != "重力式") {
                            decimal? v_gyousyuuzai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_nousyukuki_" + i.ToString())].tempValueDec;

                            if(v_gyousyuuzai_nousyukuki2 == null) {
                                continue;
                            }

                            decimal? oc_gyousyuuzai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_nousyukuki_" + i.ToString())].tempValueDec;

                            if(oc_gyousyuuzai_nousyukuki2 == null) {
                                continue;
                            }

                            v_gyousyuuzai_nousyukuki = (decimal)v_gyousyuuzai_nousyukuki2;
                            oc_gyousyuuzai_nousyukuki = (decimal)oc_gyousyuuzai_nousyukuki2;

                            sum += v_gyousyuuzai_nousyukuki * oc_gyousyuuzai_nousyukuki / 100;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ug_gyousyuuzai_nousyukuki_all") {
                //高分子使用濃縮設備計
                decimal v_gyousyuuzai_nousyukuki = 0;
                decimal oc_gyousyuuzai_nousyukuki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["v_gyousyuuzai_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else {
                            if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_nousyukuki_" + i.ToString()].Value.ToString(), out v_gyousyuuzai_nousyukuki)) {
                                continue;
                            }
                        }

                        if(dgv1Row.Cells["oc_gyousyuuzai_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else {
                            if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_nousyukuki_" + i.ToString()].Value.ToString(), out oc_gyousyuuzai_nousyukuki)) {
                                continue;
                            }
                        }

                        sum += v_gyousyuuzai_nousyukuki * oc_gyousyuuzai_nousyukuki / 100;
                        count++;
                    }
                    else if(ccSet != null) {
                        decimal? v_gyousyuuzai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_nousyukuki_" + i.ToString())].tempValueDec;

                        if(v_gyousyuuzai_nousyukuki2 == null) {
                            continue;
                        }

                        decimal? oc_gyousyuuzai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_nousyukuki_" + i.ToString())].tempValueDec;

                        if(oc_gyousyuuzai_nousyukuki2 == null) {
                            continue;
                        }

                        v_gyousyuuzai_nousyukuki = (decimal)v_gyousyuuzai_nousyukuki2;
                        oc_gyousyuuzai_nousyukuki = (decimal)oc_gyousyuuzai_nousyukuki2;

                        sum += v_gyousyuuzai_nousyukuki * oc_gyousyuuzai_nousyukuki / 100;
                        count++;
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ug_kihoujozai_nousyukuki_juuryoku") {
                //起泡剤使用量重力濃計
                decimal v_kihoujozai_nousyukuki = 0;
                decimal oc_kihoujozai_nousyukuki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() == "重力式") {
                            if(dgv1Row.Cells["v_kihoujozai_nousyukuki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["v_kihoujozai_nousyukuki_" + i.ToString()].Value.ToString(), out v_kihoujozai_nousyukuki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["oc_kihoujozai_nousyukuki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["oc_kihoujozai_nousyukuki_" + i.ToString()].Value.ToString(), out oc_kihoujozai_nousyukuki)) {
                                    continue;
                                }
                            }

                            sum += v_kihoujozai_nousyukuki * oc_kihoujozai_nousyukuki / 100;
                            count++;
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr == "重力式") {
                            decimal? v_kihoujozai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("v_kihoujozai_nousyukuki_" + i.ToString())].tempValueDec;

                            if(v_kihoujozai_nousyukuki2 == null) {
                                continue;
                            }

                            decimal? oc_kihoujozai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_kihoujozai_nousyukuki_" + i.ToString())].tempValueDec;

                            if(oc_kihoujozai_nousyukuki2 == null) {
                                continue;
                            }

                            v_kihoujozai_nousyukuki = (decimal)v_kihoujozai_nousyukuki2;
                            oc_kihoujozai_nousyukuki = (decimal)oc_kihoujozai_nousyukuki2;

                            sum += v_kihoujozai_nousyukuki * oc_kihoujozai_nousyukuki / 100;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ug_kihoujozai_nousyukuki_ex") {
                //起泡剤使用量他濃計
                decimal v_kihoujozai_nousyukuki = 0;
                decimal oc_kihoujozai_nousyukuki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() != "重力式") {
                            if(dgv1Row.Cells["v_kihoujozai_nousyukuki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["v_kihoujozai_nousyukuki_" + i.ToString()].Value.ToString(), out v_kihoujozai_nousyukuki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["oc_kihoujozai_nousyukuki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["oc_kihoujozai_nousyukuki_" + i.ToString()].Value.ToString(), out oc_kihoujozai_nousyukuki)) {
                                    continue;
                                }
                            }

                            sum += v_kihoujozai_nousyukuki * oc_kihoujozai_nousyukuki / 100;
                            count++;
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr != "重力式") {
                            decimal? v_kihoujozai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("v_kihoujozai_nousyukuki_" + i.ToString())].tempValueDec;

                            if(v_kihoujozai_nousyukuki2 == null) {
                                continue;
                            }

                            decimal? oc_kihoujozai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_kihoujozai_nousyukuki_" + i.ToString())].tempValueDec;

                            if(oc_kihoujozai_nousyukuki2 == null) {
                                continue;
                            }

                            v_kihoujozai_nousyukuki = (decimal)v_kihoujozai_nousyukuki2;
                            oc_kihoujozai_nousyukuki = (decimal)oc_kihoujozai_nousyukuki2;

                            sum += v_kihoujozai_nousyukuki * oc_kihoujozai_nousyukuki / 100;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ug_kihoujozai_nousyukuki_all") {
                //起泡剤使用濃縮設備計
                decimal v_kihoujozai_nousyukuki = 0;
                decimal oc_kihoujozai_nousyukuki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["v_kihoujozai_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else {
                            if(!decimal.TryParse(dgv1Row.Cells["v_kihoujozai_nousyukuki_" + i.ToString()].Value.ToString(), out v_kihoujozai_nousyukuki)) {
                                continue;
                            }
                        }

                        if(dgv1Row.Cells["oc_kihoujozai_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else {
                            if(!decimal.TryParse(dgv1Row.Cells["oc_kihoujozai_nousyukuki_" + i.ToString()].Value.ToString(), out oc_kihoujozai_nousyukuki)) {
                                continue;
                            }
                        }

                        sum += v_kihoujozai_nousyukuki * oc_kihoujozai_nousyukuki / 100;
                        count++;
                    }
                    else if(ccSet != null) {
                        decimal? v_kihoujozai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("v_kihoujozai_nousyukuki_" + i.ToString())].tempValueDec;

                        if(v_kihoujozai_nousyukuki2 == null) {
                            continue;
                        }

                        decimal? oc_kihoujozai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_kihoujozai_nousyukuki_" + i.ToString())].tempValueDec;

                        if(oc_kihoujozai_nousyukuki2 == null) {
                            continue;
                        }

                        v_kihoujozai_nousyukuki = (decimal)v_kihoujozai_nousyukuki2;
                        oc_kihoujozai_nousyukuki = (decimal)oc_kihoujozai_nousyukuki2;

                        sum += v_kihoujozai_nousyukuki * oc_kihoujozai_nousyukuki / 100;
                        count++;
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ov_nousyuku_output_all") {
                //濃縮汚泥発生量計
                decimal ov_nousyuku_output_juuryoku = 0;
                decimal ov_nousyuku_output_ex = 0;
                bool nullFlug = true;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ov_nousyuku_output_juuryoku"].Value == null) {
                        ov_nousyuku_output_juuryoku = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nousyuku_output_juuryoku"].Value.ToString(), out ov_nousyuku_output_juuryoku)) {
                            ov_nousyuku_output_juuryoku = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_nousyuku_output_ex"].Value == null) {
                        ov_nousyuku_output_ex = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nousyuku_output_ex"].Value.ToString(), out ov_nousyuku_output_ex)) {
                            ov_nousyuku_output_ex = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ov_nousyuku_output_juuryoku2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyuku_output_juuryoku")].tempValueDec;

                    if(ov_nousyuku_output_juuryoku2 == null) {
                        ov_nousyuku_output_juuryoku2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_nousyuku_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyuku_output_ex")].tempValueDec;

                    if(ov_nousyuku_output_ex2 == null) {
                        ov_nousyuku_output_ex2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    ov_nousyuku_output_juuryoku = (decimal)ov_nousyuku_output_juuryoku2;
                    ov_nousyuku_output_ex = (decimal)ov_nousyuku_output_ex2;
                }

                if(nullFlug) {
                    return null;
                }

                decimal retValue2 = ov_nousyuku_output_juuryoku + ov_nousyuku_output_ex;
                retValue = retValue2;
            }
            else if(ccName == "sg_nousyuku_output_juuryoku") {
                //濃縮汚泥固形物量重力
                decimal oc_nousyuku_output_juuryoku = 0;
                decimal ov_nousyuku_output_juuryoku = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oc_nousyuku_output_juuryoku"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_nousyuku_output_juuryoku"].Value.ToString(), out oc_nousyuku_output_juuryoku)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_nousyuku_output_juuryoku"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nousyuku_output_juuryoku"].Value.ToString(), out ov_nousyuku_output_juuryoku)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oc_nousyuku_output_juuryoku2 = ccSet[ColumnsConfigSet.nameToIndex("oc_nousyuku_output_juuryoku")].tempValueDec;

                    if(oc_nousyuku_output_juuryoku2 == null) {
                        return null;
                    }

                    decimal? ov_nousyuku_output_juuryoku2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyuku_output_juuryoku")].tempValueDec;

                    if(ov_nousyuku_output_juuryoku2 == null) {
                        return null;
                    }

                    oc_nousyuku_output_juuryoku = (decimal)oc_nousyuku_output_juuryoku2;
                    ov_nousyuku_output_juuryoku = (decimal)ov_nousyuku_output_juuryoku2;
                }

                decimal retValue2 = ov_nousyuku_output_juuryoku * oc_nousyuku_output_juuryoku / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_nousyuku_output_ex") {
                //濃縮汚泥固形物量他
                decimal oc_nousyuku_output_ex = 0;
                decimal ov_nousyuku_output_ex = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oc_nousyuku_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_nousyuku_output_ex"].Value.ToString(), out oc_nousyuku_output_ex)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_nousyuku_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nousyuku_output_ex"].Value.ToString(), out ov_nousyuku_output_ex)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oc_nousyuku_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("oc_nousyuku_output_ex")].tempValueDec;

                    if(oc_nousyuku_output_ex2 == null) {
                        return null;
                    }

                    decimal? ov_nousyuku_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyuku_output_ex")].tempValueDec;

                    if(ov_nousyuku_output_ex2 == null) {
                        return null;
                    }

                    oc_nousyuku_output_ex = (decimal)oc_nousyuku_output_ex2;
                    ov_nousyuku_output_ex = (decimal)ov_nousyuku_output_ex2;
                }

                decimal retValue2 = ov_nousyuku_output_ex * oc_nousyuku_output_ex / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_nousyuku_output_all") {
                //濃縮汚泥固形物量計
                decimal oc_nousyuku_output_juuryoku = 0;
                decimal ov_nousyuku_output_juuryoku = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oc_nousyuku_output_juuryoku"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_nousyuku_output_juuryoku"].Value.ToString(), out oc_nousyuku_output_juuryoku)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_nousyuku_output_juuryoku"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nousyuku_output_juuryoku"].Value.ToString(), out ov_nousyuku_output_juuryoku)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oc_nousyuku_output_juuryoku2 = ccSet[ColumnsConfigSet.nameToIndex("oc_nousyuku_output_juuryoku")].tempValueDec;

                    if(oc_nousyuku_output_juuryoku2 == null) {
                        return null;
                    }

                    decimal? ov_nousyuku_output_juuryoku2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyuku_output_juuryoku")].tempValueDec;

                    if(ov_nousyuku_output_juuryoku2 == null) {
                        return null;
                    }

                    oc_nousyuku_output_juuryoku = (decimal)oc_nousyuku_output_juuryoku2;
                    ov_nousyuku_output_juuryoku = (decimal)ov_nousyuku_output_juuryoku2;
                }

                decimal retValue2 = ov_nousyuku_output_juuryoku * oc_nousyuku_output_juuryoku / 100 * 1000;

                decimal oc_nousyuku_output_ex = 0;
                decimal ov_nousyuku_output_ex = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oc_nousyuku_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_nousyuku_output_ex"].Value.ToString(), out oc_nousyuku_output_ex)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_nousyuku_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nousyuku_output_ex"].Value.ToString(), out ov_nousyuku_output_ex)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oc_nousyuku_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("oc_nousyuku_output_ex")].tempValueDec;

                    if(oc_nousyuku_output_ex2 == null) {
                        return null;
                    }

                    decimal? ov_nousyuku_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyuku_output_ex")].tempValueDec;

                    if(ov_nousyuku_output_ex2 == null) {
                        return null;
                    }

                    oc_nousyuku_output_ex = (decimal)oc_nousyuku_output_ex2;
                    ov_nousyuku_output_ex = (decimal)ov_nousyuku_output_ex2;
                }

                decimal retValue3 = ov_nousyuku_output_ex * oc_nousyuku_output_ex / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2 + retValue3, ccName);
            }
            else if(ccName == "dt_nousyukuki_juuryoku") {
                //運転時間重力濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() == "重力式") {
                            if(dgv1Row.Cells["dt_nousyukuki_" + i.ToString()].Value != null) {
                                decimal dt_nousyukuki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["dt_nousyukuki_" + i.ToString()].Value.ToString(), out dt_nousyukuki)) {
                                    sum += dt_nousyukuki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr == "重力式") {
                            decimal? dt_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("dt_nousyukuki_" + i.ToString())].tempValueDec;

                            if(dt_nousyukuki2 != null) {
                                sum += (decimal)dt_nousyukuki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "dt_nousyukuki_ex") {
                //運転時間他濃計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_nousyukuki_" + i.ToString()].Value.ToString() != "重力式") {
                            if(dgv1Row.Cells["dt_nousyukuki_" + i.ToString()].Value != null) {
                                decimal dt_nousyukuki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["dt_nousyukuki_" + i.ToString()].Value.ToString(), out dt_nousyukuki)) {
                                    sum += dt_nousyukuki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_nousyukuki_" + i.ToString())].tempValueStr != "重力式") {
                            decimal? dt_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("dt_nousyukuki_" + i.ToString())].tempValueDec;

                            if(dt_nousyukuki2 != null) {
                                sum += (decimal)dt_nousyukuki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "dt_nousyukuki_all") {
                //運転時間濃縮設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxNousyukuki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["dt_nousyukuki_" + i.ToString()].Value != null) {
                            decimal dt_nousyukuki = 0;

                            if(decimal.TryParse(dgv1Row.Cells["dt_nousyukuki_" + i.ToString()].Value.ToString(), out dt_nousyukuki)) {
                                sum += dt_nousyukuki;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? dt_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("dt_nousyukuki_" + i.ToString())].tempValueDec;

                        if(dt_nousyukuki2 != null) {
                            sum += (decimal)dt_nousyukuki2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }

            #endregion


            #region 消化計算定義

            else if(ccName == "sg_syouka_input") {
                //消化槽投汚固形物量
                decimal oc_syouka_input = 0;
                decimal ov_syouka_input = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oc_syouka_input"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_syouka_input"].Value.ToString(), out oc_syouka_input)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_syouka_input"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_syouka_input"].Value.ToString(), out ov_syouka_input)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oc_syouka_input2 = ccSet[ColumnsConfigSet.nameToIndex("oc_syouka_input")].tempValueDec;

                    if(oc_syouka_input2 == null) {
                        return null;
                    }

                    decimal? ov_syouka_input2 = ccSet[ColumnsConfigSet.nameToIndex("ov_syouka_input")].tempValueDec;

                    if(ov_syouka_input2 == null) {
                        return null;
                    }

                    oc_syouka_input = (decimal)oc_syouka_input2;
                    ov_syouka_input = (decimal)ov_syouka_input2;
                }

                decimal retValue2 = ov_syouka_input * oc_syouka_input / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_syouka_output") {
                //消化汚泥固形物量
                decimal oc_syouka_output = 0;
                decimal ov_syouka_output = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oc_syouka_output"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_syouka_output"].Value.ToString(), out oc_syouka_output)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_syouka_output"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_syouka_output"].Value.ToString(), out ov_syouka_output)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oc_syouka_output2 = ccSet[ColumnsConfigSet.nameToIndex("oc_syouka_output")].tempValueDec;

                    if(oc_syouka_output2 == null) {
                        return null;
                    }

                    decimal? ov_syouka_output2 = ccSet[ColumnsConfigSet.nameToIndex("ov_syouka_output")].tempValueDec;

                    if(ov_syouka_output2 == null) {
                        return null;
                    }

                    oc_syouka_output = (decimal)oc_syouka_output2;
                    ov_syouka_output = (decimal)ov_syouka_output2;
                }

                decimal retValue2 = ov_syouka_output * oc_syouka_output / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "r_syouka") {
                //消化率
                decimal oy_syouka_input = 0;
                decimal oy_syouka_output = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oy_syouka_input"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oy_syouka_input"].Value.ToString(), out oy_syouka_input)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["oy_syouka_output"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oy_syouka_output"].Value.ToString(), out oy_syouka_output)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oy_syouka_input2 = ccSet[ColumnsConfigSet.nameToIndex("oy_syouka_input")].tempValueDec;

                    if(oy_syouka_input2 == null) {
                        return null;
                    }

                    decimal? oy_syouka_output2 = ccSet[ColumnsConfigSet.nameToIndex("oy_syouka_output")].tempValueDec;

                    if(oy_syouka_output2 == null) {
                        return null;
                    }

                    oy_syouka_input = (decimal)oy_syouka_input2;
                    oy_syouka_output = (decimal)oy_syouka_output2;
                }

                if((oy_syouka_input * (100 - oy_syouka_output)) <= 0) {
                    return null;
                }

                decimal retValue2 = (1 - (100 - oy_syouka_input) * oy_syouka_output / (oy_syouka_input * (100 - oy_syouka_output))) * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "d_syouka") {
                //消化日数
                decimal pv_syouka = 0;
                decimal ov_syouka_input = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["pv_syouka"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["pv_syouka"].Value.ToString(), out pv_syouka)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_syouka_input"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_syouka_input"].Value.ToString(), out ov_syouka_input)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? pv_syouka2 = ccSet[ColumnsConfigSet.nameToIndex("pv_syouka")].tempValueDec;

                    if(pv_syouka2 == null) {
                        return null;
                    }

                    decimal? ov_syouka_input2 = ccSet[ColumnsConfigSet.nameToIndex("ov_syouka_input")].tempValueDec;

                    if(ov_syouka_input2 == null) {
                        return null;
                    }

                    pv_syouka = (decimal)pv_syouka2;
                    ov_syouka_input = (decimal)ov_syouka_input2;
                }

                if(ov_syouka_input <= 0) {
                    return null;
                }

                decimal retValue2 = pv_syouka / ov_syouka_input;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "gr_syoukagas") {
                //ガス発生倍率
                decimal v_syoukag_hassei = 0;
                decimal ov_syouka_input = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["v_syoukag_hassei"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["v_syoukag_hassei"].Value.ToString(), out v_syoukag_hassei)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_syouka_input"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_syouka_input"].Value.ToString(), out ov_syouka_input)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? v_syoukag_hassei2 = ccSet[ColumnsConfigSet.nameToIndex("v_syoukag_hassei")].tempValueDec;

                    if(v_syoukag_hassei2 == null) {
                        return null;
                    }

                    decimal? ov_syouka_input2 = ccSet[ColumnsConfigSet.nameToIndex("ov_syouka_input")].tempValueDec;

                    if(ov_syouka_input2 == null) {
                        return null;
                    }

                    v_syoukag_hassei = (decimal)v_syoukag_hassei2;
                    ov_syouka_input = (decimal)ov_syouka_input2;
                }

                if(ov_syouka_input <= 0) {
                    return null;
                }

                decimal retValue2 = v_syoukag_hassei / ov_syouka_input;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "dt_syoukatank_all") {
                //運転時間消化槽計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxSyoukatank; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["dt_syoukatank_" + i.ToString()].Value != null) {
                            decimal dt_syoukatank = 0;

                            if(decimal.TryParse(dgv1Row.Cells["dt_syoukatank_" + i.ToString()].Value.ToString(), out dt_syoukatank)) {
                                sum += dt_syoukatank;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? dt_syoukatank2 = ccSet[ColumnsConfigSet.nameToIndex("dt_syoukatank_" + i.ToString())].tempValueDec;

                        if(dt_syoukatank2 != null) {
                            sum += (decimal)dt_syoukatank2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }

            #endregion


            #region 汚泥脱水計算定義

            else if(ccName == "ov_dassuiki_input_enshin") {
                //供給汚泥量遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "遠心分離") {
                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                decimal ov_dassuiki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "遠心分離") {
                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                sum += (decimal)ov_dassuiki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ov_dassuiki_input_bp") {
                //供給汚泥量BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "ベルトプレス") {
                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                decimal ov_dassuiki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "ベルトプレス") {
                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                sum += (decimal)ov_dassuiki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ov_dassuiki_input_ex") {
                //供給汚泥量他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "遠心分離" && dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "ベルトプレス") {
                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                decimal ov_dassuiki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "遠心分離" && ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "ベルトプレス") {
                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                sum += (decimal)ov_dassuiki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ov_dassuiki_input_all") {
                //供給汚泥量脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                            decimal ov_dassuiki = 0;

                            if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                sum += ov_dassuiki;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                        if(ov_dassuiki2 != null) {
                            sum += (decimal)ov_dassuiki2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "sg_dassuiki_input_enshin") {
                //固形物量遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal oci_dassuiki = 0;
                    decimal ov_dassuiki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "遠心分離") {
                            if(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value != null) {
                                if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value.ToString(), out oci_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki * oci_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "遠心分離") {
                            decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + i.ToString())].tempValueDec;

                            if(oci_dassuiki2 == null) {
                                continue;
                            }

                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                oci_dassuiki = (decimal)oci_dassuiki2;
                                ov_dassuiki = (decimal)ov_dassuiki2;

                                sum += ov_dassuiki * oci_dassuiki;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_dassuiki_input_bp") {
                //固形物量BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal oci_dassuiki = 0;
                    decimal ov_dassuiki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "ベルトプレス") {
                            if(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value != null) {
                                if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value.ToString(), out oci_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki * oci_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "ベルトプレス") {
                            decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + i.ToString())].tempValueDec;

                            if(oci_dassuiki2 == null) {
                                continue;
                            }

                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                oci_dassuiki = (decimal)oci_dassuiki2;
                                ov_dassuiki = (decimal)ov_dassuiki2;

                                sum += ov_dassuiki * oci_dassuiki;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_dassuiki_input_ex") {
                //固形物量他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal oci_dassuiki = 0;
                    decimal ov_dassuiki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "遠心分離" && dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "ベルトプレス") {
                            if(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value != null) {
                                if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value.ToString(), out oci_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki * oci_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "遠心分離" && ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "ベルトプレス") {
                            decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + i.ToString())].tempValueDec;

                            if(oci_dassuiki2 == null) {
                                continue;
                            }

                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                oci_dassuiki = (decimal)oci_dassuiki2;
                                ov_dassuiki = (decimal)ov_dassuiki2;

                                sum += ov_dassuiki * oci_dassuiki;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_dassuiki_input_all") {
                //固形物量脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal oci_dassuiki = 0;
                    decimal ov_dassuiki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value != null) {
                            if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value.ToString(), out oci_dassuiki)) {
                                continue;
                            }
                        }

                        if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                            if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                sum += ov_dassuiki * oci_dassuiki;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + i.ToString())].tempValueDec;

                        if(oci_dassuiki2 == null) {
                            continue;
                        }

                        decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                        if(ov_dassuiki2 != null) {
                            oci_dassuiki = (decimal)oci_dassuiki2;
                            ov_dassuiki = (decimal)ov_dassuiki2;

                            sum += ov_dassuiki * oci_dassuiki;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal retValue2 = sum / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "ug_gyousyuuzai_dassuiki_enshin") {
                //高分子使用量遠心計
                decimal v_gyousyuuzai_dassuiki = 0;
                decimal oc_gyousyuuzai_dassuiki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "遠心分離") {
                            if(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + i.ToString()].Value.ToString(), out v_gyousyuuzai_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + i.ToString()].Value.ToString(), out oc_gyousyuuzai_dassuiki)) {
                                    continue;
                                }
                            }

                            sum += v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                            count++;
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "遠心分離") {
                            decimal? v_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_dassuiki_" + i.ToString())].tempValueDec;

                            if(v_gyousyuuzai_dassuiki2 == null) {
                                continue;
                            }

                            decimal? oc_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_dassuiki_" + i.ToString())].tempValueDec;

                            if(oc_gyousyuuzai_dassuiki2 == null) {
                                continue;
                            }

                            v_gyousyuuzai_dassuiki = (decimal)v_gyousyuuzai_dassuiki2;
                            oc_gyousyuuzai_dassuiki = (decimal)oc_gyousyuuzai_dassuiki2;

                            sum += v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ug_gyousyuuzai_dassuiki_bp") {
                //高分子使用量BP計
                decimal v_gyousyuuzai_dassuiki = 0;
                decimal oc_gyousyuuzai_dassuiki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "ベルトプレス") {
                            if(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + i.ToString()].Value.ToString(), out v_gyousyuuzai_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + i.ToString()].Value.ToString(), out oc_gyousyuuzai_dassuiki)) {
                                    continue;
                                }
                            }

                            sum += v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                            count++;
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "ベルトプレス") {
                            decimal? v_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_dassuiki_" + i.ToString())].tempValueDec;

                            if(v_gyousyuuzai_dassuiki2 == null) {
                                continue;
                            }

                            decimal? oc_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_dassuiki_" + i.ToString())].tempValueDec;

                            if(oc_gyousyuuzai_dassuiki2 == null) {
                                continue;
                            }

                            v_gyousyuuzai_dassuiki = (decimal)v_gyousyuuzai_dassuiki2;
                            oc_gyousyuuzai_dassuiki = (decimal)oc_gyousyuuzai_dassuiki2;

                            sum += v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ug_gyousyuuzai_dassuiki_ex") {
                //高分子使用量他計
                decimal v_gyousyuuzai_dassuiki = 0;
                decimal oc_gyousyuuzai_dassuiki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "遠心分離" && dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "ベルトプレス") {
                            if(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + i.ToString()].Value.ToString(), out v_gyousyuuzai_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + i.ToString()].Value == null) {
                                continue;
                            }
                            else {
                                if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + i.ToString()].Value.ToString(), out oc_gyousyuuzai_dassuiki)) {
                                    continue;
                                }
                            }

                            sum += v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                            count++;
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "遠心分離" && ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "ベルトプレス") {
                            decimal? v_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_dassuiki_" + i.ToString())].tempValueDec;

                            if(v_gyousyuuzai_dassuiki2 == null) {
                                continue;
                            }

                            decimal? oc_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_dassuiki_" + i.ToString())].tempValueDec;

                            if(oc_gyousyuuzai_dassuiki2 == null) {
                                continue;
                            }

                            v_gyousyuuzai_dassuiki = (decimal)v_gyousyuuzai_dassuiki2;
                            oc_gyousyuuzai_dassuiki = (decimal)oc_gyousyuuzai_dassuiki2;

                            sum += v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ug_gyousyuuzai_dassuiki_all") {
                //高分子使用量脱水設備計
                decimal v_gyousyuuzai_dassuiki = 0;
                decimal oc_gyousyuuzai_dassuiki = 0;
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else {
                            if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + i.ToString()].Value.ToString(), out v_gyousyuuzai_dassuiki)) {
                                continue;
                            }
                        }

                        if(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else {
                            if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + i.ToString()].Value.ToString(), out oc_gyousyuuzai_dassuiki)) {
                                continue;
                            }
                        }

                        sum += v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                        count++;
                    }
                    else if(ccSet != null) {
                        decimal? v_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_dassuiki_" + i.ToString())].tempValueDec;

                        if(v_gyousyuuzai_dassuiki2 == null) {
                            continue;
                        }

                        decimal? oc_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_dassuiki_" + i.ToString())].tempValueDec;

                        if(oc_gyousyuuzai_dassuiki2 == null) {
                            continue;
                        }

                        v_gyousyuuzai_dassuiki = (decimal)v_gyousyuuzai_dassuiki2;
                        oc_gyousyuuzai_dassuiki = (decimal)oc_gyousyuuzai_dassuiki2;

                        sum += v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                        count++;
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "dt_dassuiki_enshin") {
                //運転時間遠心計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "遠心分離") {
                            if(dgv1Row.Cells["dt_dassuiki_" + i.ToString()].Value != null) {
                                decimal dt_dassuiki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["dt_dassuiki_" + i.ToString()].Value.ToString(), out dt_dassuiki)) {
                                    sum += dt_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "遠心分離") {
                            decimal? dt_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("dt_dassuiki_" + i.ToString())].tempValueDec;

                            if(dt_dassuiki2 != null) {
                                sum += (decimal)dt_dassuiki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "dt_dassuiki_bp") {
                //運転時間BP計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "ベルトプレス") {
                            if(dgv1Row.Cells["dt_dassuiki_" + i.ToString()].Value != null) {
                                decimal dt_dassuiki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["dt_dassuiki_" + i.ToString()].Value.ToString(), out dt_dassuiki)) {
                                    sum += dt_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "ベルトプレス") {
                            decimal? dt_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("dt_dassuiki_" + i.ToString())].tempValueDec;

                            if(dt_dassuiki2 != null) {
                                sum += (decimal)dt_dassuiki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "dt_dassuiki_ex") {
                //運転時間他計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }
                        else if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "遠心分離" && dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "ベルトプレス") {
                            if(dgv1Row.Cells["dt_dassuiki_" + i.ToString()].Value != null) {
                                decimal dt_dassuiki = 0;

                                if(decimal.TryParse(dgv1Row.Cells["dt_dassuiki_" + i.ToString()].Value.ToString(), out dt_dassuiki)) {
                                    sum += dt_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "遠心分離" && ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "ベルトプレス") {
                            decimal? dt_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("dt_dassuiki_" + i.ToString())].tempValueDec;

                            if(dt_dassuiki2 != null) {
                                sum += (decimal)dt_dassuiki2;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "dt_dassuiki_all") {
                //運転時間脱水設備計
                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["dt_dassuiki_" + i.ToString()].Value != null) {
                            decimal dt_dassuiki = 0;

                            if(decimal.TryParse(dgv1Row.Cells["dt_dassuiki_" + i.ToString()].Value.ToString(), out dt_dassuiki)) {
                                sum += dt_dassuiki;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? dt_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("dt_dassuiki_" + i.ToString())].tempValueDec;

                        if(dt_dassuiki2 != null) {
                            sum += (decimal)dt_dassuiki2;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                retValue = ColumnsConfigSet.roundingByName(sum, ccName);
            }
            else if(ccName == "ov_dassui_output_all") {
                //脱水汚泥発生量計
                decimal ov_dassui_output_enshin = 0;
                decimal ov_dassui_output_bp = 0;
                decimal ov_dassui_output_ex = 0;
                bool nullFlug = true;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ov_dassui_output_enshin"].Value == null) {
                        ov_dassui_output_enshin = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_enshin"].Value.ToString(), out ov_dassui_output_enshin)) {
                            ov_dassui_output_enshin = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_bp"].Value == null) {
                        ov_dassui_output_bp = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_bp"].Value.ToString(), out ov_dassui_output_bp)) {
                            ov_dassui_output_bp = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_ex"].Value == null) {
                        ov_dassui_output_ex = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_ex"].Value.ToString(), out ov_dassui_output_ex)) {
                            ov_dassui_output_ex = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ov_dassui_output_enshin2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_enshin")].tempValueDec;

                    if(ov_dassui_output_enshin2 == null) {
                        ov_dassui_output_enshin2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_dassui_output_bp2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_bp")].tempValueDec;

                    if(ov_dassui_output_bp2 == null) {
                        ov_dassui_output_bp2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_dassui_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_ex")].tempValueDec;

                    if(ov_dassui_output_ex2 == null) {
                        ov_dassui_output_ex2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    ov_dassui_output_enshin = (decimal)ov_dassui_output_enshin2;
                    ov_dassui_output_bp = (decimal)ov_dassui_output_bp2;
                    ov_dassui_output_ex = (decimal)ov_dassui_output_ex2;
                }

                if(nullFlug) {
                    return null;
                }

                decimal retValue2 = ov_dassui_output_enshin + ov_dassui_output_bp + ov_dassui_output_ex;
                retValue = retValue2;
            }
            else if(ccName == "sg_dassui_output_enshin") {
                //脱水汚泥固形物量遠心
                decimal owr_dassui_output_enshin = 0;
                decimal ov_dassui_output_enshin = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_enshin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_enshin"].Value.ToString(), out owr_dassui_output_enshin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_enshin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_enshin"].Value.ToString(), out ov_dassui_output_enshin)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_enshin2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_enshin")].tempValueDec;

                    if(owr_dassui_output_enshin2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_enshin2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_enshin")].tempValueDec;

                    if(ov_dassui_output_enshin2 == null) {
                        return null;
                    }

                    owr_dassui_output_enshin = (decimal)owr_dassui_output_enshin2;
                    ov_dassui_output_enshin = (decimal)ov_dassui_output_enshin2;
                }

                decimal retValue2 = (100 - owr_dassui_output_enshin) /100 * ov_dassui_output_enshin * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_dassui_output_bp") {
                //脱水汚泥固形物量BP
                decimal owr_dassui_output_bp = 0;
                decimal ov_dassui_output_bp = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_bp"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_bp"].Value.ToString(), out owr_dassui_output_bp)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_bp"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_bp"].Value.ToString(), out ov_dassui_output_bp)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_bp2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_bp")].tempValueDec;

                    if(owr_dassui_output_bp2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_bp2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_bp")].tempValueDec;

                    if(ov_dassui_output_bp2 == null) {
                        return null;
                    }

                    owr_dassui_output_bp = (decimal)owr_dassui_output_bp2;
                    ov_dassui_output_bp = (decimal)ov_dassui_output_bp2;
                }

                decimal retValue2 = (100 - owr_dassui_output_bp) / 100 * ov_dassui_output_bp * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_dassui_output_ex") {
                //脱水汚泥固形物量他
                decimal owr_dassui_output_ex = 0;
                decimal ov_dassui_output_ex = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_ex"].Value.ToString(), out owr_dassui_output_ex)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_ex"].Value.ToString(), out ov_dassui_output_ex)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_ex")].tempValueDec;

                    if(owr_dassui_output_ex2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_ex")].tempValueDec;

                    if(ov_dassui_output_ex2 == null) {
                        return null;
                    }

                    owr_dassui_output_ex = (decimal)owr_dassui_output_ex2;
                    ov_dassui_output_ex = (decimal)ov_dassui_output_ex2;
                }

                decimal retValue2 = (100 - owr_dassui_output_ex) / 100 * ov_dassui_output_ex * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sg_dassui_output_all") {
                //脱水汚泥固形物量計
                decimal owr_dassui_output_enshin = 0;
                decimal ov_dassui_output_enshin = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_enshin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_enshin"].Value.ToString(), out owr_dassui_output_enshin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_enshin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_enshin"].Value.ToString(), out ov_dassui_output_enshin)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_enshin2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_enshin")].tempValueDec;

                    if(owr_dassui_output_enshin2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_enshin2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_enshin")].tempValueDec;

                    if(ov_dassui_output_enshin2 == null) {
                        return null;
                    }

                    owr_dassui_output_enshin = (decimal)owr_dassui_output_enshin2;
                    ov_dassui_output_enshin = (decimal)ov_dassui_output_enshin2;
                }

                decimal owr_dassui_output_bp = 0;
                decimal ov_dassui_output_bp = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_bp"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_bp"].Value.ToString(), out owr_dassui_output_bp)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_bp"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_bp"].Value.ToString(), out ov_dassui_output_bp)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_bp2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_bp")].tempValueDec;

                    if(owr_dassui_output_bp2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_bp2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_bp")].tempValueDec;

                    if(ov_dassui_output_bp2 == null) {
                        return null;
                    }

                    owr_dassui_output_bp = (decimal)owr_dassui_output_bp2;
                    ov_dassui_output_bp = (decimal)ov_dassui_output_bp2;
                }

                decimal owr_dassui_output_ex = 0;
                decimal ov_dassui_output_ex = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_ex"].Value.ToString(), out owr_dassui_output_ex)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_ex"].Value.ToString(), out ov_dassui_output_ex)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_ex")].tempValueDec;

                    if(owr_dassui_output_ex2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_ex")].tempValueDec;

                    if(ov_dassui_output_ex2 == null) {
                        return null;
                    }

                    owr_dassui_output_ex = (decimal)owr_dassui_output_ex2;
                    ov_dassui_output_ex = (decimal)ov_dassui_output_ex2;
                }

                decimal retValue2 = (100 - owr_dassui_output_enshin) / 100 * ov_dassui_output_enshin * 1000 + (100 - owr_dassui_output_bp) / 100 * ov_dassui_output_bp * 1000 + (100 - owr_dassui_output_ex) / 100 * ov_dassui_output_ex * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sgcr_dassui_output_enshin") {
                //固形物回収率遠心
                decimal owr_dassui_output = 0;
                decimal ov_dassui_output = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_enshin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_enshin"].Value.ToString(), out owr_dassui_output)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_enshin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_enshin"].Value.ToString(), out ov_dassui_output)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_enshin")].tempValueDec;

                    if(owr_dassui_output2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_enshin")].tempValueDec;

                    if(ov_dassui_output2 == null) {
                        return null;
                    }

                    owr_dassui_output = (decimal)owr_dassui_output2;
                    ov_dassui_output = (decimal)ov_dassui_output2;
                }

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal oci_dassuiki = 0;
                    decimal ov_dassuiki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "遠心分離") {
                            if(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value != null) {
                                if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value.ToString(), out oci_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki * oci_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "遠心分離") {
                            decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + i.ToString())].tempValueDec;

                            if(oci_dassuiki2 == null) {
                                continue;
                            }

                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                oci_dassuiki = (decimal)oci_dassuiki2;
                                ov_dassuiki = (decimal)ov_dassuiki2;

                                sum += ov_dassuiki * oci_dassuiki;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal sg_dassuiki_input = sum / 100 * 1000;

                if(sg_dassuiki_input <= 0) {
                    return null;
                }

                decimal retValue2 = (100 - owr_dassui_output) / 100 * ov_dassui_output * 1000 /sg_dassuiki_input * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sgcr_dassui_output_bp") {
                //固形物回収率BP
                decimal owr_dassui_output = 0;
                decimal ov_dassui_output = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_bp"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_bp"].Value.ToString(), out owr_dassui_output)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_bp"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_bp"].Value.ToString(), out ov_dassui_output)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_bp")].tempValueDec;

                    if(owr_dassui_output2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_bp")].tempValueDec;

                    if(ov_dassui_output2 == null) {
                        return null;
                    }

                    owr_dassui_output = (decimal)owr_dassui_output2;
                    ov_dassui_output = (decimal)ov_dassui_output2;
                }

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal oci_dassuiki = 0;
                    decimal ov_dassuiki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() == "ベルトプレス") {
                            if(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value != null) {
                                if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value.ToString(), out oci_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki * oci_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr == "ベルトプレス") {
                            decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + i.ToString())].tempValueDec;

                            if(oci_dassuiki2 == null) {
                                continue;
                            }

                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                oci_dassuiki = (decimal)oci_dassuiki2;
                                ov_dassuiki = (decimal)ov_dassuiki2;

                                sum += ov_dassuiki * oci_dassuiki;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal sg_dassuiki_input = sum / 100 * 1000;

                if(sg_dassuiki_input <= 0) {
                    return null;
                }

                decimal retValue2 = (100 - owr_dassui_output) / 100 * ov_dassui_output * 1000 / sg_dassuiki_input * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sgcr_dassui_output_ex") {
                //固形物回収率他
                decimal owr_dassui_output = 0;
                decimal ov_dassui_output = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_ex"].Value.ToString(), out owr_dassui_output)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_ex"].Value.ToString(), out ov_dassui_output)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_ex")].tempValueDec;

                    if(owr_dassui_output2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_ex")].tempValueDec;

                    if(ov_dassui_output2 == null) {
                        return null;
                    }

                    owr_dassui_output = (decimal)owr_dassui_output2;
                    ov_dassui_output = (decimal)ov_dassui_output2;
                }

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal oci_dassuiki = 0;
                    decimal ov_dassuiki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "遠心分離" && dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value.ToString() != "ベルトプレス") {
                            if(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value != null) {
                                if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value.ToString(), out oci_dassuiki)) {
                                    continue;
                                }
                            }

                            if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                                if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                    sum += ov_dassuiki * oci_dassuiki;
                                    count++;
                                }
                            }
                        }
                    }
                    else if(ccSet != null) {
                        if(ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "遠心分離" && ccSet[ColumnsConfigSet.nameToIndex("type_dassuiki_" + i.ToString())].tempValueStr != "ベルトプレス") {
                            decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + i.ToString())].tempValueDec;

                            if(oci_dassuiki2 == null) {
                                continue;
                            }

                            decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                            if(ov_dassuiki2 != null) {
                                oci_dassuiki = (decimal)oci_dassuiki2;
                                ov_dassuiki = (decimal)ov_dassuiki2;

                                sum += ov_dassuiki * oci_dassuiki;
                                count++;
                            }
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal sg_dassuiki_input = sum / 100 * 1000;

                if(sg_dassuiki_input <= 0) {
                    return null;
                }

                decimal retValue2 = (100 - owr_dassui_output) / 100 * ov_dassui_output * 1000 / sg_dassuiki_input * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName == "sgcr_dassui_output_all") {
                //固形物回収率全体
                decimal owr_dassui_output_enshin = 0;
                decimal ov_dassui_output_enshin = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_enshin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_enshin"].Value.ToString(), out owr_dassui_output_enshin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_enshin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_enshin"].Value.ToString(), out ov_dassui_output_enshin)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_enshin2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_enshin")].tempValueDec;

                    if(owr_dassui_output_enshin2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_enshin2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_enshin")].tempValueDec;

                    if(ov_dassui_output_enshin2 == null) {
                        return null;
                    }

                    owr_dassui_output_enshin = (decimal)owr_dassui_output_enshin2;
                    ov_dassui_output_enshin = (decimal)ov_dassui_output_enshin2;
                }

                decimal owr_dassui_output_bp = 0;
                decimal ov_dassui_output_bp = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_bp"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_bp"].Value.ToString(), out owr_dassui_output_bp)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_bp"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_bp"].Value.ToString(), out ov_dassui_output_bp)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_bp2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_bp")].tempValueDec;

                    if(owr_dassui_output_bp2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_bp2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_bp")].tempValueDec;

                    if(ov_dassui_output_bp2 == null) {
                        return null;
                    }

                    owr_dassui_output_bp = (decimal)owr_dassui_output_bp2;
                    ov_dassui_output_bp = (decimal)ov_dassui_output_bp2;
                }

                decimal owr_dassui_output_ex = 0;
                decimal ov_dassui_output_ex = 0;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["owr_dassui_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["owr_dassui_output_ex"].Value.ToString(), out owr_dassui_output_ex)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassui_output_ex"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassui_output_ex"].Value.ToString(), out ov_dassui_output_ex)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? owr_dassui_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("owr_dassui_output_ex")].tempValueDec;

                    if(owr_dassui_output_ex2 == null) {
                        return null;
                    }

                    decimal? ov_dassui_output_ex2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassui_output_ex")].tempValueDec;

                    if(ov_dassui_output_ex2 == null) {
                        return null;
                    }

                    owr_dassui_output_ex = (decimal)owr_dassui_output_ex2;
                    ov_dassui_output_ex = (decimal)ov_dassui_output_ex2;
                }

                decimal sum = 0;
                int count = 0;

                for(int i = 1; i <= MaxSizeConfig.maxDassuiki; i++) {
                    decimal oci_dassuiki = 0;
                    decimal ov_dassuiki = 0;

                    if(dgv1Row != null) {
                        if(dgv1Row.Cells["type_dassuiki_" + i.ToString()].Value == null) {
                            continue;
                        }

                        if(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value != null) {
                            if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + i.ToString()].Value.ToString(), out oci_dassuiki)) {
                                continue;
                            }
                        }

                        if(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value != null) {
                            if(decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + i.ToString()].Value.ToString(), out ov_dassuiki)) {
                                sum += ov_dassuiki * oci_dassuiki;
                                count++;
                            }
                        }
                    }
                    else if(ccSet != null) {
                        decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + i.ToString())].tempValueDec;

                        if(oci_dassuiki2 == null) {
                            continue;
                        }

                        decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + i.ToString())].tempValueDec;

                        if(ov_dassuiki2 != null) {
                            oci_dassuiki = (decimal)oci_dassuiki2;
                            ov_dassuiki = (decimal)ov_dassuiki2;

                            sum += ov_dassuiki * oci_dassuiki;
                            count++;
                        }
                    }
                }

                if(count <= 0) {
                    return null;
                }

                decimal sg_dassuiki_input = sum / 100 * 1000;

                if(sg_dassuiki_input <= 0) {
                    return null;
                }

                decimal retValue2 = ((100 - owr_dassui_output_enshin) / 100 * ov_dassui_output_enshin * 1000 + (100 - owr_dassui_output_bp) / 100 * ov_dassui_output_bp * 1000 + (100 - owr_dassui_output_ex) / 100 * ov_dassui_output_ex * 1000) / sg_dassuiki_input * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }

            #endregion


            #region 汚泥等搬出

            else if(ccName == "ov_hansyutsu_all") {
                //汚泥等搬出量計
                decimal ov_hansyutsu_chinsa = 0;
                decimal ov_hansyutsu_shisa = 0;
                decimal ov_hansyutsu_namaodei = 0;
                decimal ov_hansyutsu_yojouodei = 0;
                decimal ov_hansyutsu_nousyukuodei = 0;
                decimal ov_hansyutsu_syoukaodei = 0;
                decimal ov_hansyutsu_dassuiodei = 0;
                bool nullFlug = true;

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["ov_hansyutsu_chinsa"].Value == null) {
                        ov_hansyutsu_chinsa = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_hansyutsu_chinsa"].Value.ToString(), out ov_hansyutsu_chinsa)) {
                            ov_hansyutsu_chinsa = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_hansyutsu_shisa"].Value == null) {
                        ov_hansyutsu_shisa = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_hansyutsu_shisa"].Value.ToString(), out ov_hansyutsu_shisa)) {
                            ov_hansyutsu_shisa = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_hansyutsu_namaodei"].Value == null) {
                        ov_hansyutsu_namaodei = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_hansyutsu_namaodei"].Value.ToString(), out ov_hansyutsu_namaodei)) {
                            ov_hansyutsu_namaodei = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_hansyutsu_yojouodei"].Value == null) {
                        ov_hansyutsu_yojouodei = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_hansyutsu_yojouodei"].Value.ToString(), out ov_hansyutsu_yojouodei)) {
                            ov_hansyutsu_yojouodei = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_hansyutsu_nousyukuodei"].Value == null) {
                        ov_hansyutsu_nousyukuodei = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_hansyutsu_nousyukuodei"].Value.ToString(), out ov_hansyutsu_nousyukuodei)) {
                            ov_hansyutsu_nousyukuodei = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_hansyutsu_syoukaodei"].Value == null) {
                        ov_hansyutsu_syoukaodei = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_hansyutsu_syoukaodei"].Value.ToString(), out ov_hansyutsu_syoukaodei)) {
                            ov_hansyutsu_syoukaodei = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }

                    if(dgv1Row.Cells["ov_hansyutsu_dassuiodei"].Value == null) {
                        ov_hansyutsu_dassuiodei = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_hansyutsu_dassuiodei"].Value.ToString(), out ov_hansyutsu_dassuiodei)) {
                            ov_hansyutsu_dassuiodei = 0;
                        }
                        else {
                            nullFlug = false;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? ov_hansyutsu_chinsa2 = ccSet[ColumnsConfigSet.nameToIndex("ov_hansyutsu_chinsa")].tempValueDec;

                    if(ov_hansyutsu_chinsa2 == null) {
                        ov_hansyutsu_chinsa2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_hansyutsu_shisa2 = ccSet[ColumnsConfigSet.nameToIndex("ov_hansyutsu_shisa")].tempValueDec;

                    if(ov_hansyutsu_shisa2 == null) {
                        ov_hansyutsu_shisa2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_hansyutsu_namaodei2 = ccSet[ColumnsConfigSet.nameToIndex("ov_hansyutsu_namaodei")].tempValueDec;

                    if(ov_hansyutsu_namaodei2 == null) {
                        ov_hansyutsu_namaodei2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_hansyutsu_yojouodei2 = ccSet[ColumnsConfigSet.nameToIndex("ov_hansyutsu_yojouodei")].tempValueDec;

                    if(ov_hansyutsu_yojouodei2 == null) {
                        ov_hansyutsu_yojouodei2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_hansyutsu_nousyukuodei2 = ccSet[ColumnsConfigSet.nameToIndex("ov_hansyutsu_nousyukuodei")].tempValueDec;

                    if(ov_hansyutsu_nousyukuodei2 == null) {
                        ov_hansyutsu_nousyukuodei2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_hansyutsu_syoukaodei2 = ccSet[ColumnsConfigSet.nameToIndex("ov_hansyutsu_syoukaodei")].tempValueDec;

                    if(ov_hansyutsu_syoukaodei2 == null) {
                        ov_hansyutsu_syoukaodei2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    decimal? ov_hansyutsu_dassuiodei2 = ccSet[ColumnsConfigSet.nameToIndex("ov_hansyutsu_dassuiodei")].tempValueDec;

                    if(ov_hansyutsu_dassuiodei2 == null) {
                        ov_hansyutsu_dassuiodei2 = 0;
                    }
                    else {
                        nullFlug = false;
                    }

                    ov_hansyutsu_chinsa = (decimal)ov_hansyutsu_chinsa2;
                    ov_hansyutsu_shisa = (decimal)ov_hansyutsu_shisa2;
                    ov_hansyutsu_namaodei = (decimal)ov_hansyutsu_namaodei2;
                    ov_hansyutsu_yojouodei = (decimal)ov_hansyutsu_yojouodei2;
                    ov_hansyutsu_nousyukuodei = (decimal)ov_hansyutsu_nousyukuodei2;
                    ov_hansyutsu_syoukaodei = (decimal)ov_hansyutsu_syoukaodei2;
                    ov_hansyutsu_dassuiodei = (decimal)ov_hansyutsu_dassuiodei2;
                }

                if(nullFlug) {
                    return null;
                }

                decimal retValue2 = ov_hansyutsu_chinsa + ov_hansyutsu_shisa + ov_hansyutsu_namaodei + ov_hansyutsu_yojouodei + ov_hansyutsu_nousyukuodei + ov_hansyutsu_syoukaodei + ov_hansyutsu_dassuiodei;
                retValue = retValue2;
            }

            #endregion


            #region 反応T個別計算定義

            else if(ccName.Length > 18 && ccName.Substring(0, 18) == "vsspss_hannoutank_") {
                //反応TnVSS/SS
                decimal mlvss_hannoutunk = 0;
                decimal mlss_hannoutunk = 0;

                string tankNo = ccName.Replace("vsspss_hannoutank_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["mlvss_hannoutank_" + tankNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["mlvss_hannoutank_" + tankNo].Value.ToString(), out mlvss_hannoutunk)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["mlss_hannoutank_" + tankNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["mlss_hannoutank_" + tankNo].Value.ToString(), out mlss_hannoutunk)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? mlvss_hannoutunk2 = ccSet[ColumnsConfigSet.nameToIndex("mlvss_hannoutank_" + tankNo)].tempValueDec;

                    if(mlvss_hannoutunk2 == null) {
                        return null;
                    }

                    decimal? mlss_hannoutunk2 = ccSet[ColumnsConfigSet.nameToIndex("mlss_hannoutank_" + tankNo)].tempValueDec;

                    if(mlss_hannoutunk2 == null) {
                        return null;
                    }

                    mlvss_hannoutunk = (decimal)mlvss_hannoutunk2;
                    mlss_hannoutunk = (decimal)mlss_hannoutunk2;
                }

                if(mlss_hannoutunk <= 0) {
                    return null;
                }

                decimal retValue2 = mlvss_hannoutunk / mlss_hannoutunk * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName.Length > 15 && ccName.Substring(0, 15) == "svi_hannoutank_") {
                //反応TnSVI
                decimal sv_hannoutank = 0;
                decimal mlss_hannoutunk = 0;

                string tankNo = ccName.Replace("svi_hannoutank_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["sv_hannoutank_" + tankNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["sv_hannoutank_" + tankNo].Value.ToString(), out sv_hannoutank)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["mlss_hannoutank_" + tankNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["mlss_hannoutank_" + tankNo].Value.ToString(), out mlss_hannoutunk)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? sv_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("sv_hannoutank_" + tankNo)].tempValueDec;

                    if(sv_hannoutank2 == null) {
                        return null;
                    }

                    decimal? mlss_hannoutunk2 = ccSet[ColumnsConfigSet.nameToIndex("mlss_hannoutank_" + tankNo)].tempValueDec;

                    if(mlss_hannoutunk2 == null) {
                        return null;
                    }

                    mlss_hannoutunk = (decimal)mlss_hannoutunk2;
                    sv_hannoutank = (decimal)sv_hannoutank2;
                }

                if(mlss_hannoutunk <= 0) {
                    return null;
                }

                decimal retValue2 = sv_hannoutank * 10000 / mlss_hannoutunk;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName.Length > 20 && ccName.Substring(0, 20) == "bodpmlss_hannoutank_") {
                //反応TnBOD/MLSS負荷
                decimal wv_syochin = 0;
                decimal wv_kani = 0;
                decimal bod_syo_etsuryuu = 0;
                decimal pv_hannoutank = 0;
                decimal mlss_hannoutank = 0;

                string tankNo = ccName.Replace("bodpmlss_hannoutank_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["wv_syochin"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_syochin"].Value.ToString(), out wv_syochin)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["wv_kani"].Value == null) {
                        wv_kani = 0;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["wv_kani"].Value.ToString(), out wv_kani)) {
                            wv_kani = 0;
                        }
                    }

                    if(dgv1Row.Cells["bod_syo_etsuryuu"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["bod_syo_etsuryuu"].Value.ToString(), out bod_syo_etsuryuu)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["pv_hannoutank"].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["pv_hannoutank"].Value.ToString(), out pv_hannoutank)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["mlss_hannoutank_" + tankNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["mlss_hannoutank_" + tankNo].Value.ToString(), out mlss_hannoutank)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? wv_syochin2 = ccSet[ColumnsConfigSet.nameToIndex("wv_syochin")].tempValueDec;

                    if(wv_syochin2 == null) {
                        return null;
                    }

                    decimal? wv_kani2 = ccSet[ColumnsConfigSet.nameToIndex("wv_kani")].tempValueDec;

                    if(wv_kani2 == null) {
                        wv_kani2 = 0;
                    }

                    decimal? bod_syo_etsuryuu2 = ccSet[ColumnsConfigSet.nameToIndex("bod_syo_etsuryuu")].tempValueDec;

                    if(bod_syo_etsuryuu2 == null) {
                        return null;
                    }

                    decimal? pv_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("pv_hannoutank")].tempValueDec;

                    if(pv_hannoutank2 == null) {
                        return null;
                    }

                    decimal? mlss_hannoutank2 = ccSet[ColumnsConfigSet.nameToIndex("mlss_hannoutank_" + tankNo)].tempValueDec;

                    if(mlss_hannoutank2 == null) {
                        return null;
                    }

                    wv_syochin = (decimal)wv_syochin2;
                    wv_kani = (decimal)wv_kani2;
                    bod_syo_etsuryuu = (decimal)bod_syo_etsuryuu2;
                    pv_hannoutank = (decimal)pv_hannoutank2;
                    mlss_hannoutank = (decimal)mlss_hannoutank2;
                }

                if(pv_hannoutank <= 0 || mlss_hannoutank <= 0) {
                    return null;
                }

                decimal retValue2 = (wv_syochin - wv_kani) * bod_syo_etsuryuu / (pv_hannoutank * mlss_hannoutank);
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }

            #endregion


            #region 濃縮設備個別計算定義

            else if(ccName.Length > 14 && ccName.Substring(0, 14) == "sg_nousyukuki_") {
                //濃縮設備n投入汚泥固形物量
                decimal oci_nousyukuki = 0;
                decimal ov_nousyukuki = 0;

                string nousyukukiNo = ccName.Replace("sg_nousyukuki_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oci_nousyukuki_" + nousyukukiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oci_nousyukuki_" + nousyukukiNo].Value.ToString(), out oci_nousyukuki)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_nousyukuki_" + nousyukukiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_nousyukuki_" + nousyukukiNo].Value.ToString(), out ov_nousyukuki)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oci_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_nousyukuki_" + nousyukukiNo)].tempValueDec;

                    if(oci_nousyukuki2 == null) {
                        return null;
                    }

                    decimal? ov_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_nousyukuki_" + nousyukukiNo)].tempValueDec;

                    if(ov_nousyukuki2 == null) {
                        return null;
                    }

                    oci_nousyukuki = (decimal)oci_nousyukuki2;
                    ov_nousyukuki = (decimal)ov_nousyukuki2;
                }

                decimal retValue2 = oci_nousyukuki * ov_nousyukuki / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName.Length > 26 && ccName.Substring(0, 26) == "ug_gyousyuuzai_nousyukuki_") {
                //濃縮設備n高分子凝集剤使用量
                decimal v_gyousyuuzai_nousyukuki = 0;
                decimal oc_gyousyuuzai_nousyukuki = 0;

                string nousyukukiNo = ccName.Replace("ug_gyousyuuzai_nousyukuki_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["v_gyousyuuzai_nousyukuki_" + nousyukukiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_nousyukuki_" + nousyukukiNo].Value.ToString(), out v_gyousyuuzai_nousyukuki)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["oc_gyousyuuzai_nousyukuki_" + nousyukukiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_nousyukuki_" + nousyukukiNo].Value.ToString(), out oc_gyousyuuzai_nousyukuki)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? v_gyousyuuzai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_nousyukuki_" + nousyukukiNo)].tempValueDec;

                    if(v_gyousyuuzai_nousyukuki2 == null) {
                        return null;
                    }

                    decimal? oc_gyousyuuzai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_nousyukuki_" + nousyukukiNo)].tempValueDec;

                    if(oc_gyousyuuzai_nousyukuki2 == null) {
                        return null;
                    }

                    v_gyousyuuzai_nousyukuki = (decimal)v_gyousyuuzai_nousyukuki2;
                    oc_gyousyuuzai_nousyukuki = (decimal)oc_gyousyuuzai_nousyukuki2;
                }

                decimal retValue2 = v_gyousyuuzai_nousyukuki * oc_gyousyuuzai_nousyukuki / 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName.Length > 25 && ccName.Substring(0, 25) == "ug_kihoujozai_nousyukuki_") {
                //濃縮設備n起泡助剤使用量
                decimal v_kihoujozai_nousyukuki = 0;
                decimal oc_kihoujozai_nousyukuki = 0;

                string nousyukukiNo = ccName.Replace("ug_kihoujozai_nousyukuki_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["v_kihoujozai_nousyukuki_" + nousyukukiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["v_kihoujozai_nousyukuki_" + nousyukukiNo].Value.ToString(), out v_kihoujozai_nousyukuki)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["oc_kihoujozai_nousyukuki_" + nousyukukiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_kihoujozai_nousyukuki_" + nousyukukiNo].Value.ToString(), out oc_kihoujozai_nousyukuki)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? v_kihoujozai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("v_kihoujozai_nousyukuki_" + nousyukukiNo)].tempValueDec;

                    if(v_kihoujozai_nousyukuki2 == null) {
                        return null;
                    }

                    decimal? oc_kihoujozai_nousyukuki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_kihoujozai_nousyukuki_" + nousyukukiNo)].tempValueDec;

                    if(oc_kihoujozai_nousyukuki2 == null) {
                        return null;
                    }

                    v_kihoujozai_nousyukuki = (decimal)v_kihoujozai_nousyukuki2;
                    oc_kihoujozai_nousyukuki = (decimal)oc_kihoujozai_nousyukuki2;
                }

                decimal retValue2 = v_kihoujozai_nousyukuki * oc_kihoujozai_nousyukuki / 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }

            #endregion


            #region 脱水設備個別計算定義

            else if(ccName.Length > 12 && ccName.Substring(0, 12) == "sg_dassuiki_") {
                //脱水設備n投入汚泥固形物量
                decimal oci_dassuiki = 0;
                decimal ov_dassuiki = 0;

                string dassuikiNo = ccName.Replace("sg_dassuiki_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["oci_dassuiki_" + dassuikiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + dassuikiNo].Value.ToString(), out oci_dassuiki)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassuiki_" + dassuikiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + dassuikiNo].Value.ToString(), out ov_dassuiki)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + dassuikiNo)].tempValueDec;

                    if(oci_dassuiki2 == null) {
                        return null;
                    }

                    decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + dassuikiNo)].tempValueDec;

                    if(ov_dassuiki2 == null) {
                        return null;
                    }

                    oci_dassuiki = (decimal)oci_dassuiki2;
                    ov_dassuiki = (decimal)ov_dassuiki2;
                }

                decimal retValue2 = oci_dassuiki * ov_dassuiki / 100 * 1000;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName.Length > 24 && ccName.Substring(0, 24) == "ug_gyousyuuzai_dassuiki_") {
                //脱水設備n高分子凝集剤使用量
                decimal v_gyousyuuzai_dassuiki = 0;
                decimal oc_gyousyuuzai_dassuiki = 0;

                string dassuikiNo = ccName.Replace("ug_gyousyuuzai_dassuiki_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + dassuikiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + dassuikiNo].Value.ToString(), out v_gyousyuuzai_dassuiki)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + dassuikiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + dassuikiNo].Value.ToString(), out oc_gyousyuuzai_dassuiki)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? v_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_dassuiki_" + dassuikiNo)].tempValueDec;

                    if(v_gyousyuuzai_dassuiki2 == null) {
                        return null;
                    }

                    decimal? oc_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_dassuiki_" + dassuikiNo)].tempValueDec;

                    if(oc_gyousyuuzai_dassuiki2 == null) {
                        return null;
                    }

                    v_gyousyuuzai_dassuiki = (decimal)v_gyousyuuzai_dassuiki2;
                    oc_gyousyuuzai_dassuiki = (decimal)oc_gyousyuuzai_dassuiki2;
                }

                decimal retValue2 = v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }
            else if(ccName.Length > 23 && ccName.Substring(0, 23) == "r_gyousyuuzai_dassuiki_") {
                //脱水設備n高分子凝集剤注入率
                decimal v_gyousyuuzai_dassuiki = 0;
                decimal oc_gyousyuuzai_dassuiki = 0;
                decimal oci_dassuiki = 0;
                decimal ov_dassuiki = 0;

                string dassuikiNo = ccName.Replace("r_gyousyuuzai_dassuiki_", "");

                if(dgv1Row != null) {
                    if(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + dassuikiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["v_gyousyuuzai_dassuiki_" + dassuikiNo].Value.ToString(), out v_gyousyuuzai_dassuiki)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + dassuikiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oc_gyousyuuzai_dassuiki_" + dassuikiNo].Value.ToString(), out oc_gyousyuuzai_dassuiki)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["oci_dassuiki_" + dassuikiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["oci_dassuiki_" + dassuikiNo].Value.ToString(), out oci_dassuiki)) {
                            return null;
                        }
                    }

                    if(dgv1Row.Cells["ov_dassuiki_" + dassuikiNo].Value == null) {
                        return null;
                    }
                    else {
                        if(!decimal.TryParse(dgv1Row.Cells["ov_dassuiki_" + dassuikiNo].Value.ToString(), out ov_dassuiki)) {
                            return null;
                        }
                    }
                }
                else if(ccSet != null) {
                    decimal? v_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("v_gyousyuuzai_dassuiki_" + dassuikiNo)].tempValueDec;

                    if(v_gyousyuuzai_dassuiki2 == null) {
                        return null;
                    }

                    decimal? oc_gyousyuuzai_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oc_gyousyuuzai_dassuiki_" + dassuikiNo)].tempValueDec;

                    if(oc_gyousyuuzai_dassuiki2 == null) {
                        return null;
                    }

                    decimal? oci_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("oci_dassuiki_" + dassuikiNo)].tempValueDec;

                    if(oci_dassuiki2 == null) {
                        return null;
                    }

                    decimal? ov_dassuiki2 = ccSet[ColumnsConfigSet.nameToIndex("ov_dassuiki_" + dassuikiNo)].tempValueDec;

                    if(ov_dassuiki2 == null) {
                        return null;
                    }

                    v_gyousyuuzai_dassuiki = (decimal)v_gyousyuuzai_dassuiki2;
                    oc_gyousyuuzai_dassuiki = (decimal)oc_gyousyuuzai_dassuiki2;
                    oci_dassuiki = (decimal)oci_dassuiki2;
                    ov_dassuiki = (decimal)ov_dassuiki2;
                }

                if((oci_dassuiki / 100 * ov_dassuiki * 1000) <= 0) {
                    return null;
                }

                decimal retValue2 = (v_gyousyuuzai_dassuiki * oc_gyousyuuzai_dassuiki / 100) / (oci_dassuiki / 100 * ov_dassuiki * 1000) * 100;
                retValue = ColumnsConfigSet.roundingByName(retValue2, ccName);
            }

            #endregion


            return retValue;
        }




        //二次データを求める計算（文字列戻り）
        public static string calc2ndString(string ccName, DataGridViewRow dgv1Row = null, List<ColumnConfig> ccSet = null) {
            string retValue = null;

            switch(ccName) {
                case "rei":

                    break;
            }

            return retValue;
        }

    }
}
